package de.ntdote.medicalcalendarlog.service

import android.Manifest
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.pm.PackageManager
import android.util.Log
import androidx.core.content.ContextCompat
import androidx.work.ExistingPeriodicWorkPolicy
import androidx.work.PeriodicWorkRequestBuilder
import androidx.work.WorkManager
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import java.util.concurrent.TimeUnit

/**
 * BroadcastReceiver that listens for device boot completion events
 * and reschedules the hourly reminder worker.
 */
class BootReceiver : BroadcastReceiver() {
    
    companion object {
        private const val TAG = "MCL"
    }
    
    /**
     * Check if calendar permissions are granted
     */
    private fun hasCalendarPermissions(context: Context): Boolean {
        val readCalendarPermission = ContextCompat.checkSelfPermission(
            context, 
            Manifest.permission.READ_CALENDAR
        )
        
        val writeCalendarPermission = ContextCompat.checkSelfPermission(
            context, 
            Manifest.permission.WRITE_CALENDAR
        )
        
        return readCalendarPermission == PackageManager.PERMISSION_GRANTED &&
               writeCalendarPermission == PackageManager.PERMISSION_GRANTED
    }
    
    override fun onReceive(context: Context, intent: Intent) {
        if (intent.action == Intent.ACTION_BOOT_COMPLETED) {
            Log.d(TAG, "Device boot completed, checking permissions before rescheduling services")
            
            // Check if calendar permissions are granted
            if (!hasCalendarPermissions(context)) {
                Log.w(TAG, "Cannot reschedule services after boot - calendar permissions not granted")
                return
            }
            
            // Schedule the hourly worker
            scheduleHourlyWorker(context)
            
            // Trigger reminder recalculation (ContentObserver is already registered in Application class)
            CoroutineScope(Dispatchers.IO).launch {
                ReminderService.recalculateAllReminders(context, "boot_completed")
            }
        }
    }
    
    /**
     * Schedule the hourly worker to run periodically
     */
    private fun scheduleHourlyWorker(context: Context) {
        // Double-check permissions before scheduling
        if (!hasCalendarPermissions(context)) {
            Log.w(TAG, "Cannot schedule hourly worker - calendar permissions not granted")
            return
        }
        val hourlyWorkRequest = PeriodicWorkRequestBuilder<HourlyReminderWorker>(
            70, TimeUnit.MINUTES, // Run every 70 minutes (55-70 min window)
            15, TimeUnit.MINUTES // Flex period of 15 minutes
        )
            .addTag("hourly_reminder_worker")
            .build()
        
        // Enqueue the work request, replacing any existing one
        WorkManager.getInstance(context).enqueueUniquePeriodicWork(
            "hourly_reminder_worker",
            ExistingPeriodicWorkPolicy.REPLACE, // Replace existing work if any
            hourlyWorkRequest
        )
        
        Log.d(TAG, "Hourly reminder worker scheduled after device boot")
    }
}
