package de.ntdote.medicalcalendarlog.repository

import android.content.Context
import androidx.datastore.core.DataStore
import androidx.datastore.preferences.core.Preferences
import androidx.datastore.preferences.core.booleanPreferencesKey
import androidx.datastore.preferences.core.edit
import androidx.datastore.preferences.core.intPreferencesKey
import androidx.datastore.preferences.core.longPreferencesKey
import androidx.datastore.preferences.core.stringPreferencesKey
import androidx.datastore.preferences.preferencesDataStore
import de.ntdote.medicalcalendarlog.data.Template
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.flow.map
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json

val Context.dataStore: DataStore<Preferences> by preferencesDataStore(name = "settings")

class PreferencesRepository(private val context: Context) {
    
    companion object {
        private val SELECTED_CALENDAR_ID = longPreferencesKey("selected_calendar_id")
        private val DAYS_BACK = intPreferencesKey("days_back")
        private val TEMPLATES = stringPreferencesKey("templates")
        // Keeping the same key name for backward compatibility but renaming the variable
        private val DEBUG_MODE_ENABLED = booleanPreferencesKey("reminder_toasts_enabled")
        private val PRIVACY_POPUP_DISMISS_TIMESTAMP = longPreferencesKey("privacy_popup_dismiss_timestamp")
        private val LAST_MIGRATED_VERSION = intPreferencesKey("last_migrated_version")
        private val HOURLY_WORKER_RUNS = stringPreferencesKey("hourly_worker_runs")
        private val NEXT_REMINDER_TIMESTAMP = longPreferencesKey("next_reminder_timestamp")
        private val NEXT_REMINDER_TEMPLATE_NAME = stringPreferencesKey("next_reminder_template_name")
        private val WINDOW_START_TIMESTAMP = longPreferencesKey("window_start_timestamp")
        private val WINDOW_LAST_UPDATED = longPreferencesKey("window_last_updated")
        
        // 24 hours in milliseconds
        private const val DISMISS_DURATION_MS = 24 * 60 * 60 * 1000L
        // 10 days in milliseconds - window advance interval
        private const val WINDOW_ADVANCE_INTERVAL_MS = 10 * 24 * 60 * 60 * 1000L
    }

    val selectedCalendarId: Flow<Long?> = context.dataStore.data.map { preferences ->
        preferences[SELECTED_CALENDAR_ID]
    }

    val daysBack: Flow<Int> = context.dataStore.data.map { preferences ->
        preferences[DAYS_BACK] ?: 90
    }

    val templates: Flow<List<Template>> = context.dataStore.data.map { preferences ->
        val templatesJson = preferences[TEMPLATES] ?: "[]"
        try {
            Json.decodeFromString<List<Template>>(templatesJson)
        } catch (e: Exception) {
            emptyList()
        }
    }

    val debugModeEnabled: Flow<Boolean> = context.dataStore.data.map { preferences ->
        preferences[DEBUG_MODE_ENABLED] ?: false // Default: disabled
    }

    val shouldShowPrivacyPopup: Flow<Boolean> = context.dataStore.data.map { preferences ->
        val dismissTimestamp = preferences[PRIVACY_POPUP_DISMISS_TIMESTAMP] ?: 0L
        
        // If timestamp is 0, popup has never been dismissed - show it
        if (dismissTimestamp == 0L) {
            return@map true
        }
        
        // If timestamp is Long.MAX_VALUE, user chose "don't show again" - don't show
        if (dismissTimestamp == Long.MAX_VALUE) {
            return@map false
        }
        
        // Check if 24 hours have passed since dismissal
        val currentTime = System.currentTimeMillis()
        val timeSinceDismiss = currentTime - dismissTimestamp
        
        // Show popup if more than 24 hours have passed
        timeSinceDismiss >= DISMISS_DURATION_MS
    }
    
    val nextReminderTimestamp: Flow<Long> = context.dataStore.data.map { preferences ->
        preferences[NEXT_REMINDER_TIMESTAMP] ?: 0L
    }
    
    val nextReminderTemplateName: Flow<String> = context.dataStore.data.map { preferences ->
        preferences[NEXT_REMINDER_TEMPLATE_NAME] ?: ""
    }
    
    val windowStartTimestamp: Flow<Long> = context.dataStore.data.map { preferences ->
        preferences[WINDOW_START_TIMESTAMP] ?: 0L
    }
    
    val windowLastUpdated: Flow<Long> = context.dataStore.data.map { preferences ->
        preferences[WINDOW_LAST_UPDATED] ?: 0L
    }
    
    val hourlyWorkerRuns: Flow<List<Long>> = context.dataStore.data.map { preferences ->
        val runsJson = preferences[HOURLY_WORKER_RUNS] ?: "[]"
        try {
            Json.decodeFromString<List<Long>>(runsJson)
        } catch (e: Exception) {
            emptyList()
        }
    }

    suspend fun setSelectedCalendarId(calendarId: Long) {
        context.dataStore.edit { preferences ->
            preferences[SELECTED_CALENDAR_ID] = calendarId
        }
    }

    suspend fun setDaysBack(days: Int) {
        context.dataStore.edit { preferences ->
            preferences[DAYS_BACK] = days
            // Reset window when daysBack changes to force recalculation
            preferences[WINDOW_START_TIMESTAMP] = 0L
            preferences[WINDOW_LAST_UPDATED] = 0L
            android.util.Log.d("MCL", "PreferencesRepository: daysBack changed to $days, reset window timestamps")
        }
    }

    suspend fun setDebugModeEnabled(enabled: Boolean) {
        context.dataStore.edit { preferences ->
            preferences[DEBUG_MODE_ENABLED] = enabled
        }
    }

    suspend fun dismissPrivacyPopup(permanently: Boolean = false) {
        context.dataStore.edit { preferences ->
            if (permanently) {
                // Store Long.MAX_VALUE to indicate permanent dismissal
                preferences[PRIVACY_POPUP_DISMISS_TIMESTAMP] = Long.MAX_VALUE
            } else {
                // Store current timestamp for 24-hour dismissal
                preferences[PRIVACY_POPUP_DISMISS_TIMESTAMP] = System.currentTimeMillis()
            }
        }
    }

    /**
     * Reset privacy popup timestamp to force users to see it again.
     * Used during migrations when important disclaimer content is updated.
     */
    suspend fun resetPrivacyPopupForMigration() {
        context.dataStore.edit { preferences ->
            preferences[PRIVACY_POPUP_DISMISS_TIMESTAMP] = 0L
            android.util.Log.d("MCL", "PreferencesRepository: Privacy popup reset for migration - users will see updated disclaimer")
        }
    }

    suspend fun saveTemplates(templates: List<Template>) {
        context.dataStore.edit { preferences ->
            preferences[TEMPLATES] = Json.encodeToString(templates)
        }
    }

    suspend fun addTemplate(template: Template) {
        context.dataStore.edit { preferences ->
            val currentTemplatesJson = preferences[TEMPLATES] ?: "[]"
            val currentTemplates = try {
                Json.decodeFromString<List<Template>>(currentTemplatesJson).toMutableList()
            } catch (e: Exception) {
                mutableListOf()
            }
            
            currentTemplates.add(template)
            preferences[TEMPLATES] = Json.encodeToString(currentTemplates)
        }
    }

    suspend fun removeTemplate(templateId: String) {
        context.dataStore.edit { preferences ->
            val currentTemplatesJson = preferences[TEMPLATES] ?: "[]"
            val currentTemplates = try {
                Json.decodeFromString<List<Template>>(currentTemplatesJson)
            } catch (e: Exception) {
                emptyList()
            }
            
            val updatedTemplates = currentTemplates.filter { it.id != templateId }
            preferences[TEMPLATES] = Json.encodeToString(updatedTemplates)
        }
    }

    suspend fun updateTemplate(template: Template) {
        context.dataStore.edit { preferences ->
            val currentTemplatesJson = preferences[TEMPLATES] ?: "[]"
            val currentTemplates = try {
                Json.decodeFromString<List<Template>>(currentTemplatesJson).toMutableList()
            } catch (e: Exception) {
                mutableListOf()
            }
            
            val index = currentTemplates.indexOfFirst { it.id == template.id }
            if (index != -1) {
                currentTemplates[index] = template
            } else {
                currentTemplates.add(template)
            }
            
            preferences[TEMPLATES] = Json.encodeToString(currentTemplates)
        }
    }

    suspend fun replaceAllTemplates(templates: List<Template>) {
        context.dataStore.edit { preferences ->
            preferences[TEMPLATES] = Json.encodeToString(templates)
        }
    }
    
    /**
     * Record a worker run with its type
     */
    suspend fun recordWorkerRun(timestamp: Long, type: String) {
        context.dataStore.edit { preferences ->
            // Only record hourly worker runs for delta calculations (exclude manual runs)
            if (type == "hourly") {
                recordHourlyWorkerRun(preferences, timestamp)
            }
            // Manual runs are simply ignored and won't appear anywhere in the debug card
        }
    }
    
    /**
     * Record an hourly worker run in the list of last 12 runs
     */
    private fun recordHourlyWorkerRun(preferences: androidx.datastore.preferences.core.MutablePreferences, timestamp: Long) {
        val currentRunsJson = preferences[HOURLY_WORKER_RUNS] ?: "[]"
        val currentRuns = try {
            Json.decodeFromString<List<Long>>(currentRunsJson).toMutableList()
        } catch (e: Exception) {
            mutableListOf()
        }
        
        // Add new run to the beginning of the list
        currentRuns.add(0, timestamp)
        
        // Keep only the last 12 runs
        val trimmedRuns = currentRuns.take(12)
        
        preferences[HOURLY_WORKER_RUNS] = Json.encodeToString(trimmedRuns)
    }
    
    /**
     * Get time deltas between consecutive hourly worker runs in minutes
     */
    suspend fun getHourlyWorkerDeltas(): List<Int> {
        val runs = hourlyWorkerRuns.first()
        if (runs.size < 2) return emptyList()
        
        val deltas = mutableListOf<Int>()
        for (i in 0 until runs.size - 1) {
            val deltaMs = runs[i] - runs[i + 1]  // More recent - older
            val deltaMinutes = (deltaMs / (1000 * 60)).toInt()
            deltas.add(deltaMinutes)
        }
        
        return deltas
    }
    
    /**
     * Initialize the migration framework at app startup.
     * This method must be called by Application.onCreate()
     */
    suspend fun initializeMigrations(currentVersionCode: Int) {
        var shouldResetPopup = false
        
        context.dataStore.edit { preferences ->
            val lastMigratedVersion = preferences[LAST_MIGRATED_VERSION] ?: 0
            
            if (lastMigratedVersion >= currentVersionCode) {
                android.util.Log.d("MCL", "PreferencesRepository: No migration needed - current: $lastMigratedVersion >= target: $currentVersionCode")
                return@edit
            }
            
            android.util.Log.d("MCL", "PreferencesRepository: Starting migration from version $lastMigratedVersion to $currentVersionCode")
            
            // Get current templates for migration
            val currentTemplatesJson = preferences[TEMPLATES] ?: "[]"
            val currentTemplates = try {
                Json.decodeFromString<List<Template>>(currentTemplatesJson)
            } catch (e: Exception) {
                android.util.Log.w("MCL", "PreferencesRepository: Failed to parse templates for migration", e)
                emptyList()
            }
            
            // Apply migrations using centralized framework
            val migratedTemplates = de.ntdote.medicalcalendarlog.utils.DataMigrations.migrateTemplates(
                templates = currentTemplates,
                sourceVersion = lastMigratedVersion,
                targetVersion = currentVersionCode
            )
            
            // Save migrated templates if changes were made
            if (migratedTemplates != currentTemplates) {
                android.util.Log.d("MCL", "PreferencesRepository: Templates changed after migration, saving ${migratedTemplates.size} templates")
                preferences[TEMPLATES] = Json.encodeToString(migratedTemplates)
            } else {
                android.util.Log.d("MCL", "PreferencesRepository: No template changes after migration")
            }
            
            // Check if we need to reset privacy popup (migrating to version 38+)
            if (lastMigratedVersion < 38 && currentVersionCode >= 38) {
                shouldResetPopup = true
            }
            
            // Update last migrated version
            preferences[LAST_MIGRATED_VERSION] = currentVersionCode
            android.util.Log.d("MCL", "PreferencesRepository: Updated last migrated version to $currentVersionCode")
        }
        
        // Reset privacy popup if needed (must be done outside the edit block)
        if (shouldResetPopup) {
            android.util.Log.d("MCL", "PreferencesRepository: Migration to version 38+ detected, resetting privacy popup")
            resetPrivacyPopupForMigration()
        }
    }
    
    /**
     * Initialize or update the window start timestamp.
     * Returns the current window start timestamp (newly set or existing).
     */
    suspend fun updateWindowIfNeeded(currentTime: Long, daysBack: Int): Long {
        var resultTimestamp = 0L
        context.dataStore.edit { preferences ->
            val windowStart = preferences[WINDOW_START_TIMESTAMP] ?: 0L
            val windowLastUpdated = preferences[WINDOW_LAST_UPDATED] ?: 0L
            
            // Initialize window if not set (first run or after reset)
            if (windowStart == 0L) {
                val newWindowStart = currentTime - (daysBack * 24 * 60 * 60 * 1000L)
                preferences[WINDOW_START_TIMESTAMP] = newWindowStart
                preferences[WINDOW_LAST_UPDATED] = currentTime
                resultTimestamp = newWindowStart
                android.util.Log.d("MCL", "PreferencesRepository: Initialized window - start=${java.util.Date(newWindowStart)}, daysBack=$daysBack")
                return@edit
            }
            
            // Check if 10 days have passed since last update
            val timeSinceUpdate = currentTime - windowLastUpdated
            if (timeSinceUpdate >= WINDOW_ADVANCE_INTERVAL_MS) {
                // Advance window by 10 days
                val newWindowStart = windowStart + WINDOW_ADVANCE_INTERVAL_MS
                preferences[WINDOW_START_TIMESTAMP] = newWindowStart
                preferences[WINDOW_LAST_UPDATED] = currentTime
                resultTimestamp = newWindowStart
                android.util.Log.d("MCL", "PreferencesRepository: Advanced window - old=${java.util.Date(windowStart)}, new=${java.util.Date(newWindowStart)}, elapsed=${timeSinceUpdate}ms")
            } else {
                // Window is still valid
                resultTimestamp = windowStart
                android.util.Log.d("MCL", "PreferencesRepository: Window still valid - start=${java.util.Date(windowStart)}, lastUpdated=${java.util.Date(windowLastUpdated)}, elapsed=${timeSinceUpdate}ms")
            }
        }
        return resultTimestamp
    }
    
    /**
     * Reset the window (e.g., on reboot or config import).
     * This forces a full recalculation on the next run.
     */
    suspend fun resetWindow() {
        context.dataStore.edit { preferences ->
            preferences[WINDOW_START_TIMESTAMP] = 0L
            preferences[WINDOW_LAST_UPDATED] = 0L
            android.util.Log.d("MCL", "PreferencesRepository: Window reset - will reinitialize on next update")
        }
    }
    
    /**
     * Legacy method for compatibility with existing code
     */
    suspend fun migrateTemplatesOnVersionUpgrade(currentVersionCode: Int) {
        initializeMigrations(currentVersionCode)
    }
    
    /**
     * Set next reminder info for debugging
     */
    suspend fun setNextReminderInfo(timestamp: Long, templateName: String) {
        context.dataStore.edit { preferences ->
            preferences[NEXT_REMINDER_TIMESTAMP] = timestamp
            preferences[NEXT_REMINDER_TEMPLATE_NAME] = templateName
        }
    }
}
