package de.ntdote.medicalcalendarlog.repository

import android.Manifest
import android.content.ContentResolver
import android.content.ContentUris
import android.content.ContentValues
import android.content.Context
import android.content.pm.PackageManager
import android.database.Cursor
import android.provider.CalendarContract
import android.util.Log
import androidx.core.content.ContextCompat
import de.ntdote.medicalcalendarlog.data.BackupInfo
import de.ntdote.medicalcalendarlog.data.CalendarEvent
import de.ntdote.medicalcalendarlog.data.CalendarInfo
import de.ntdote.medicalcalendarlog.data.Template
import de.ntdote.medicalcalendarlog.utils.ConfigExportImport
import de.ntdote.medicalcalendarlog.service.HourlyReminderWorker
import java.util.Calendar
import java.util.Date
import java.util.TimeZone
import java.util.zip.CRC32

class CalendarRepository(private val contentResolver: ContentResolver, private val context: Context? = null) {
    
    companion object {
        private const val TAG = "MCL"
    }
    
    /**
     * Check if calendar permissions are granted
     */
    fun hasCalendarPermissions(): Boolean {
        if (context == null) return false
        
        val readCalendarPermission = ContextCompat.checkSelfPermission(
            context, 
            Manifest.permission.READ_CALENDAR
        )
        
        val writeCalendarPermission = ContextCompat.checkSelfPermission(
            context, 
            Manifest.permission.WRITE_CALENDAR
        )
        
        return readCalendarPermission == PackageManager.PERMISSION_GRANTED &&
               writeCalendarPermission == PackageManager.PERMISSION_GRANTED
    }

    /**
     * Get detailed information about calendar status
     */
    data class CalendarStatus(
        val hasPermissions: Boolean,
        val calendarsFound: Int,
        val errorMessage: String? = null,
        val exception: Exception? = null
    )
    
    fun getCalendarStatus(): CalendarStatus {
        if (!hasCalendarPermissions()) {
            return CalendarStatus(
                hasPermissions = false,
                calendarsFound = 0,
                errorMessage = "Calendar permissions not granted"
            )
        }
        
        try {
            val calendars = getAvailableCalendars()
            return CalendarStatus(
                hasPermissions = true,
                calendarsFound = calendars.size,
                errorMessage = if (calendars.isEmpty()) "No calendars found. Make sure calendar sync is enabled and you have at least one calendar." else null
            )
        } catch (e: SecurityException) {
            return CalendarStatus(
                hasPermissions = false,
                calendarsFound = 0,
                errorMessage = "Security exception when accessing calendars. Permissions may have been revoked.",
                exception = e
            )
        } catch (e: Exception) {
            return CalendarStatus(
                hasPermissions = true,
                calendarsFound = 0,
                errorMessage = "Error accessing calendars: ${e.message}",
                exception = e
            )
        }
    }

    fun getAvailableCalendars(): List<CalendarInfo> {
        if (!hasCalendarPermissions()) {
            Log.w(TAG, "CalendarRepository: Cannot get available calendars - permissions not granted")
            return emptyList()
        }
        
        val calendars = mutableListOf<CalendarInfo>()
        val projection = arrayOf(
            CalendarContract.Calendars._ID,
            CalendarContract.Calendars.CALENDAR_DISPLAY_NAME,
            CalendarContract.Calendars.ACCOUNT_NAME,
            CalendarContract.Calendars.OWNER_ACCOUNT
        )

        try {
            val cursor: Cursor? = contentResolver.query(
                CalendarContract.Calendars.CONTENT_URI,
                projection,
                null,
                null,
                null
            )

            cursor?.use {
                while (it.moveToNext()) {
                    val id = it.getLong(0)
                    val displayName = it.getString(1) ?: ""
                    val accountName = it.getString(2) ?: ""
                    val ownerName = it.getString(3) ?: ""

                    calendars.add(CalendarInfo(id, displayName, accountName, ownerName))
                }
                Log.d(TAG, "CalendarRepository: Found ${calendars.size} calendars")
            }
            
            if (cursor == null) {
                Log.w(TAG, "CalendarRepository: Cursor is null - calendar content provider may not be available")
            }
        } catch (e: SecurityException) {
            Log.e(TAG, "CalendarRepository: Security exception when accessing calendars", e)
            return emptyList()
        } catch (e: Exception) {
            Log.e(TAG, "CalendarRepository: Error accessing calendars", e)
            return emptyList()
        }

        return calendars
    }

    fun findPreferredCalendar(): CalendarInfo? {
        val calendars = getAvailableCalendars()
        
        // First, look for calendars containing "med" or "log"
        val preferredCalendar = calendars.find { calendar ->
            calendar.displayName.contains("med", ignoreCase = true) ||
            calendar.displayName.contains("log", ignoreCase = true)
        }
        
        return preferredCalendar ?: calendars.firstOrNull()
    }

    fun getEventsForPeriod(calendarId: Long, daysBack: Int): List<CalendarEvent> {
        val calendar = Calendar.getInstance()
        // Add 10% buffer for proper baseline calculations
        calendar.add(Calendar.DAY_OF_YEAR, -((daysBack + 45)))
        val startTime = calendar.timeInMillis
        
        // Delegate to the core function which handles permissions and backup filtering
        return getEventsFromTimestamp(calendarId, startTime)
    }

    /**
     * Wake up the calendar provider by performing a lightweight query
     * This prevents "process frozen" errors when inserting events
     */
    private fun wakeUpCalendarProvider(calendarId: Long): Boolean {
        try {
            val cursor = contentResolver.query(
                ContentUris.withAppendedId(CalendarContract.Calendars.CONTENT_URI, calendarId),
                arrayOf(CalendarContract.Calendars._ID),
                null,
                null,
                null
            )
            cursor?.close()
            return true
        } catch (e: Exception) {
            Log.w(TAG, "CalendarRepository: Failed to wake up calendar provider: ${e.message}")
            return false
        }
    }

    fun createEvent(calendarId: Long, title: String, templateId: String? = null, templateName: String? = null): Long? {
        if (!hasCalendarPermissions()) {
            Log.w(TAG, "CalendarRepository: Cannot create event - permissions not granted")
            return null
        }
        
        // Wake up calendar provider to prevent frozen process errors
        wakeUpCalendarProvider(calendarId)
        
        val now = System.currentTimeMillis()
        val oneMinuteAgo = now - (1 * 60 * 1000) // 1 minute ago

        val values = ContentValues().apply {
            put(CalendarContract.Events.DTSTART, oneMinuteAgo)
            put(CalendarContract.Events.DTEND, now)
            put(CalendarContract.Events.TITLE, title)
            put(CalendarContract.Events.CALENDAR_ID, calendarId)
            put(CalendarContract.Events.EVENT_TIMEZONE, TimeZone.getDefault().id)
        }

        var eventId: Long? = null
        var lastException: Exception? = null
        
        // Retry up to 3 times to handle frozen provider issues
        for (attempt in 1..3) {
            try {
                val uri = contentResolver.insert(CalendarContract.Events.CONTENT_URI, values)
                eventId = uri?.let { ContentUris.parseId(it) }
                
                if (eventId != null) {
                    if (attempt > 1) {
                        Log.d(TAG, "CalendarRepository: Event created successfully on attempt $attempt")
                    }
                    break
                }
            } catch (e: SecurityException) {
                Log.e(TAG, "CalendarRepository: Security exception when creating event", e)
                return null
            } catch (e: Exception) {
                lastException = e
                val errorMsg = e.message ?: e.toString()
                
                // Check if this is a frozen provider error (binder transaction failure)
                val isFrozenError = errorMsg.contains("FAILED BINDER TRANSACTION") ||
                                   errorMsg.contains("Transaction failed") ||
                                   errorMsg.contains("DeadObjectException")
                
                if (isFrozenError && attempt < 3) {
                    Log.w(TAG, "CalendarRepository: Provider frozen on attempt $attempt, retrying after delay...")
                    // Wait before retry (provider needs time to restart)
                    Thread.sleep(500)
                    // Wake up provider again
                    wakeUpCalendarProvider(calendarId)
                } else {
                    Log.e(TAG, "CalendarRepository: Error creating event on attempt $attempt: ${e.message}", e)
                    if (attempt == 3) {
                        break
                    }
                }
            }
        }
        
        // Post-event actions (centralized)
        if (eventId != null && context != null) {
            // 1. Cancel any existing notifications for this template
            if (templateId != null) {
                val displayName = templateName ?: templateId
                Log.d(TAG, "CalendarRepository: Cancelling notifications for template '$displayName' after event creation")
                // Cancel time-based notification if it exists
                HourlyReminderWorker.cancelNotificationIfExists(context, templateId, false)
                // Cancel concentration notification if it exists
                HourlyReminderWorker.cancelNotificationIfExists(context, templateId, true)
            }
            
            // 2. Update all widgets to reflect the new event
            de.ntdote.medicalcalendarlog.service.TemplateWidgetProvider.updateAllWidgets(context)
        } else if (lastException != null) {
            Log.e(TAG, "CalendarRepository: Failed to create event after 3 attempts", lastException)
        }
        
        return eventId
    }

    /**
     * Create an event from a template with optional dosage
     * This centralizes all event creation logic including title formatting
     * 
     * @param calendarId The calendar to create the event in
     * @param template The template to create an event for
     * @param dosage Optional dosage value (for templates that require input)
     * @return Event ID if successful, null otherwise
     */
    fun createEventFromTemplate(calendarId: Long, template: Template, dosage: Double? = null): Long? {
        // Create event title based on template type and dosage
        val title = when {
            // Templates with dosage: append value directly
            dosage != null -> "${template.name}$dosage"
            // Templates without dosage: use template name as-is
            else -> template.name
        }
        
        Log.d(TAG, "CalendarRepository: Creating event from template '${template.name}' with title '$title'")
        
        // Use the existing createEvent method which handles all post-event actions
        return createEvent(calendarId, title, template.id, template.name)
    }

    /**
     * Get events from a specific timestamp forward
     * Used for fixed-window checksum comparison
     */
    fun getEventsFromTimestamp(calendarId: Long, startTimestamp: Long): List<CalendarEvent> {
        if (!hasCalendarPermissions()) {
            Log.w(TAG, "CalendarRepository: Cannot get events - permissions not granted")
            return emptyList()
        }
        
        val events = mutableListOf<CalendarEvent>()
        
        val projection = arrayOf(
            CalendarContract.Events._ID,
            CalendarContract.Events.TITLE,
            CalendarContract.Events.DTSTART,
            CalendarContract.Events.DTEND,
            CalendarContract.Events.CALENDAR_ID
        )

        val selection = "${CalendarContract.Events.CALENDAR_ID} = ? AND ${CalendarContract.Events.DTSTART} >= ? AND ${CalendarContract.Events.TITLE} NOT LIKE ?"
        val selectionArgs = arrayOf(calendarId.toString(), startTimestamp.toString(), "${ConfigExportImport.BACKUP_EVENT_PREFIX}%")

        try {
            val cursor: Cursor? = contentResolver.query(
                CalendarContract.Events.CONTENT_URI,
                projection,
                selection,
                selectionArgs,
                "${CalendarContract.Events.DTSTART} ASC"
            )

            cursor?.use {
                while (it.moveToNext()) {
                    val id = it.getLong(0)
                    val title = it.getString(1) ?: ""
                    val start = Date(it.getLong(2))
                    val end = Date(it.getLong(3))
                    val calId = it.getLong(4)

                    val dosage = extractDosageFromTitle(title)
                    events.add(CalendarEvent(id, title, start, end, calId, dosage))
                }
            }
        } catch (e: SecurityException) {
            Log.e(TAG, "CalendarRepository: Security exception when accessing events", e)
            return emptyList()
        } catch (e: Exception) {
            Log.e(TAG, "CalendarRepository: Error accessing events", e)
            return emptyList()
        }

        return events
    }
    
    /**
     * Calculate CRC32 checksum for events list
     * Used to detect calendar changes without processing events
     */
    fun calculateEventsChecksum(events: List<CalendarEvent>): Long {
        val crc = CRC32()
        
        // Sort events by start time to ensure consistent ordering
        val sortedEvents = events.sortedBy { event -> event.startTime.time }
        
        // Build a deterministic string representation of events
        val eventsData = sortedEvents.joinToString(separator = "|") { event ->
            "${event.id}:${event.title}:${event.startTime.time}:${event.dosage}"
        }
        
        crc.update(eventsData.toByteArray())
        val checksum = crc.value
        
        Log.d(TAG, "CalendarRepository: Calculated checksum=$checksum for ${events.size} events")
        return checksum
    }

    private fun extractDosageFromTitle(title: String): Double {
        // Look for numbers in the title, assuming they represent dosage
        val regex = Regex("""(\d+(?:\.\d+)?)""")
        val match = regex.find(title)
        return match?.value?.toDoubleOrNull() ?: 1.0
    }
    
    /**
     * Get all configuration backups from the calendar
     * Returns list sorted by timestamp (newest first)
     */
    fun getAllBackups(calendarId: Long): List<BackupInfo> {
        if (!hasCalendarPermissions()) {
            Log.w(TAG, "CalendarRepository: Cannot get backups - permissions not granted")
            return emptyList()
        }
        
        val backups = mutableListOf<BackupInfo>()
        
        val projection = arrayOf(
            CalendarContract.Events._ID,
            CalendarContract.Events.TITLE,
            CalendarContract.Events.DTSTART,
            CalendarContract.Events.DESCRIPTION
        )

        val selection = "${CalendarContract.Events.CALENDAR_ID} = ? AND ${CalendarContract.Events.TITLE} LIKE ?"
        val selectionArgs = arrayOf(calendarId.toString(), "${ConfigExportImport.BACKUP_EVENT_PREFIX}%")

        try {
            val cursor: Cursor? = contentResolver.query(
                CalendarContract.Events.CONTENT_URI,
                projection,
                selection,
                selectionArgs,
                "${CalendarContract.Events.DTSTART} DESC" // Newest first
            )

            cursor?.use {
                while (it.moveToNext()) {
                    val eventId = it.getLong(0)
                    val title = it.getString(1) ?: ""
                    val timestamp = Date(it.getLong(2))
                    val description = it.getString(3) ?: ""
                    
                    // Parse the JSON from description to extract metadata
                    // Note: parseBackupMetadata/parseTemplateCount now handle HTML sanitization internally
                    try {
                        val metadata = ConfigExportImport.parseBackupMetadata(description)
                        val templateCount = ConfigExportImport.parseTemplateCount(description)
                        
                        backups.add(
                            BackupInfo(
                                eventId = eventId,
                                timestamp = timestamp,
                                versionCode = metadata.versionCode ?: 34, // Default to 34 for old backups
                                versionName = metadata.appVersion,
                                templateCount = templateCount,
                                jsonData = description
                            )
                        )
                    } catch (e: Exception) {
                        Log.w(TAG, "CalendarRepository: Failed to parse backup event $eventId: ${e.message}")
                        // Skip malformed backups
                    }
                }
            }
        } catch (e: SecurityException) {
            Log.e(TAG, "CalendarRepository: Security exception when accessing backup events", e)
            return emptyList()
        } catch (e: Exception) {
            Log.e(TAG, "CalendarRepository: Error accessing backup events", e)
            return emptyList()
        }

        Log.d(TAG, "CalendarRepository: Found ${backups.size} backups")
        return backups
    }
    
    /**
     * Delete a backup event from the calendar
     * @param eventId The event ID to delete
     * @return true if successful, false otherwise
     */
    fun deleteBackup(eventId: Long): Boolean {
        if (!hasCalendarPermissions()) {
            Log.w(TAG, "CalendarRepository: Cannot delete backup - permissions not granted")
            return false
        }
        
        // First, get the calendar account info for this event
        var accountName: String? = null
        var accountType: String? = null
        
        try {
            val eventUri = ContentUris.withAppendedId(CalendarContract.Events.CONTENT_URI, eventId)
            val eventCursor = contentResolver.query(
                eventUri,
                arrayOf(CalendarContract.Events.CALENDAR_ID),
                null,
                null,
                null
            )
            
            val calendarId = eventCursor?.use {
                if (it.moveToFirst()) it.getLong(0) else null
            }
            
            if (calendarId != null) {
                val calendarUri = ContentUris.withAppendedId(CalendarContract.Calendars.CONTENT_URI, calendarId)
                val calendarCursor = contentResolver.query(
                    calendarUri,
                    arrayOf(
                        CalendarContract.Calendars.ACCOUNT_NAME,
                        CalendarContract.Calendars.ACCOUNT_TYPE
                    ),
                    null,
                    null,
                    null
                )
                
                calendarCursor?.use {
                    if (it.moveToFirst()) {
                        accountName = it.getString(0)
                        accountType = it.getString(1)
                    }
                }
            }
        } catch (e: Exception) {
            Log.w(TAG, "CalendarRepository: Failed to get calendar account info: ${e.message}")
        }
        
        // Delete with CALLER_IS_SYNCADAPTER and account parameters to bypass notification
        try {
            val deleteUriBuilder = ContentUris.withAppendedId(CalendarContract.Events.CONTENT_URI, eventId)
                .buildUpon()
                .appendQueryParameter(CalendarContract.CALLER_IS_SYNCADAPTER, "true")
            
            // Add account parameters if available
            if (accountName != null && accountType != null) {
                deleteUriBuilder
                    .appendQueryParameter(CalendarContract.Calendars.ACCOUNT_NAME, accountName)
                    .appendQueryParameter(CalendarContract.Calendars.ACCOUNT_TYPE, accountType)
            }
            
            val deleteUri = deleteUriBuilder.build()
            contentResolver.delete(deleteUri, null, null)
        } catch (e: SecurityException) {
            Log.e(TAG, "CalendarRepository: Security exception when deleting backup", e)
            return false
        } catch (e: Exception) {
            // CalendarProvider may throw exceptions during notification sending (e.g.,
            // "Unknown URL content://com.android.calendar") but the deletion often still succeeds.
            // We'll verify deletion below rather than failing immediately.
            Log.w(TAG, "CalendarRepository: Exception during backup deletion (will verify): ${e.message}")
        }
        
        // Always verify the event is actually deleted by querying for it
        // This is more reliable than checking rowsDeleted since CalendarProvider
        // may throw exceptions during notification but still delete the event
        try {
            val verifyUri = ContentUris.withAppendedId(CalendarContract.Events.CONTENT_URI, eventId)
            val cursor = contentResolver.query(
                verifyUri, 
                arrayOf(CalendarContract.Events._ID), 
                null, 
                null, 
                null
            )
            
            val stillExists = cursor?.use { it.count > 0 } ?: false
            
            if (!stillExists) {
                Log.d(TAG, "CalendarRepository: Successfully deleted backup event $eventId")
                return true
            } else {
                Log.w(TAG, "CalendarRepository: Backup event $eventId still exists after deletion attempt")
                return false
            }
        } catch (e: Exception) {
            Log.e(TAG, "CalendarRepository: Error verifying backup deletion", e)
            return false
        }
    }
}
