package de.ntdote.medicalcalendarlog.data

import kotlinx.serialization.Serializable

@Serializable
data class Template(
    val id: String,
    val name: String,
    val templateType: TemplateType = TemplateType.GENERAL,
    val metricUnit: String = "", // Used for METRIC type
    val drugs: List<Drug> = emptyList(), // Used for DECAYING type - list of drugs in this template
    val extendedData: ExtendedTemplateData? = null, // DEPRECATED: Kept for migration from older versions
    val reminderEnabled: Boolean = false,
    val reminderThresholdHours: Int? = null, // DEPRECATED: Use reminderThresholdMillis instead (kept for migration)
    val reminderThresholdMillis: Long? = null, // Time in milliseconds before warning (new in v36)
    val concentrationReminderEnabled: Boolean = false, // DEPRECATED: Moved to individual drugs (kept for migration)
    val concentrationReminderThreshold: Double? = null, // DEPRECATED: Moved to individual drugs (kept for migration)
    val visible: Boolean = true, // Controls visibility in graph display
    val enabled: Boolean = true, // Controls whether template appears in UI (MainScreen, GraphItemSelector)
    val requiresInput: Boolean = false // Whether template requires user input when logging (replaces ### placeholder and unifies METRIC behavior)
) {
    /**
     * Get the reminder threshold in milliseconds.
     * Note: The old reminderThresholdHours field is deprecated and only kept for data migration.
     */
    fun getEffectiveReminderThresholdMillis(): Long? {
        return reminderThresholdMillis
    }

    /**
     * Get all drugs that have decay parameters (source drugs).
     */
    fun getSourceDrugs(): List<Drug> = drugs.filter { it.decayType != null }
    
    /**
     * Get all drugs that don't have decay parameters (derived drugs).
     */
    fun getDerivedDrugs(): List<Drug> = drugs.filter { it.decayType == null }
    
    /**
     * Check if this template has any source drugs (drugs with decay parameters).
     */
    fun hasSourceDrugs(): Boolean = drugs.any { it.decayType != null }
    
    /**
     * Check if this template has any derived drugs (drugs without decay parameters).
     */
    fun hasDerivedDrugs(): Boolean = drugs.any { it.decayType == null }
    
    /**
     * Check if this template has exactly one drug.
     */
    fun isSingleDrug(): Boolean = drugs.size == 1
    
    /**
     * Check if this template requires source template lookup for derived drugs.
     */
    fun requiresSourceLookup(): Boolean = getDerivedDrugs().isNotEmpty()
    
    /**
     * Get all drug types in this template.
     */
    fun getDrugTypes(): List<String> = drugs.map { it.drugType }
    
    /**
     * Get display name for this template showing all drug types.
     * For single drug: returns the drug type
     * For multiple drugs: returns comma-separated list
     */
    fun getDrugTypesDisplay(): String {
        val drugTypes = getDrugTypes()
        return when (drugTypes.size) {
            0 -> "No drugs"
            1 -> drugTypes.first()
            else -> drugTypes.joinToString(", ")
        }
    }
}

@Serializable
data class Drug(
    val drugType: String,                       // e.g., "Aspirin" - the name/type of the drug
    val factor: Double,                         // e.g., 500.0 - amount per unit (dosage)
    val unit: String,                           // e.g., "mg" - unit of measurement
    val decayType: DecayType? = null,           // HALF_LIFE or CONSTANT - null for derived drugs
    val hourlyDecayRate: Double? = null,        // For CONSTANT decay type
    val halfLifeHours: Double? = null,          // For HALF_LIFE decay type
    val concentrationReminderEnabled: Boolean = false,  // Individual concentration-based reminder
    val concentrationReminderThreshold: Double? = null, // Threshold for concentration reminder
    val visible: Boolean = true                 // Controls visibility in graph display (per drug)
) {
    init {
        // Ensure only relevant decay parameters are set based on decay type
        when (decayType) {
            DecayType.CONSTANT -> {
                require(hourlyDecayRate != null) { "hourlyDecayRate must be set for CONSTANT decay" }
                require(halfLifeHours == null) { "halfLifeHours must be null for CONSTANT decay" }
            }
            DecayType.HALF_LIFE -> {
                require(halfLifeHours != null) { "halfLifeHours must be set for HALF_LIFE decay" }
                require(hourlyDecayRate == null) { "hourlyDecayRate must be null for HALF_LIFE decay" }
            }
            null -> {
                // Derived drug - no decay parameters required, they will be resolved from source template
                require(hourlyDecayRate == null) { "hourlyDecayRate must be null for derived drugs" }
                require(halfLifeHours == null) { "halfLifeHours must be null for derived drugs" }
            }
        }
    }
    
    companion object {
        /**
         * Create a source Drug with decay parameters
         */
        fun createSource(
            drugType: String,
            factor: Double,
            unit: String,
            decayType: DecayType,
            hourlyDecayRate: Double? = null,
            halfLifeHours: Double? = null,
            concentrationReminderEnabled: Boolean = false,
            concentrationReminderThreshold: Double? = null,
            visible: Boolean = true
        ): Drug {
            return when (decayType) {
                DecayType.CONSTANT -> Drug(
                    drugType = drugType,
                    factor = factor,
                    unit = unit,
                    decayType = decayType,
                    hourlyDecayRate = hourlyDecayRate,
                    halfLifeHours = null, // Force null for constant decay
                    concentrationReminderEnabled = concentrationReminderEnabled,
                    concentrationReminderThreshold = concentrationReminderThreshold,
                    visible = visible
                )
                DecayType.HALF_LIFE -> Drug(
                    drugType = drugType,
                    factor = factor,
                    unit = unit,
                    decayType = decayType,
                    hourlyDecayRate = null, // Force null for half-life decay
                    halfLifeHours = halfLifeHours,
                    concentrationReminderEnabled = concentrationReminderEnabled,
                    concentrationReminderThreshold = concentrationReminderThreshold,
                    visible = visible
                )
            }
        }

        /**
         * Create a derived Drug without decay parameters (will reference source template)
         */
        fun createDerived(
            drugType: String,
            factor: Double,
            unit: String,
            concentrationReminderEnabled: Boolean = false,
            concentrationReminderThreshold: Double? = null,
            visible: Boolean = true
        ): Drug {
            return Drug(
                drugType = drugType,
                factor = factor,
                unit = unit,
                decayType = null, // Derived drug
                hourlyDecayRate = null,
                halfLifeHours = null,
                concentrationReminderEnabled = concentrationReminderEnabled,
                concentrationReminderThreshold = concentrationReminderThreshold,
                visible = visible
            )
        }

    }
}

// DEPRECATED: Old multi-drug component structure - kept for migration only
@Serializable
data class MultiDrugComponent(
    val drugName: String,
    val amountPerUnit: Double,
    val unit: String = "mg"
)

// DEPRECATED: Old extended template data structure - kept for migration only
@Serializable
data class ExtendedTemplateData(
    val factor: Double?,
    val unit: String,
    val typeOfDrug: String,
    val decayType: DecayType?,
    val hourlyDecayRate: Double? = null, // For constant decay
    val halfLifeHours: Double? = null, // For half-life decay
    val multiDrugComponents: List<MultiDrugComponent>? = null // Old multi-drug support
) {
    init {
        // Ensure only relevant decay parameters are set based on decay type
        when (decayType) {
            DecayType.CONSTANT -> {
                require(hourlyDecayRate != null) { "hourlyDecayRate must be set for CONSTANT decay" }
                require(halfLifeHours == null) { "halfLifeHours must be null for CONSTANT decay" }
            }
            DecayType.HALF_LIFE -> {
                require(halfLifeHours != null) { "halfLifeHours must be set for HALF_LIFE decay" }
                require(hourlyDecayRate == null) { "hourlyDecayRate must be null for HALF_LIFE decay" }
            }
            null -> {
                // Both can be null if no decay type is specified
            }
        }
    }
    
    companion object {
        /**
         * Create ExtendedTemplateData with proper nulling based on decay type
         */
        fun create(
            factor: Double?,
            unit: String,
            typeOfDrug: String,
            decayType: DecayType?,
            hourlyDecayRate: Double? = null,
            halfLifeHours: Double? = null,
            multiDrugComponents: List<MultiDrugComponent>? = null
        ): ExtendedTemplateData {
            return when (decayType) {
                DecayType.CONSTANT -> ExtendedTemplateData(
                    factor = factor,
                    unit = unit,
                    typeOfDrug = typeOfDrug,
                    decayType = decayType,
                    hourlyDecayRate = hourlyDecayRate,
                    halfLifeHours = null, // Force null for constant decay
                    multiDrugComponents = multiDrugComponents
                )
                DecayType.HALF_LIFE -> ExtendedTemplateData(
                    factor = factor,
                    unit = unit,
                    typeOfDrug = typeOfDrug,
                    decayType = decayType,
                    hourlyDecayRate = null, // Force null for half-life decay
                    halfLifeHours = halfLifeHours,
                    multiDrugComponents = multiDrugComponents
                )
                null -> ExtendedTemplateData(
                    factor = factor,
                    unit = unit,
                    typeOfDrug = typeOfDrug,
                    decayType = null,
                    hourlyDecayRate = null,
                    halfLifeHours = null,
                    multiDrugComponents = multiDrugComponents
                )
            }
        }
    }
}

@Serializable
enum class TemplateType {
    METRIC,    // Absolute values (weight, mood, etc.) - typically requires input
    DECAYING,  // Medication with concentration tracking
    GENERAL    // Simple medication logging without decay
}

@Serializable
enum class DecayType {
    CONSTANT,
    HALF_LIFE
}
