package de.ntdote.medicalcalendarlog.data

import androidx.compose.ui.graphics.Color
import java.util.Date

/**
 * Sealed class representing different types of items that can be graphed
 */
sealed class GraphableItem {
    abstract val id: String
    abstract val name: String
    abstract val color: Color
    abstract val isSelected: Boolean
    
    data class GeneralTemplate(
        override val id: String,
        override val name: String,
        override val color: Color,
        override val isSelected: Boolean = true,
        val template: Template
    ) : GraphableItem()
    
    data class MetricTemplate(
        override val id: String,
        override val name: String,
        override val color: Color,
        override val isSelected: Boolean = true,
        val template: Template,
        val unit: String
    ) : GraphableItem()
    
    data class Drug(
        override val id: String,
        override val name: String,
        override val color: Color,
        override val isSelected: Boolean = true,
        val drugType: String,
        val templates: List<Template>,
        val unit: String
    ) : GraphableItem()
    
    data class DecayingTemplate(
        override val id: String,
        override val name: String,
        override val color: Color,
        override val isSelected: Boolean = true,
        val template: Template
    ) : GraphableItem()
}

/**
 * Data class for graph points with time and value
 */
data class GraphPoint(
    val time: Date,
    val value: Double,
    val event: CalendarEvent? = null
)

/**
 * Data class for graph data with scaling information
 */
data class GraphData(
    val item: GraphableItem,
    val points: List<GraphPoint>,
    val minValue: Double = 0.0,
    val maxValue: Double,
    val unit: String
)

/**
 * Data class for graph state management
 */
data class GraphState(
    val items: List<GraphableItem> = emptyList(),
    val daysBack: Int = 90,
    val zoomLevel: Float = 1f,
    val panOffset: Float = 0f,
    val showEventText: Boolean = true,
    val startTime: Date = Date(System.currentTimeMillis() - 90 * 24 * 60 * 60 * 1000L),
    val endTime: Date = Date()
)
