package de.ntdote.medicalcalendarlog

import android.Manifest
import android.app.AlertDialog
import android.content.Intent
import android.content.pm.PackageManager
import android.net.Uri
import android.os.Bundle
import android.provider.Settings
import android.util.Log
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.activity.enableEdgeToEdge
import androidx.activity.result.contract.ActivityResultContracts
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Surface
import androidx.compose.runtime.Composable
import androidx.compose.runtime.mutableStateOf
import androidx.compose.ui.Modifier
import androidx.core.content.ContextCompat
import androidx.navigation.compose.rememberNavController
import de.ntdote.medicalcalendarlog.navigation.MedicalCalendarNavigation
import de.ntdote.medicalcalendarlog.service.ReminderService
import de.ntdote.medicalcalendarlog.service.TemplateWidgetProvider
import de.ntdote.medicalcalendarlog.ui.theme.MedicalCalendarlogTheme
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch

class MainActivity : ComponentActivity() {
    
    // Track permission state
    private val hasRequiredPermissions = mutableStateOf(false)
    
    private val requestPermissionLauncher = registerForActivityResult(
        ActivityResultContracts.RequestMultiplePermissions()
    ) { permissions ->
        // Check if all required permissions are granted
        val allGranted = permissions.entries.all { it.value }
        
        if (allGranted) {
            Log.d("MCL", "All permissions granted")
            hasRequiredPermissions.value = true
            initializeServices()
        } else {
            Log.d("MCL", "Some permissions denied: $permissions")
            hasRequiredPermissions.value = false
            showPermissionExplanationDialog()
        }
    }
    
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        enableEdgeToEdge()
        
        // Check if hourly worker needs to run due to user interaction
        de.ntdote.medicalcalendarlog.service.HourlyReminderWorker.checkAndRunWorkerIfNeeded(this, "MainActivity")
        
        // Check and request permissions
        if (checkCalendarPermissions()) {
            hasRequiredPermissions.value = true
            initializeServices()
        } else {
            requestCalendarPermissions()
        }
        
        setContent {
            MedicalCalendarlogTheme {
                Surface(
                    modifier = Modifier.fillMaxSize(),
                    color = MaterialTheme.colorScheme.background
                ) {
                    MedicalCalendarApp()
                }
            }
        }
    }
    
    override fun onDestroy() {
        super.onDestroy()
        // Clean up reminder service if it was initialized
        if (hasRequiredPermissions.value) {
            ReminderService.cleanup(this)
        }
    }
    
    /**
     * Check if all required calendar permissions are granted
     */
    private fun checkCalendarPermissions(): Boolean {
        val permissions = getRequiredPermissions()
        
        return permissions.all {
            ContextCompat.checkSelfPermission(this, it) == PackageManager.PERMISSION_GRANTED
        }
    }
    
    /**
     * Request calendar and notification permissions
     */
    private fun requestCalendarPermissions() {
        val permissions = getRequiredPermissions()
        
        val permissionsToRequest = permissions.filter {
            ContextCompat.checkSelfPermission(this, it) != PackageManager.PERMISSION_GRANTED
        }
        
        if (permissionsToRequest.isNotEmpty()) {
            requestPermissionLauncher.launch(permissionsToRequest.toTypedArray())
        } else {
            hasRequiredPermissions.value = true
            initializeServices()
        }
    }
    
    /**
     * Get list of required permissions
     */
    private fun getRequiredPermissions(): Array<String> {
        return arrayOf(
            Manifest.permission.READ_CALENDAR,
            Manifest.permission.WRITE_CALENDAR,
            Manifest.permission.POST_NOTIFICATIONS
        )
    }
    
    /**
     * Initialize services that require permissions
     */
    private fun initializeServices() {
        // ReminderService ContentObserver is now registered in Application.onCreate()
        // We only need to trigger an initial recalculation here
        CoroutineScope(Dispatchers.IO).launch {
            ReminderService.recalculateAllReminders(this@MainActivity, "permissions_granted")
        }
        
        // Schedule the hourly worker in the Application class
        (application as MedicalCalendarLogApplication).scheduleHourlyWorker()
    }

    
    /**
     * Check if there are any scheduled reminder workers
     */
    private fun hasScheduledReminders(): Boolean {
        return try {
            val workManager = androidx.work.WorkManager.getInstance(this)
            val workInfos = workManager.getWorkInfosByTag("reminder").get()
            val hasReminders = workInfos.any { !it.state.isFinished }
            Log.d("MCL", "Checking for scheduled reminders: $hasReminders (${workInfos.size} work items found)")
            hasReminders
        } catch (e: Exception) {
            Log.e("MCL", "Error checking for scheduled reminders", e)
            false // If we can't check, assume no reminders and initialize
        }
    }
    
    /**
     * Show dialog explaining why permissions are needed
     */
    private fun showPermissionExplanationDialog() {
        AlertDialog.Builder(this)
            .setTitle("Permissions Required")
            .setMessage("This app needs calendar permissions to function properly. Without these permissions, the app cannot read or create calendar events.")
            .setPositiveButton("Settings") { _, _ ->
                // Open app settings so user can grant permissions
                val intent = Intent(Settings.ACTION_APPLICATION_DETAILS_SETTINGS)
                val uri = Uri.fromParts("package", packageName, null)
                intent.data = uri
                startActivity(intent)
            }
            .setNegativeButton("Exit") { _, _ ->
                // Close the app if permissions are denied
                finish()
            }
            .setCancelable(false)
            .show()
    }
}

@Composable
fun MedicalCalendarApp() {
    val navController = rememberNavController()
    MedicalCalendarNavigation(navController = navController)
}
