#!/bin/bash 

# Medical Calendar Log - Unified Release Tool
# Interactive script to automate releases for Google Play, F-Droid, and Codeberg

set -e  # Exit on any error

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
BLUE='\033[0;34m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# Function to print colored output
print_info() {
    echo -e "${BLUE}ℹ️  $1${NC}"
}

print_success() {
    echo -e "${GREEN}✅ $1${NC}"
}

print_warning() {
    echo -e "${YELLOW}⚠️  $1${NC}"
}

print_error() {
    echo -e "${RED}❌ $1${NC}"
}

# Function to extract version from build.gradle.kts
get_current_version() {
    local version_name=$(grep 'versionName = ' app/build.gradle.kts | sed 's/.*versionName = "\([^"]*\)".*/\1/')
    local version_code=$(grep 'versionCode = ' app/build.gradle.kts | sed 's/.*versionCode = \([0-9]*\).*/\1/')
    echo "$version_name:$version_code"
}

# Function to increment patch version
increment_patch_version() {
    local version=$1
    local major=$(echo $version | cut -d. -f1)
    local minor=$(echo $version | cut -d. -f2)
    local patch=$(echo $version | cut -d. -f3)
    local new_patch=$((patch + 1))
    echo "$major.$minor.$new_patch"
}

# Function to validate version format
validate_version() {
    local version=$1
    if [[ ! $version =~ ^[0-9]+\.[0-9]+\.[0-9]+$ ]]; then
        return 1
    fi
    return 0
}

# Function to update build.gradle.kts
update_build_gradle() {
    local new_version=$1
    local new_code=$2
    
    # Create backup
    cp app/build.gradle.kts app/build.gradle.kts.backup
    
    # Update version name and code
    sed -i "s/versionName = \".*\"/versionName = \"$new_version\"/" app/build.gradle.kts
    sed -i "s/versionCode = .*/versionCode = $new_code/" app/build.gradle.kts
    
    print_success "Updated build.gradle.kts to version $new_version (code: $new_code)"
}

# Function to build signed APK (for both Codeberg and F-Droid)
build_signed_apk() {
    local version=$1
    local keystore_path=$2
    local password=$3
    
    print_info "🔨 Building signed APK for version $version..."
    
    # Set environment variables for signing
    export ENABLE_SIGNING=true
    export KEYSTORE_PATH="$keystore_path"
    export KEYSTORE_PASSWORD="$password"
    export KEY_ALIAS="my-key-alias"
    export KEY_PASSWORD="$password"
    
    # Clean previous builds
    ./gradlew clean
    
    # Build with proper NDK configuration
    print_info "Running build with explicit NDK path..."
    ./gradlew assembleRelease --stacktrace
    
    # Check if APK was built successfully
    local apk_path="app/build/outputs/apk/release/app-release.apk"
    if [[ ! -f "$apk_path" ]]; then
        print_error "Failed to build APK at $apk_path"
        return 1
    fi
    
    # Create releases directory if it doesn't exist
    mkdir -p releases
    
    # Rename APK to include version
    local versioned_apk="releases/MedicalCalendarLog-${version}-signed.apk"
    cp "$apk_path" "$versioned_apk"
    
    print_success "APK built successfully: $versioned_apk"
    echo "$versioned_apk"
}

# Function to build Google Play AAB
build_google_play_aab() {
    local version=$1
    local keystore_path=$2
    local password=$3
    
    print_info "🔨 Building Google Play AAB for version $version..."
    
    # Set environment variables for signing
    export ENABLE_SIGNING=true
    export KEYSTORE_PATH="$keystore_path"
    export KEYSTORE_PASSWORD="$password"
    export KEY_ALIAS="my-key-alias"
    export KEY_PASSWORD="$password"
    
    # Build the Play Store bundle with signing
    ./gradlew bundleRelease
    
    # Check if AAB was built successfully
    local aab_path="app/build/outputs/bundle/release/app-release.aab"
    if [[ ! -f "$aab_path" ]]; then
        print_error "Failed to build AAB at $aab_path"
        return 1
    fi
    
    # Create releases directory if it doesn't exist
    mkdir -p releases
    
    # Rename AAB to include version
    local versioned_aab="releases/MedicalCalendarLog-${version}.aab"
    cp "$aab_path" "$versioned_aab"
    
    print_success "AAB built successfully: $versioned_aab"
    echo "$versioned_aab"
}

# Function to upload AAB to Google Play
upload_to_google_play() {
    local aab_path=$1
    local version=$2
    local version_code=$3
    
    print_info "🚀 Uploading AAB to Google Play..."
    
    # Check if fastlane is installed
    if ! command -v fastlane &> /dev/null; then
        print_warning "fastlane not found. Please install fastlane to enable automatic Google Play uploads."
        print_info "Manual upload instructions:"
        echo "1. Go to Google Play Console: https://play.google.com/console"
        echo "2. Select your app: Medical Calendar Log"
        echo "3. Go to Production > Release dashboard > Create new release"
        echo "4. Upload the AAB file: $aab_path"
        echo "5. Add release notes and submit for review"
        return 1
    fi
    
    # Check if Google Play API key exists
    local api_key_path="/home/sven/AndroidStudioProjects/play-console-key.json"
    if [[ ! -f "$api_key_path" ]]; then
        print_error "Google Play API key not found at $api_key_path"
        print_info "Please ensure your Google Play API key is available at this location."
        return 1
    fi
    
    # Create temporary fastlane configuration if it doesn't exist
    local fastlane_dir="fastlane"
    local appfile="$fastlane_dir/Appfile"
    
    mkdir -p "$fastlane_dir"
    
    # Create or update Appfile with correct package name and JSON key path
    if [[ ! -f "$appfile" ]] || ! grep -q "json_key_file" "$appfile"; then
        echo "# Automatically generated by unified-release.sh" > "$appfile"
        echo "package_name(\"de.ntdote.medicalcalendarlog\")" >> "$appfile"
        echo "json_key_file(\"$api_key_path\")" >> "$appfile"
    fi
    
    # Use fastlane supply to upload the AAB
    print_info "Uploading to Google Play using fastlane supply..."
    
    # Run fastlane supply with error handling
    if fastlane supply \
        --aab "$aab_path" \
        --track production \
        --release_status completed \
        --rollout 1.0 \
        --skip_upload_metadata \
        --skip_upload_images \
        --skip_upload_screenshots \
        --skip_upload_changelogs; then
        
        print_success "Successfully uploaded AAB to Google Play!"
        print_info "The app will be available to 100% of users after Google's review process."
        return 0
    else
        print_error "Failed to upload AAB to Google Play."
        print_info "Manual upload instructions:"
        echo "1. Go to Google Play Console: https://play.google.com/console"
        echo "2. Select your app: Medical Calendar Log"
        echo "3. Go to Production > Release dashboard > Create new release"
        echo "4. Upload the AAB file: $aab_path"
        echo "5. Add release notes and submit for review"
        return 1
    fi
}

# Function to create changelog files
create_changelog_files() {
    local version_code=$1
    local changelog_de=$2
    local changelog_en=$3
    
    # Create German changelog
    local de_changelog_file="fastlane/metadata/android/de/changelogs/${version_code}.txt"
    echo "$changelog_de" > "$de_changelog_file"
    print_success "Created German changelog: $de_changelog_file"
    
    # Create English changelog
    local en_changelog_file="fastlane/metadata/android/en-US/changelogs/${version_code}.txt"
    echo "$changelog_en" > "$en_changelog_file"
    print_success "Created English changelog: $en_changelog_file"
    
    return 0
}

# Function to get previous changelog
get_previous_changelog() {
    local prev_version_code=$1
    local lang=$2
    
    local changelog_file="fastlane/metadata/android/${lang}/changelogs/${prev_version_code}.txt"
    
    if [[ -f "$changelog_file" ]]; then
        cat "$changelog_file"
    else
        echo ""
    fi
}

# Function to get changelog input from user
get_changelog_input() {
    local version=$1
    local prev_version_code=$2
    
    # Default changelog values
    local default_changelog_de="Fehlerbehebungen und Stabilitätsverbesserungen"
    local default_changelog_en="Bug fixes and stability improvements"
    
    echo
    print_info "📝 Changelog Management"
    echo "Creating changelog entries for version $version"
    echo
    
    # Check for command line arguments for non-interactive mode
    if [[ "$USE_DEFAULT_CHANGELOG" == "true" ]]; then
        CHANGELOG_DE="$default_changelog_de"
        CHANGELOG_EN="$default_changelog_en"
        print_info "Using default changelog entries"
        return 0
    fi
    
    # Check for rerelease mode - reuse previous changelogs
    if [[ "$RERELEASE_MODE" == "true" ]] && [[ -n "$prev_version_code" ]]; then
        local prev_changelog_de=$(get_previous_changelog "$prev_version_code" "de")
        local prev_changelog_en=$(get_previous_changelog "$prev_version_code" "en-US")
        
        if [[ -n "$prev_changelog_de" ]] && [[ -n "$prev_changelog_en" ]]; then
            # Reuse previous changelogs without prefix
            CHANGELOG_DE="${prev_changelog_de}"
            CHANGELOG_EN="${prev_changelog_en}"
            
            print_info "🔄 Rerelease mode: Reusing previous changelogs"
            echo "German: $CHANGELOG_DE"
            echo "English: $CHANGELOG_EN"
            return 0
        else
            print_warning "Previous changelogs not found, using defaults"
            CHANGELOG_DE="$default_changelog_de"
            CHANGELOG_EN="$default_changelog_en"
            return 0
        fi
    fi
    
    # German changelog with simple input
    print_info "🇩🇪 German changelog entry:"
    echo "Default: $default_changelog_de"
    read -p "Enter German changelog (or press Enter for default): " changelog_de
    
    if [[ -z "$changelog_de" ]]; then
        changelog_de="$default_changelog_de"
        print_info "Using default German changelog"
    fi
    
    # English changelog with simple input
    print_info "🇺🇸 English changelog entry:"
    echo "Default: $default_changelog_en"
    read -p "Enter English changelog (or press Enter for default): " changelog_en
    
    if [[ -z "$changelog_en" ]]; then
        changelog_en="$default_changelog_en"
        print_info "Using default English changelog"
    fi
    
    echo
    print_info "📋 Changelog Preview:"
    echo "German: $changelog_de"
    echo "English: $changelog_en"
    echo
    
    # Confirmation
    read -p "Are these changelog entries correct? (Y/n): " -n 1 -r
    echo
    if [[ $REPLY =~ ^[Nn]$ ]]; then
        print_warning "Restarting changelog input..."
        get_changelog_input "$version"
        return $?
    fi
    
    # Return values via global variables (bash limitation)
    CHANGELOG_DE="$changelog_de"
    CHANGELOG_EN="$changelog_en"
    
    return 0
}

# Function to create Codeberg release using tea CLI
create_codeberg_release() {
    local version=$1
    local apk_path=$2
    local changelog_en=$3
    local release_message=$4
    
    print_info "Creating Codeberg release using tea CLI..."
    
    # Check if tea is installed
    if ! command -v tea &> /dev/null; then
        print_warning "tea CLI not found. Falling back to manual instructions."
        create_codeberg_release_manual "$version" "$apk_path" "$changelog_en"
        return 1
    fi
    
    # Create release with tea CLI
    print_info "Creating release $version on Codeberg..."
    tea releases create \
        --repo "svewa/MedicalCalendarLog" \
        --tag "$version" \
        --title "Release $version: $release_message" \
        --note "$changelog_en" \
        --asset "$apk_path"
    
    if [ $? -eq 0 ]; then
        print_success "Codeberg release created successfully!"
    else
        print_warning "Failed to create Codeberg release with tea CLI. Falling back to manual instructions."
        print_info "You may need to run: tea auth login --hostname codeberg.org"
        create_codeberg_release_manual "$version" "$apk_path" "$changelog_en"
        return 1
    fi
    
    return 0
}

# Function to provide Codeberg release instructions (manual fallback)
create_codeberg_release_manual() {
    local version=$1
    local apk_path=$2
    local changelog_en=$3
    
    print_info "📋 Manual Codeberg Release Instructions"
    echo
    print_info "For F-Droid deterministic builds, attach the APK to the git tag:"
    echo
    echo "1. 🌐 Go to: https://codeberg.org/svewa/MedicalCalendarLog/tags"
    echo "2. 🔍 Find the tag: $version (will appear after git push)"
    echo "3. 📝 Click on the tag to view tag details"
    echo "4. 🎯 Look for 'Create Release' or 'Edit' button on the tag page"
    echo "5. 📄 Add release notes:"
    echo "   $changelog_en"
    echo "6. 📎 Upload APK: $apk_path"
    echo "7. ✅ Save the release"
    echo
    print_success "APK ready for upload: $apk_path"
    print_info "This APK enables F-Droid's reproducible build verification"
    print_info "F-Droid will compare their build with your uploaded APK"
    
    return 0
}

## Function to update fdroiddata metadata
#update_fdroiddata_metadata() {
#    local version=$1
#    local version_code=$2
#    local commit_hash=$3
#    
#    local metadata_file="fdroiddata/metadata/de.ntdote.medicalcalendarlog.yml"
#    
#    if [[ ! -f "$metadata_file" ]]; then
#        print_warning "F-Droid metadata file not found at $metadata_file"
#        print_info "Skipping fdroiddata update. You may need to update it manually."
#        return 1
#    fi
#    
#    print_info "📝 Updating F-Droid metadata file..."
#    
#    # Create backup
#    cp "$metadata_file" "${metadata_file}.backup"
#    
#    # Update CurrentVersion and CurrentVersionCode
#    sed -i "s/^CurrentVersion: .*/CurrentVersion: $version/" "$metadata_file"
#    sed -i "s/^CurrentVersionCode: .*/CurrentVersionCode: $version_code/" "$metadata_file"
#    
#    # Replace ALL Build entries with only the latest one
#    # This ensures F-Droid only builds from the correct commit
#    print_info "Replacing all Build entries with latest version $version..."
#    
#    local temp_file="${metadata_file}.tmp"
#    awk -v version="$version" -v code="$version_code" -v commit="$commit_hash" '
#    /^Builds:/ { 
#        # Found Builds section - print header and new entry
#        print "Builds:"
#        print "  - versionName: " version
#        print "    versionCode: " code
#        print "    commit: " commit
#        print "    subdir: app"
#        print "    gradle:"
#        print "      - yes"
#        print ""
#        in_builds=1
#        next
#    }
#    in_builds && /^[A-Z]/ { 
#        # We hit the next section after Builds, stop skipping
#        in_builds=0
#    }
#    !in_builds { print }
#    ' "$metadata_file" > "$temp_file"
#    
#    mv "$temp_file" "$metadata_file"
#    
#    print_success "Updated F-Droid metadata file (replaced all builds with latest)"
#    
#    # Show diff
#    print_info "Changes to metadata file:"
#    diff -u "${metadata_file}.backup" "$metadata_file" || true
#    
#    # Commit and push to fdroiddata if it's a git repository
#    if git -C fdroiddata rev-parse --git-dir > /dev/null 2>&1; then
#        print_info "Committing fdroiddata changes..."
#        
#        # Add and commit the metadata file (run git commands inside fdroiddata directory)
#        git -C fdroiddata add metadata/de.ntdote.medicalcalendarlog.yml
#        git -C fdroiddata commit -m "Update Medical Calendar Log to $version ($version_code)" || {
#            print_warning "No changes to commit in fdroiddata (file may already be up to date)"
#            rm -f "${metadata_file}.backup"
#            return 0
#        }
#        
#        # Ask user if they want to push
#        read -p "Push fdroiddata changes to remote? (Y/n): " -n 1 -r
#        echo
#        if [[ ! $REPLY =~ ^[Nn]$ ]]; then
#            git -C fdroiddata push || {
#                print_warning "Failed to push fdroiddata changes. You may need to push manually."
#                return 1
#            }
#            print_success "Pushed fdroiddata changes to remote"
#        else
#            print_info "Skipped pushing fdroiddata changes. Remember to push manually later."
#        fi
#    else
#        print_info "fdroiddata directory is not a git repository. Skipping git operations."
#    fi
#    
#    # Clean up backup
#    rm -f "${metadata_file}.backup"
#    
#    return 0
#}

# Function to create git tag and push
create_release() {
    local version=$1
    local version_code=$2
    local message=$3
    local changelog_de=$4
    local changelog_en=$5
    local apk_path=$6
    
    # Check if we have uncommitted changes (except build.gradle.kts and changelogs)
    if git diff --quiet HEAD -- . ':!app/build.gradle.kts' ':!fastlane/metadata/android/*/changelogs/*'; then
        print_info "No uncommitted changes detected (except version bump and changelogs)"
    else
        print_warning "You have uncommitted changes. Please commit them first."
        git status --porcelain | grep -v -E '(app/build.gradle.kts|fastlane/metadata/android/.*/changelogs/)' || true
        read -p "Continue anyway? (y/N): " -n 1 -r
        echo
        if [[ ! $REPLY =~ ^[Yy]$ ]]; then
            print_error "Release cancelled"
            # Restore backup
            mv app/build.gradle.kts.backup app/build.gradle.kts
            # Remove created changelog files
            rm -f "fastlane/metadata/android/de/changelogs/${version_code}.txt"
            rm -f "fastlane/metadata/android/en-US/changelogs/${version_code}.txt"
            exit 1
        fi
    fi
    
    # Create changelog files
    create_changelog_files "$version_code" "$changelog_de" "$changelog_en"
    
    # Add and commit all changes
    git add app/build.gradle.kts
    git add "fastlane/metadata/android/de/changelogs/${version_code}.txt"
    git add "fastlane/metadata/android/en-US/changelogs/${version_code}.txt"
    git commit -m "Release version $version

$changelog_en"
    
    # Create tag WITHOUT 'v' prefix (F-Droid requirement)
    print_info "Creating git tag: $version (without 'v' prefix for F-Droid compatibility)"
    git tag -a "$version" -m "$message"
    
    # Push commits and tags
    print_info "Pushing to repository..."
    git push origin main
    git push origin "$version"
    
    # Create Codeberg release with APK
    create_codeberg_release "$version" "$apk_path" "$changelog_en"
    
    # Get the commit hash for the tag we just created
    local commit_hash=$(git rev-parse "$version")
    
#    # Update fdroiddata metadata
#    print_info "Updating F-Droid metadata..."
#    update_fdroiddata_metadata "$version" "$version_code" "$commit_hash"
    
    # Clean up backup
    rm -f app/build.gradle.kts.backup
    
    print_success "Release $version complete!"
    print_info "✅ Git tag created: $version (no 'v' prefix)"
    print_info "✅ APK built and ready: $apk_path"
    print_info "✅ F-Droid will automatically build and publish within 24 hours"
    print_info " Monitor progress at: https://monitor.f-droid.org/builds/build"
    
    if [[ -f "$apk_path" ]]; then
        print_info "📱 APK available for manual distribution: $apk_path"
    fi
}

# Main script
main() {
    # Parse command line arguments
    USE_DEFAULT_CHANGELOG="false"
    RERELEASE_MODE="false"
    
    while [[ $# -gt 0 ]]; do
        case $1 in
            --default-changelog)
                USE_DEFAULT_CHANGELOG="true"
                shift
                ;;
            --rerelease)
                RERELEASE_MODE="true"
                shift
                ;;
            --help|-h)
                echo "Usage: $0 [OPTIONS]"
                echo "Options:"
                echo "  --default-changelog    Use default changelog entries without prompting"
                echo "  --rerelease            Quick bug-fix release: auto-increment patch, reuse changelogs"
                echo "  --help, -h             Show this help message"
                exit 0
                ;;
            *)
                print_error "Unknown option: $1"
                echo "Use --help for usage information"
                exit 1
                ;;
        esac
    done
    
    echo
    echo "📱 Medical Calendar Log - Unified Release Tool"
    echo "=============================================="
    echo "🆕 Features:"
    echo "  • Builds a single signed APK for both Codeberg and F-Droid"
    echo "  • Builds and uploads Google Play AAB automatically"
    echo "  • Creates Codeberg releases with APK attachments"
    echo "  • Uses correct tag naming (no 'v' prefix) for F-Droid"
    echo "  • Quick bug-fix rereleases (--rerelease)"
    echo "  • Default changelog support (--default-changelog)"
    echo
    
    # Check if we're in the right directory
    if [[ ! -f "app/build.gradle.kts" ]]; then
        print_error "app/build.gradle.kts not found. Please run this script from the project root."
        exit 1
    fi
    
    # Check if gradlew exists
    if [[ ! -f "./gradlew" ]]; then
        print_error "gradlew not found. Please ensure you're in the Android project root."
        exit 1
    fi
    
    # Get current version
    local current_info=$(get_current_version)
    local current_version=$(echo $current_info | cut -d: -f1)
    local current_code=$(echo $current_info | cut -d: -f2)
    
    print_info "Current version: $current_version (code: $current_code)"
    
    # Suggest next version
    local suggested_version=$(increment_patch_version $current_version)
    local suggested_code=$((current_code + 1))
    
    print_info "Suggested version: $suggested_version (code: $suggested_code)"
    echo
    
    # In rerelease mode, auto-accept suggested version
    if [[ "$RERELEASE_MODE" == "true" ]]; then
        new_version=$current_version
        print_success "🔄 Rerelease mode: using $new_version"
    else
        # Get new version from user (default to current version)
        read -p "Enter release version [$current_version]: " new_version
        if [[ -z "$new_version" ]]; then
            new_version=$current_version
        fi
    fi
    
    # Validate version format
    if ! validate_version "$new_version"; then
        print_error "Invalid version format. Please use semantic versioning (e.g., 1.0.4)"
        exit 1
    fi
    
    # Calculate new version code (increment only if version changed)
    local new_code
    if [[ "$new_version" == "$current_version" ]]; then
        new_code=$current_code
        print_info "Releasing current version $new_version (code: $new_code)"
    else
        new_code=$((current_code + 1))
        print_info "Version will change: $current_version → $new_version (code: $current_code → $new_code)"
    fi
    
    # Get changelog input (will auto-reuse in rerelease mode)
    get_changelog_input "$new_version" "$current_code"
    
    # Get release message
    local default_message="Release version $new_version"
    if [[ "$RERELEASE_MODE" == "true" ]]; then
        release_message="$default_message"
        print_info "Using default release message: $release_message"
    else
        read -p "Enter release message [$default_message]: " release_message
        if [[ -z "$release_message" ]]; then
            release_message=$default_message
        fi
    fi
    
    # Get keystore path
    local default_keystore_path="/home/sven/AndroidStudioProjects/my-release-key.keystore"
    if [[ "$RERELEASE_MODE" == "true" ]]; then
        keystore_path=$default_keystore_path
        print_info "Using default keystore: $keystore_path"
    else
        read -p "Enter keystore path [$default_keystore_path]: " keystore_path
        if [[ -z "$keystore_path" ]]; then
            keystore_path=$default_keystore_path
        fi
    fi
    
    # Check if keystore exists
    if [ ! -f "$keystore_path" ]; then
        print_error "Error: Keystore not found at $keystore_path"
        exit 1
    fi
    
    # Prompt for password (hidden input)
    echo "Enter keystore password (will be used for both keystore and key):"
    read -s PASSWORD
    echo
    
    # Validate password is not empty
    if [ -z "$PASSWORD" ]; then
        print_error "Error: Password cannot be empty"
        exit 1
    fi
    
    echo
    print_info "📋 Release Summary:"
    echo "  Version: $current_version → $new_version"
    echo "  Code: $current_code → $new_code"
    echo "  Tag: $new_version (no 'v' prefix - F-Droid compatible)"
    echo "  Message: $release_message"
    echo "  German Changelog: $(echo "$CHANGELOG_DE" | head -n1)..."
    echo "  English Changelog: $(echo "$CHANGELOG_EN" | head -n1)..."
    echo "  Will build: Signed APK for Codeberg/F-Droid"
    echo "  Will build: Google Play AAB and upload to Google Play"
    echo "  Will create: Codeberg release with APK"
    echo
    
    # Final confirmation (skip in rerelease mode)
    if [[ "$RERELEASE_MODE" == "true" ]]; then
        print_success "🔄 Rerelease mode: Auto-proceeding with release"
    else
        read -p "Proceed with release? (Y/n): " -n 1 -r
        echo
        if [[ $REPLY =~ ^[Nn]$ ]]; then
            print_warning "Release cancelled"
            exit 0
        fi
    fi
    
    # Perform the release
    print_info "Starting unified release process..."
    
    # Only update build.gradle if version or code actually changed
    if [[ "$new_version" != "$current_version" ]] || [[ "$new_code" != "$current_code" ]]; then
        update_build_gradle "$new_version" "$new_code"
    else
        print_info "Version unchanged ($new_version, code $new_code) - skipping build.gradle update"
    fi
    
    # Create changelog files BEFORE committing
    create_changelog_files "$new_code" "$CHANGELOG_DE" "$CHANGELOG_EN"
    
    # Check if we have uncommitted changes (except build.gradle.kts and changelogs)
    if git diff --quiet HEAD -- . ':!app/build.gradle.kts' ':!fastlane/metadata/android/*/changelogs/*'; then
        print_info "No uncommitted changes detected (except version bump and changelogs)"
    else
        print_warning "You have uncommitted changes. Please commit them first."
        git status --porcelain | grep -v -E '(app/build.gradle.kts|fastlane/metadata/android/.*/changelogs/)' || true
        read -p "Continue anyway? (y/N): " -n 1 -r
        echo
        if [[ ! $REPLY =~ ^[Yy]$ ]]; then
            print_error "Release cancelled"
            # Restore backup
            mv app/build.gradle.kts.backup app/build.gradle.kts
            # Remove created changelog files
            rm -f "fastlane/metadata/android/de/changelogs/${new_code}.txt"
            rm -f "fastlane/metadata/android/en-US/changelogs/${new_code}.txt"
            exit 1
        fi
    fi
    
    # Commit version bump and changelogs (only add build.gradle if it changed)
    if [[ "$new_version" != "$current_version" ]] || [[ "$new_code" != "$current_code" ]]; then
        git add app/build.gradle.kts
    fi
    git add "fastlane/metadata/android/de/changelogs/${new_code}.txt" 2>/dev/null || true
    git add "fastlane/metadata/android/en-US/changelogs/${new_code}.txt" 2>/dev/null || true
    
    # Only commit if there are changes
    if ! git diff --cached --quiet; then
        git commit -m "Release version $new_version

$CHANGELOG_EN"
        print_success "Changes committed"
    else
        print_info "No new changes to commit (changelogs may already exist)"
    fi
    
    # Create tag WITHOUT 'v' prefix (F-Droid requirement)
    print_info "Creating git tag: $new_version (without 'v' prefix for F-Droid compatibility)"
    
    # Delete existing tag if it exists (for rerelease mode)
    if git tag -l | grep -q "^${new_version}$"; then
        print_warning "Tag $new_version already exists, deleting it first..."
        git tag -d "$new_version"
        git push origin --delete "$new_version" 2>/dev/null || print_info "Tag doesn't exist on remote"
    fi
    
    git tag -a "$new_version" -m "$release_message"
    
    print_success "Git commit and tag created BEFORE building"
    print_info "This ensures the APK is built from the exact tagged commit"
    
    # NOW build the APK and AAB from the tagged commit
    # Build signed APK for both Codeberg and F-Droid
    # Capture only the last line of output which contains the APK path
    local apk_path=$(build_signed_apk "$new_version" "$keystore_path" "$PASSWORD" | tail -1)
    if [[ $? -ne 0 ]]; then
        print_error "Failed to build APK"
        # Restore backup
        mv app/build.gradle.kts.backup app/build.gradle.kts
        # Remove tag and commit
        git tag -d "$new_version"
        git reset --hard HEAD~1
        exit 1
    fi
    
    # Build Google Play AAB
    # Capture only the last line of output which contains the AAB path
    local aab_path=$(build_google_play_aab "$new_version" "$keystore_path" "$PASSWORD" | tail -1)
    if [[ $? -ne 0 ]]; then
        print_warning "Failed to build Google Play AAB, but continuing with release"
    else
        print_success "Google Play AAB built successfully: $aab_path"
        
        # Upload to Google Play
        # upload_to_google_play "$aab_path" "$new_version" "$new_code"
    fi
    
    # Push commits and tags
    print_info "Pushing to repository..."
    git push origin main
    git push origin "$new_version"
    
    # Create Codeberg release with APK
    create_codeberg_release "$new_version" "$apk_path" "$CHANGELOG_EN" "$release_message"
    
    # Get the commit hash for the tag we just created
    local commit_hash=$(git rev-parse "$new_version")
    
#    # Update fdroiddata metadata
#    print_info "Updating F-Droid metadata..."
#    update_fdroiddata_metadata "$new_version" "$new_code" "$commit_hash"
    
    # Clean up backup
    rm -f app/build.gradle.kts.backup
    
    print_success "Release $new_version complete!"
    print_info "✅ Git tag created: $new_version (no 'v' prefix)"
    print_info "✅ APK built from tagged commit: $apk_path"
    print_info "✅ F-Droid will automatically build and publish within 24 hours"
    print_info "   Monitor progress at: https://monitor.f-droid.org/builds/build"
    
    if [[ -f "$apk_path" ]]; then
        print_info "📱 APK available for manual distribution: $apk_path"
    fi
    
    echo
    print_success "🎉 Unified release $new_version completed successfully!"
    print_info "📋 What happened:"
    echo "  ✅ Version updated in build.gradle.kts"
    echo "  ✅ Signed APK built for both Codeberg and F-Droid"
    if [[ -f "$aab_path" ]]; then
        echo "  ✅ Google Play AAB built"
    fi
    echo "  ✅ Git tag created without 'v' prefix"
    echo "  ✅ Codeberg release created"
    echo "  ✅ Changes pushed to repository"
    echo
    
    # Auto-bump version for next development cycle (skip in rerelease mode)
    if [[ "$RERELEASE_MODE" != "true" ]]; then
        print_info "🔄 Preparing for next development cycle..."
        local next_dev_version=$(increment_patch_version "$new_version")
        local next_dev_code=$((new_code + 1))
        
        print_info "Bumping version to $next_dev_version (code: $next_dev_code) for development"
        
        # Update build.gradle.kts for next version
        sed -i "s/versionName = \".*\"/versionName = \"$next_dev_version\"/" app/build.gradle.kts
        sed -i "s/versionCode = .*/versionCode = $next_dev_code/" app/build.gradle.kts
        
        # Commit the version bump
        git add app/build.gradle.kts
        git commit -m "Bump version to $next_dev_version for development" || {
            print_warning "Failed to create development version bump commit (may already exist)"
        }
        
        # Push the development version bump
        git push origin main || {
            print_warning "Failed to push development version bump"
        }
        
        print_success "✅ Version bumped to $next_dev_version for next development cycle"
    else
        print_info "🔄 Rerelease mode: Skipping automatic version bump for next development cycle"
    fi
    echo
    
    print_info "🔄 Next steps:"
    echo "  • F-Droid will detect the new tag automatically"
    echo "  • F-Droid will strip signatures from your APK and rebuild"
    echo "  • Users will receive the update within 24-48 hours"
    if [[ -f "$aab_path" ]]; then
        echo "  • Google Play will review the uploaded AAB"
        echo "  • After approval, the app will be available to 100% of users"
    fi
    echo "  • Repository is ready for next development cycle at $next_dev_version"
    echo
}

# Run main function
main "$@"
