package de.ntdote.medicalcalendarlog

import de.ntdote.medicalcalendarlog.data.*
import de.ntdote.medicalcalendarlog.ui.components.AdvancedGraphCanvas
import org.junit.Test
import org.junit.Assert.*
import java.util.*

class TemplateMatchingTest {

    @Test
    fun testTemplateMatchingForMultipleAlcoholTypes() {
        // Create beer and wine templates
        val beerTemplate = Template(
            id = "beer_template",
            name = "beer",
            templateType = TemplateType.DECAYING,
            metricUnit = "‰",
            extendedData = ExtendedTemplateData(
                factor = 1.0,
                unit = "‰",
                typeOfDrug = "alcohol",
                decayType = DecayType.HALF_LIFE,
                halfLifeHours = 1.0,
                hourlyDecayRate = null
            )
        )
        
        val wineTemplate = Template(
            id = "wine_template", 
            name = "wine",
            templateType = TemplateType.DECAYING,
            metricUnit = "‰",
            extendedData = ExtendedTemplateData(
                factor = 1.0,
                unit = "‰",
                typeOfDrug = "alcohol",
                decayType = DecayType.HALF_LIFE,
                halfLifeHours = 1.0,
                hourlyDecayRate = null
            )
        )
        
        // Create test events
        val beerEvent = CalendarEvent(
            id = 1L,
            title = "beer 0.5",
            startTime = Date(),
            endTime = Date(),
            calendarId = 1L,
            dosage = 0.5
        )
        
        val wineEvent = CalendarEvent(
            id = 2L,
            title = "wine 0.1", 
            startTime = Date(System.currentTimeMillis() + 3600000), // 1 hour later
            endTime = Date(System.currentTimeMillis() + 3600000),
            calendarId = 1L,
            dosage = 0.1
        )
        
        // Create GraphableItem.Drug with both templates
        val alcoholDrug = GraphableItem.Drug(
            id = "alcohol",
            name = "alcohol",
            color = androidx.compose.ui.graphics.Color.Red,
            drugType = "alcohol",
            templates = listOf(beerTemplate, wineTemplate),
            unit = "‰"
        )
        
        // Test that both events can find their matching templates
        val beerMatches = alcoholDrug.templates.any { template ->
            beerEvent.title.contains(template.name)
        }
        
        val wineMatches = alcoholDrug.templates.any { template ->
            wineEvent.title.contains(template.name)
        }
        
        assertTrue("Beer event should match beer template", beerMatches)
        assertTrue("Wine event should match wine template", wineMatches)
        
        // Test specific template finding for beer event
        val beerSpecificTemplate = alcoholDrug.templates.find { template ->
            beerEvent.title.contains(template.name)
        }
        
        assertNotNull("Should find specific template for beer event", beerSpecificTemplate)
        assertEquals("Should find beer template", "beer", beerSpecificTemplate?.name)
        
        // Test specific template finding for wine event
        val wineSpecificTemplate = alcoholDrug.templates.find { template ->
            wineEvent.title.contains(template.name)
        }
        
        assertNotNull("Should find specific template for wine event", wineSpecificTemplate)
        assertEquals("Should find wine template", "wine", wineSpecificTemplate?.name)
        
        // Test fallback logic for edge cases
        val alcoholEvent = CalendarEvent(
            id = 3L,
            title = "alcohol 0.2",
            startTime = Date(),
            endTime = Date(),
            calendarId = 1L,
            dosage = 0.2
        )
        
        // This should not match directly but could use fallback
        val alcoholDirectMatch = alcoholDrug.templates.find { template ->
            alcoholEvent.title.contains(template.name)
        }
        
        assertNull("Alcohol event should not directly match beer/wine templates", alcoholDirectMatch)
        
        // Test the fallback logic from the fixed code
        val eventTitle = alcoholEvent.title.lowercase()
        val fallbackTemplate = alcoholDrug.templates.find { template ->
            val templateName = template.name.lowercase()
            when {
                eventTitle.contains("beer") && templateName.contains("beer") -> true
                eventTitle.contains("wine") && templateName.contains("wine") -> true
                eventTitle.contains("alcohol") && (templateName.contains("beer") || templateName.contains("wine")) -> true
                else -> false
            }
        }
        
        assertNotNull("Should find fallback template for alcohol event", fallbackTemplate)
        assertTrue("Fallback should be beer or wine template", 
            fallbackTemplate?.name == "beer" || fallbackTemplate?.name == "wine")
    }
    
    @Test
    fun testTemplateMatchingEdgeCases() {
        val beerTemplate = Template(
            id = "beer_template",
            name = "beer",
            templateType = TemplateType.DECAYING,
            metricUnit = "‰",
            extendedData = ExtendedTemplateData(
                factor = 1.0,
                unit = "‰",
                typeOfDrug = "alcohol",
                decayType = DecayType.HALF_LIFE,
                halfLifeHours = 1.0,
                hourlyDecayRate = null
            )
        )
        
        val wineTemplate = Template(
            id = "wine_template",
            name = "wine", 
            templateType = TemplateType.DECAYING,
            metricUnit = "‰",
            extendedData = ExtendedTemplateData(
                factor = 1.0,
                unit = "‰",
                typeOfDrug = "alcohol",
                decayType = DecayType.HALF_LIFE,
                halfLifeHours = 1.0,
                hourlyDecayRate = null
            )
        )
        
        val alcoholDrug = GraphableItem.Drug(
            id = "alcohol",
            name = "alcohol", 
            color = androidx.compose.ui.graphics.Color.Red,
            drugType = "alcohol",
            templates = listOf(beerTemplate, wineTemplate),
            unit = "‰"
        )
        
        // Test various event title formats
        val testCases = listOf(
            "beer 0.5" to "beer",
            "wine 0.1" to "wine", 
            "Beer 0.5" to "beer", // Case insensitive
            "Wine 0.1" to "wine", // Case insensitive
            "beer0.5" to "beer",   // No space
            "wine0.1" to "wine",   // No space
            "some beer 0.5" to "beer", // Beer in middle
            "red wine 0.1" to "wine"   // Wine in middle
        )
        
        testCases.forEach { (eventTitle, expectedTemplate) ->
            val event = CalendarEvent(
                id = 4L,
                title = eventTitle,
                startTime = Date(),
                endTime = Date(),
                calendarId = 1L,
                dosage = 0.5
            )
            
            // Test direct matching
            val directMatch = alcoholDrug.templates.find { template ->
                event.title.contains(template.name)
            }
            
            // Test fallback matching
            val eventTitleLower = event.title.lowercase()
            val fallbackMatch = alcoholDrug.templates.find { template ->
                val templateName = template.name.lowercase()
                when {
                    eventTitleLower.contains("beer") && templateName.contains("beer") -> true
                    eventTitleLower.contains("wine") && templateName.contains("wine") -> true
                    eventTitleLower.contains("alcohol") && (templateName.contains("beer") || templateName.contains("wine")) -> true
                    else -> false
                }
            }
            
            val foundTemplate = directMatch ?: fallbackMatch
            
            assertNotNull("Should find template for event: $eventTitle", foundTemplate)
            assertEquals("Should find correct template for event: $eventTitle", 
                expectedTemplate, foundTemplate?.name)
        }
    }
}
