package de.ntdote.medicalcalendarlog.utils

import java.util.Date

/**
 * Represents a duration of time in milliseconds.
 * This is the internal representation used for all time calculations.
 * Use TimeFormatter to convert to user-friendly strings.
 */
data class TimeDuration(
    val milliseconds: Long
) : Comparable<TimeDuration> {
    
    val totalSeconds: Long get() = milliseconds / 1000
    val totalMinutes: Long get() = totalSeconds / 60
    val totalHours: Long get() = totalMinutes / 60
    val totalDays: Long get() = totalHours / 24
    
    val days: Int get() = (totalHours / 24).toInt()
    val hours: Int get() = (totalHours % 24).toInt()
    val minutes: Int get() = (totalMinutes % 60).toInt()
    val seconds: Int get() = (totalSeconds % 60).toInt()
    
    override fun compareTo(other: TimeDuration): Int {
        return milliseconds.compareTo(other.milliseconds)
    }
    
    operator fun plus(other: TimeDuration): TimeDuration {
        return TimeDuration(milliseconds + other.milliseconds)
    }
    
    operator fun minus(other: TimeDuration): TimeDuration {
        return TimeDuration(milliseconds - other.milliseconds)
    }
    
    companion object {
        /**
         * Calculate the time duration between two dates
         */
        fun between(startTime: Date, endTime: Date): TimeDuration {
            return TimeDuration(endTime.time - startTime.time)
        }
        
        /**
         * Calculate the time duration from a past date to now
         */
        fun since(startTime: Date): TimeDuration {
            return TimeDuration(System.currentTimeMillis() - startTime.time)
        }
        
        /**
         * Represents a "never" state - used when no event has occurred
         */
        val NEVER = TimeDuration(Long.MAX_VALUE)
        
        /**
         * Zero duration
         */
        val ZERO = TimeDuration(0)
        
        /**
         * Create from hours
         */
        fun fromHours(hours: Int): TimeDuration {
            return TimeDuration(hours * 60L * 60L * 1000L)
        }
        
        /**
         * Create from days
         */
        fun fromDays(days: Int): TimeDuration {
            return TimeDuration(days * 24L * 60L * 60L * 1000L)
        }
    }
    
    /**
     * Check if this duration represents a "never" state
     */
    fun isNever(): Boolean = this == NEVER
}
