package de.ntdote.medicalcalendarlog.utils

import android.util.Log
import de.ntdote.medicalcalendarlog.BuildConfig
import de.ntdote.medicalcalendarlog.data.Template

/**
 * Centralized data migration framework for handling app updates and config imports.
 * 
 * All migrations are versioned and applied sequentially. This ensures data integrity
 * when users upgrade the app or import configurations from older versions.
 * 
 * Usage:
 * - On app launch: Migrate stored templates if versionCode has changed
 * - On config import: Migrate imported templates from their source version
 */
object DataMigrations {
    private const val TAG = "MCL"
    
    /**
     * All registered migrations, sorted by version.
     * Add new migrations here as the app evolves.
     */
    private val migrations = listOf(
        Migration_v36(),
        Migration_v38(),
        Migration_v39()
        // Future migrations go here:
    )
    
    /**
     * Migrate templates from source version to target version.
     * 
     * @param templates The templates to migrate
     * @param sourceVersion The version code the templates came from (null = assume 34)
     * @param targetVersion The version code to migrate to (default = current BuildConfig)
     * @return Migrated templates
     */
    fun migrateTemplates(
        templates: List<Template>,
        sourceVersion: Int?,
        targetVersion: Int = BuildConfig.VERSION_CODE
    ): List<Template> {
        // No version info or version 0 = assume old version (34 - last version before migration system)
        // Version 0 means first app launch with migration system, data likely came from pre-migration era
        val fromVersion = when {
            sourceVersion == null -> 34
            sourceVersion == 0 -> 34 // First run, assume old data
            else -> sourceVersion
        }
        
        if (fromVersion >= targetVersion) {
            Log.d(TAG, "DataMigrations: No migration needed: source=$fromVersion, target=$targetVersion")
            return templates
        }
        
        Log.i(TAG, "DataMigrations: ╔════════════════════════════════════════════════════════════════")
        Log.i(TAG, "DataMigrations: ║ Starting data migration")
        Log.i(TAG, "DataMigrations: ║ Source version: $fromVersion (input: $sourceVersion)")
        Log.i(TAG, "DataMigrations: ║ Target version: $targetVersion")
        Log.i(TAG, "DataMigrations: ║ Templates to migrate: ${templates.size}")
        Log.i(TAG, "DataMigrations: ╚════════════════════════════════════════════════════════════════")
        
        // Find applicable migrations: all migrations with toVersion > fromVersion
        val applicableMigrations = migrations
            .filter { migration -> migration.toVersion > fromVersion }
            .sortedBy { it.toVersion }
        
        Log.d(TAG, "DataMigrations: Found ${applicableMigrations.size} applicable migrations from available ${migrations.size}")
        Log.d(TAG, "DataMigrations: Available migrations: ${migrations.map { "→${it.toVersion}" }}")
        Log.d(TAG, "DataMigrations: Applicable migrations: ${applicableMigrations.map { "→${it.toVersion}" }}")
        
        if (applicableMigrations.isEmpty()) {
            Log.w(TAG, "DataMigrations: No migrations found for version range $fromVersion → $targetVersion")
            return templates
        }
        
        Log.i(TAG, "DataMigrations: Applying ${applicableMigrations.size} migration(s):")
        applicableMigrations.forEach { migration ->
            Log.i(TAG, "DataMigrations:   • → ${migration.toVersion}: ${migration.description}")
        }
        
        // Apply migrations sequentially
        var migrated = templates
        applicableMigrations.forEach { migration ->
            Log.d(TAG, "DataMigrations: ─────────────────────────────────────────────────────────────────")
            Log.d(TAG, "DataMigrations: Applying migration → ${migration.toVersion}")
            
            val before = migrated
            migrated = migration.migrate(migrated)
            
            val changes = countChanges(before, migrated, migration)
            Log.i(TAG, "DataMigrations: ✓ Migration → ${migration.toVersion} complete: $changes")
        }
        
        Log.i(TAG, "DataMigrations: ═══════════════════════════════════════════════════════════════")
        Log.i(TAG, "DataMigrations: ✓ All migrations complete: $fromVersion → $targetVersion")
        Log.i(TAG, "DataMigrations: ═══════════════════════════════════════════════════════════════")
        
        return migrated
    }
    
    /**
     * Count and describe changes made by a migration
     */
    private fun countChanges(before: List<Template>, after: List<Template>, migration: Migration): String {
        if (before.size != after.size) {
            return "${before.size} templates → ${after.size} templates"
        }
        
        val changedCount = before.zip(after).count { (old, new) -> old != new }
        return "$changedCount of ${before.size} templates modified"
    }
}

/**
 * Base interface for all migrations
 */
interface Migration {
    val toVersion: Int
    val description: String
    
    /**
     * Perform the migration on the given templates
     * @param templates The templates to migrate
     * @return Migrated templates
     */
    fun migrate(templates: List<Template>): List<Template>
}

/**
 * Migration to v36: Multiple data format changes
 * 
 * Changes:
 * - reminderThresholdHours (Int, hours) → reminderThresholdMillis (Long, milliseconds)
 * - Template names with "###" placeholder → requiresInput = true (remove ### from name)
 * - Allows fractional hour precision (e.g., 0.035 hours = ~2 minutes)
 * - Maintains backwards compatibility
 */
class Migration_v36 : Migration {
    override val toVersion = 36
    override val description = "Convert reminder threshold to milliseconds + migrate ### placeholder to requiresInput flag"
    
    companion object {
        private const val TAG = "MCL"
        private const val HOURS_TO_MILLIS = 60 * 60 * 1000L
    }
    
    override fun migrate(templates: List<Template>): List<Template> {
        var remindersMigrated = 0
        var placeholdersMigrated = 0
        
        val result = templates.map { template ->
            var updatedTemplate = template
            
            // Migration 1: Convert reminderThresholdHours to reminderThresholdMillis
            val oldHours = template.reminderThresholdHours
            if (oldHours != null && oldHours > 0) {
                remindersMigrated++
                val millis = oldHours * HOURS_TO_MILLIS
                
                Log.d(TAG, "Migration_v36:   Migrating reminder '${template.name}': ${oldHours}h → ${millis}ms")
                
                updatedTemplate = updatedTemplate.copy(
                    reminderThresholdMillis = millis,
                    reminderThresholdHours = null // Remove old field
                )
            }
            
            // Migration 2: Convert ### placeholder to requiresInput flag
            if (updatedTemplate.name.contains("###")) {
                placeholdersMigrated++
                val cleanName = updatedTemplate.name.replace("###", "")
                
                Log.d(TAG, "Migration_v36:   Migrating placeholder '${updatedTemplate.name}' → '$cleanName' (requiresInput=true)")
                
                updatedTemplate = updatedTemplate.copy(
                    name = cleanName,
                    requiresInput = true
                )
            }
            
            updatedTemplate
        }
        
        Log.i(TAG, "Migration_v36:   Migrated $remindersMigrated reminder thresholds, $placeholdersMigrated ### placeholders")
        return result
    }
}

/**
 * Migration to v38: Reset privacy popup to show medical disclaimer
 * 
 * Changes:
 * - No template changes
 * - Privacy popup will be reset in PreferencesRepository.migrateTemplatesOnVersionUpgrade()
 * - This ensures users see the updated popup with medical disclaimer
 */
class Migration_v38 : Migration {
    override val toVersion = 38
    override val description = "Reset privacy popup to show updated medical disclaimer"
    
    companion object {
        private const val TAG = "MCL"
    }
    
    override fun migrate(templates: List<Template>): List<Template> {
        // No template changes needed for this migration
        // Privacy popup reset is handled in PreferencesRepository
        Log.i(TAG, "Migration_v38:   No template changes (privacy popup reset handled separately)")
        return templates
    }
}

/**
 * Migration to v39: Convert ExtendedTemplateData to drugs list
 * 
 * Changes:
 * - extendedData → drugs list (proper multi-drug support)
 * - multiDrugComponents → individual Drug objects with full properties
 * - concentrationReminderEnabled/Threshold → moved to individual drugs
 * - Maintains all decay parameters and reminder settings per drug
 */
class Migration_v39 : Migration {
    override val toVersion = 39
    override val description = "Convert ExtendedTemplateData to drugs list with proper multi-drug support"
    
    companion object {
        private const val TAG = "MCL"
    }
    
    override fun migrate(templates: List<Template>): List<Template> {
        var singleDrugMigrated = 0
        var multiDrugMigrated = 0
        var totalDrugsCreated = 0
        
        val result = templates.map { template ->
            // Only migrate DECAYING templates with extendedData
            if (template.templateType != de.ntdote.medicalcalendarlog.data.TemplateType.DECAYING || template.extendedData == null) {
                return@map template
            }
            
            val extData = template.extendedData
            val drugsList = mutableListOf<de.ntdote.medicalcalendarlog.data.Drug>()
            
            // Check if this is a source template (has decay parameters) or deriving template
            if (extData.decayType != null) {
                // MASTER TEMPLATE: Convert to drugs list
                
                if (extData.multiDrugComponents.isNullOrEmpty()) {
                    // Single drug template
                    singleDrugMigrated++
                    
                    val drug = de.ntdote.medicalcalendarlog.data.Drug.createSource(
                        drugType = extData.typeOfDrug,
                        factor = extData.factor ?: 1.0,
                        unit = extData.unit,
                        decayType = extData.decayType,
                        hourlyDecayRate = extData.hourlyDecayRate,
                        halfLifeHours = extData.halfLifeHours,
                        concentrationReminderEnabled = template.concentrationReminderEnabled,
                        concentrationReminderThreshold = template.concentrationReminderThreshold,
                        visible = template.visible  // Copy visibility to drug
                    )
                    
                    drugsList.add(drug)
                    totalDrugsCreated++
                    
                    Log.d(TAG, "Migration_v39:   Single-drug '${template.name}': ${extData.typeOfDrug} (${extData.factor} ${extData.unit})")
                } else {
                    // Multi-drug template
                    multiDrugMigrated++
                    
                    // Add main drug
                    val mainDrug = de.ntdote.medicalcalendarlog.data.Drug.createSource(
                        drugType = extData.typeOfDrug,
                        factor = extData.factor ?: 1.0,
                        unit = extData.unit,
                        decayType = extData.decayType,
                        hourlyDecayRate = extData.hourlyDecayRate,
                        halfLifeHours = extData.halfLifeHours,
                        concentrationReminderEnabled = template.concentrationReminderEnabled,
                        concentrationReminderThreshold = template.concentrationReminderThreshold,
                        visible = template.visible  // Copy visibility to drug
                    )
                    drugsList.add(mainDrug)
                    totalDrugsCreated++
                    
                    // Add component drugs (they deriv decay parameters from main drug)
                    extData.multiDrugComponents.forEach { component ->
                        val componentDrug = de.ntdote.medicalcalendarlog.data.Drug.createSource(
                            drugType = component.drugName,
                            factor = component.amountPerUnit,
                            unit = component.unit,
                            decayType = extData.decayType,
                            hourlyDecayRate = extData.hourlyDecayRate,
                            halfLifeHours = extData.halfLifeHours,
                            concentrationReminderEnabled = false, // Components don't have individual reminders by default
                            concentrationReminderThreshold = null,
                            visible = template.visible  // Copy visibility to drug
                        )
                        drugsList.add(componentDrug)
                        totalDrugsCreated++
                    }
                    
                    Log.d(TAG, "Migration_v39:   Multi-drug '${template.name}': ${drugsList.size} drugs (${extData.typeOfDrug} + ${extData.multiDrugComponents.size} components)")
                }
            } else {
                // INHERITING TEMPLATE: Create derived drug (no decay parameters)
                singleDrugMigrated++
                
                // For deriving templates, create a derived drug that will reference source templates
                val drug = de.ntdote.medicalcalendarlog.data.Drug.createDerived(
                    drugType = extData.typeOfDrug,
                    factor = extData.factor ?: 1.0,
                    unit = extData.unit.ifEmpty { "mg" },
                    concentrationReminderEnabled = false,
                    concentrationReminderThreshold = null,
                    visible = template.visible  // Copy visibility to drug
                )
                
                drugsList.add(drug)
                totalDrugsCreated++
                
                Log.d(TAG, "Migration_v39:   Inheriting '${template.name}': ${extData.typeOfDrug} (factor: ${extData.factor}) -> DERIVED drug")
            }
            
            // Create new template with drugs list, clear deprecated fields
            template.copy(
                drugs = drugsList,
                extendedData = null, // Clear deprecated field
                concentrationReminderEnabled = false, // Moved to individual drugs
                concentrationReminderThreshold = null // Moved to individual drugs
            )
        }
        
        Log.i(TAG, "Migration_v39:   Migrated $singleDrugMigrated single-drug + $multiDrugMigrated multi-drug templates → $totalDrugsCreated total drugs")
        return result
    }
}
