package de.ntdote.medicalcalendarlog.utils

import de.ntdote.medicalcalendarlog.data.*
import java.util.*
import kotlin.math.*

object AdvancedCalculationEngine {
    
    /**
     * Generate graph data for all types of graphable items
     */
    fun generateGraphData(
        items: List<GraphableItem>,
        events: List<CalendarEvent>,
        startTime: Date,
        endTime: Date,
        preCalculationDays: Int = 90
    ): List<GraphData> {
        val extendedStartTime = Date(startTime.time - preCalculationDays * 24 * 60 * 60 * 1000L)
        
        return items.mapNotNull { item ->
            when (item) {
                is GraphableItem.GeneralTemplate -> generateGeneralTemplateData(
                    item, events, startTime, endTime
                )
                is GraphableItem.MetricTemplate -> generateMetricTemplateData(
                    item, events, startTime, endTime
                )
                is GraphableItem.Drug -> generateDrugData(
                    item, events, extendedStartTime, endTime
                )
                is GraphableItem.DecayingTemplate -> generateDecayingTemplateData(
                    item, events, startTime, endTime
                )
            }
        }
    }
    
    /**
     * Generate data for decaying templates (timeline events at template level)
     */
    private fun generateDecayingTemplateData(
        item: GraphableItem.DecayingTemplate,
        events: List<CalendarEvent>,
        startTime: Date,
        endTime: Date
    ): GraphData? {
        val relevantEvents = events.filter { event ->
            matchesTemplate(event, item.template) &&
            event.startTime.after(startTime) && event.startTime.before(endTime)
        }
        
        if (relevantEvents.isEmpty()) return null
        
        // For decaying templates, create points at timeline level (value = 1.0)
        val points = relevantEvents.map { event ->
            GraphPoint(
                time = event.startTime,
                value = 1.0,
                event = event
            )
        }
        
        return GraphData(
            item = item,
            points = points,
            minValue = 0.0,
            maxValue = 1.0,
            unit = "events"
        )
    }
    
    /**
     * Generate data for general templates (timeline events)
     */
    private fun generateGeneralTemplateData(
        item: GraphableItem.GeneralTemplate,
        events: List<CalendarEvent>,
        startTime: Date,
        endTime: Date
    ): GraphData? {
        val relevantEvents = events.filter { event ->
            matchesTemplate(event, item.template) &&
            event.startTime.after(startTime) && event.startTime.before(endTime)
        }
        
        if (relevantEvents.isEmpty()) return null
        
        // For general templates, create points at timeline level (value = 1.0)
        val points = relevantEvents.map { event ->
            GraphPoint(
                time = event.startTime,
                value = 1.0,
                event = event
            )
        }
        
        return GraphData(
            item = item,
            points = points,
            minValue = 0.0,
            maxValue = 1.0,
            unit = "events"
        )
    }
    
    /**
     * Generate data for metric templates (value lines)
     */
    private fun generateMetricTemplateData(
        item: GraphableItem.MetricTemplate,
        events: List<CalendarEvent>,
        startTime: Date,
        endTime: Date
    ): GraphData? {
        val relevantEvents = events.filter { event ->
            matchesTemplate(event, item.template) &&
            event.startTime.after(startTime) && event.startTime.before(endTime)
        }.sortedBy { it.startTime }
        
        if (relevantEvents.isEmpty()) return null
        
        // Extract metric values from events
        val dataPoints = relevantEvents.mapNotNull { event ->
            val value = extractMetricValue(event.title, item.template.name)
            if (value != null) {
                GraphPoint(
                    time = event.startTime,
                    value = value,
                    event = event
                )
            } else null
        }
        
        if (dataPoints.isEmpty()) return null
        
        // For metrics, use simple point-to-point lines (no interpolation)
        // The Canvas will automatically draw straight lines between these points
        val maxValue = dataPoints.maxOf { it.value } * 1.05
        
        return GraphData(
            item = item,
            points = dataPoints,
            minValue = 0.0,
            maxValue = maxValue,
            unit = item.unit
        )
    }
    
    /**
     * Generate data for drugs (aggregated concentration from multiple templates)
     */
    private fun generateDrugData(
        item: GraphableItem.Drug,
        events: List<CalendarEvent>,
        extendedStartTime: Date,
        endTime: Date
    ): GraphData? {
        // Get all events for all templates of this drug
        val allRelevantEvents = events.filter { event ->
            item.templates.any { template -> matchesTemplate(event, template) }
        }.sortedBy { it.startTime }
        
        if (allRelevantEvents.isEmpty()) return null
        
        // Generate concentration history using drug aggregation
        val concentrationHistory = generateDrugConcentrationHistory(
            drugType = item.drugType,
            templates = item.templates,
            events = allRelevantEvents,
            startTime = extendedStartTime,
            endTime = endTime
        )
        
        if (concentrationHistory.isEmpty()) return null
        
        // Convert to graph points and add event markers
        val points = mutableListOf<GraphPoint>()
        
        // Add concentration curve points
        concentrationHistory.forEach { (time, concentration) ->
            points.add(GraphPoint(time = time, value = concentration))
        }
        
        // Add event marker points at their peak concentrations
        allRelevantEvents.filter { 
            it.startTime.after(extendedStartTime) && it.startTime.before(endTime) 
        }.forEach { event ->
            // Use the drug-specific calculation for peak concentration
            val peakConcentration = calculatePeakAfterEventForDrug(
                drugType = item.drugType,
                templates = item.templates,
                events = allRelevantEvents,
                targetEvent = event
            )
            points.add(GraphPoint(
                time = event.startTime,
                value = peakConcentration,
                event = event
            ))
        }
        
        val maxValue = concentrationHistory.maxOfOrNull { it.second }?.let { it * 1.05 } ?: 1.0
        
        return GraphData(
            item = item,
            points = points,
            minValue = 0.0,
            maxValue = maxValue,
            unit = item.unit
        )
    }
    
    /**
     * Generate drug concentration history by aggregating all templates of the same drug type
     */
    private fun generateDrugConcentrationHistory(
        drugType: String,
        templates: List<Template>,
        events: List<CalendarEvent>,
        startTime: Date,
        endTime: Date
    ): List<Pair<Date, Double>> {
        val history = mutableListOf<Pair<Date, Double>>()
        
        // Get the drug definition from source template using DrugResolution (like main screen)
        val referenceDrug = DrugResolution.findSourceDrug(drugType, templates) ?: return emptyList()
        
        // Collect event times for DECAYING templates to create vertical jumps
        val eventTimes = mutableSetOf<Long>()
        events.filter { event ->
            templates.any { template -> 
                template.templateType == TemplateType.DECAYING && 
                template.drugs.any { it.drugType == drugType } &&
                matchesTemplate(event, template)
            }
        }.forEach { event ->
            // Add points 1 minute before and at event end time for vertical jumps
            eventTimes.add(event.startTime.time - 60_000L) // Before (using startTime as endTime)
            eventTimes.add(event.startTime.time)           // After
        }
        
        // Generate regular time points for smooth curve
        val timeStepMillis = (endTime.time - startTime.time) / 500L
        var currentTime = startTime.time
        
        val regularTimePoints = mutableSetOf<Long>()
        while (currentTime <= endTime.time) {
            regularTimePoints.add(currentTime)
            currentTime += timeStepMillis
        }
        
        // Combine regular time steps with event-specific points and sort
        val allTimePoints = (regularTimePoints + eventTimes).filter { 
            it >= startTime.time && it <= endTime.time 
        }.sorted()
        
        // Calculate concentration at each time point
        allTimePoints.forEach { timeMillis ->
            val pointTime = Date(timeMillis)
            val concentration = calculateDrugConcentrationAtTime(
                drugType, templates, events, pointTime
            )
            history.add(Pair(pointTime, concentration))
        }
        
        return history
    }
    
    /**
     * Calculate drug concentration at a specific time by aggregating all contributing events
     */
    private fun calculateDrugConcentrationAtTime(
        drugType: String,
        templates: List<Template>,
        events: List<CalendarEvent>,
        targetTime: Date
    ): Double {
        // Get all events for this drug type that occurred at or before the target time
        val relevantEvents = events.filter { event ->
            event.startTime.time <= targetTime.time &&
            templates.any { template -> 
                template.drugs.any { it.drugType == drugType } && matchesTemplate(event, template)
            }
        }.sortedBy { it.startTime.time }
        
        if (relevantEvents.isEmpty()) return 0.0
        
        // Get the drug definition from source template using DrugResolution (like main screen)
        val referenceDrug = DrugResolution.findSourceDrug(drugType, templates) ?: return 0.0
        
        return when (referenceDrug.decayType) {
            DecayType.HALF_LIFE -> {
                val halfLifeHours = referenceDrug.halfLifeHours
                if (halfLifeHours != null && halfLifeHours > 0) {
                    // Exponential decay - each dose decays independently
                    var totalConcentration = 0.0
                    relevantEvents.forEach { event ->
                        val matchingTemplate = templates.find { template ->
                            template.drugs.any { it.drugType == drugType } && matchesTemplate(event, template)
                        }
                        
                        if (matchingTemplate != null) {
                            val drug = matchingTemplate.drugs.find { it.drugType == drugType }
                            if (drug != null) {
                                val timeDiffHours = (targetTime.time - event.startTime.time) / (1000.0 * 60.0 * 60.0)
                                val dosage = event.dosage
                                val factor = drug.factor
                                val decayConstant = ln(2.0) / halfLifeHours
                                totalConcentration += dosage * factor * exp(-decayConstant * timeDiffHours)
                            }
                        }
                    }
                    totalConcentration
                } else 0.0
            }
            DecayType.CONSTANT -> {
                val hourlyDecayRate = referenceDrug.hourlyDecayRate
                if (hourlyDecayRate != null && hourlyDecayRate > 0) {
                    // Linear decay - total concentration pool eliminates at fixed rate
                    calculateLinearDecayConcentration(relevantEvents, templates, drugType, targetTime, hourlyDecayRate)
                } else 0.0
            }
            null -> 0.0 // No decay type means no concentration tracking
            else -> 0.0 // Handle any other cases
        }
    }
    
    /**
     * Calculate concentration using proper linear elimination kinetics
     * The total concentration pool eliminates at a fixed hourly rate
     */
    private fun calculateLinearDecayConcentration(
        events: List<CalendarEvent>,
        templates: List<Template>,
        drugType: String,
        targetTime: Date,
        hourlyDecayRate: Double
    ): Double {
        // Build a timeline of all dose additions up to target time
        val doseEvents = events.mapNotNull { event ->
            val matchingTemplate = templates.find { template ->
                template.drugs.any { it.drugType == drugType } && matchesTemplate(event, template)
            }
            
            if (matchingTemplate != null) {
                val drug = matchingTemplate.drugs.find { it.drugType == drugType }
                if (drug != null) {
                    val dosage = event.dosage
                    val factor = drug.factor
                    Pair(event.startTime.time, dosage * factor)
                } else null
            } else null
        }.sortedBy { it.first }
        
        if (doseEvents.isEmpty()) return 0.0
        
        // Simulate concentration over time using proper linear elimination
        var totalConcentration = 0.0
        var lastTime = doseEvents.first().first
        
        // Process each dose event
        doseEvents.forEach { (eventTime, doseAmount) ->
            // Apply decay from last time to this event time
            val timeDiffHours = (eventTime - lastTime) / (1000.0 * 60.0 * 60.0)
            totalConcentration = maxOf(0.0, totalConcentration - (hourlyDecayRate * timeDiffHours))
            
            // Add the new dose
            totalConcentration += doseAmount
            lastTime = eventTime
        }
        
        // Apply final decay from last dose to target time
        val finalTimeDiffHours = (targetTime.time - lastTime) / (1000.0 * 60.0 * 60.0)
        totalConcentration = maxOf(0.0, totalConcentration - (hourlyDecayRate * finalTimeDiffHours))
        
        return totalConcentration
    }
    
    /**
     * Calculate peak concentration immediately after a specific event
     * @deprecated Use calculatePeakAfterEventForDrug instead
     */
    private fun calculatePeakAfterEvent(
        templates: List<Template>,
        events: List<CalendarEvent>,
        targetEvent: CalendarEvent
    ): Double {
        // Find the template for this event
        val template = templates.find { matchesTemplate(targetEvent, it) } ?: return 0.0
        
        // Find the first drug in this template (NEW data model)
        val drugType = template.drugs.firstOrNull()?.drugType ?: return 0.0
        
        // Calculate concentration at the event time (peak for this event)
        return calculateDrugConcentrationAtTime(drugType, templates, events, targetEvent.startTime)
    }
    
    /**
     * Calculate peak concentration for a specific drug immediately after a specific event
     */
    private fun calculatePeakAfterEventForDrug(
        drugType: String,
        templates: List<Template>,
        events: List<CalendarEvent>,
        targetEvent: CalendarEvent
    ): Double {
        // Calculate concentration at the event time (peak for this event)
        // Specifically for the requested drug type
        return calculateDrugConcentrationAtTime(drugType, templates, events, targetEvent.startTime)
    }
    
    /**
     * Check if an event matches a template
     */
    private fun matchesTemplate(event: CalendarEvent, template: Template): Boolean {
        // Use unified matching logic
        return EventMatching.matchesTemplate(event, template)
    }
    
    /**
     * Extract metric value from event title
     */
    private fun extractMetricValue(eventTitle: String, templateName: String): Double? {
        val pattern = "${templateName}([0-9]+\\.?[0-9]*)".toRegex()
        return pattern.find(eventTitle)?.groupValues?.get(1)?.toDoubleOrNull()
    }
    
    /**
     * Clean event text by removing numbers for display
     */
    fun cleanEventText(eventTitle: String): String {
        return eventTitle.replace(Regex("[0-9]+\\.?[0-9]*"), "").trim()
    }
    
    /**
     * Create graphable items from templates
     * 
     * Processing order:
     * 1. Process templates in config order
     * 2. For DECAYING templates: check for name collision (template.name == drug.drugType)
     *    - If collision: create Drug items only
     *    - If no collision: create DecayingTemplate item, then Drug items
     * 3. For METRIC and GENERAL: create template items as before
     */
    fun createGraphableItems(templates: List<Template>, colors: List<androidx.compose.ui.graphics.Color>): List<GraphableItem> {
        val items = mutableListOf<GraphableItem>()
        var colorIndex = 0
        
        // Filter to only enabled templates for template-level items
        val enabledTemplates = templates.filter { it.enabled }
        
        // Track which drugs have already been added (to avoid duplicates)
        val addedDrugs = mutableSetOf<String>()
        
        // Process templates in order
        enabledTemplates.forEach { template ->
            when (template.templateType) {
                TemplateType.DECAYING -> {
                    // Check for name collision: does template.name match ANY of its drug names?
                    val hasCollision = template.drugs.any { it.drugType == template.name }
                    
                    if (!hasCollision) {
                        // No collision: Add DecayingTemplate item first
                        items.add(
                            GraphableItem.DecayingTemplate(
                                id = "decaying_template_${template.id}",
                                name = template.name,
                                color = colors[colorIndex % colors.size],
                                isSelected = template.visible,
                                template = template
                            )
                        )
                        colorIndex++
                    }
                    
                    // Add Drug items for each drug in this template (regardless of collision)
                    template.drugs.forEach { drug ->
                        // Only add if not already added from another template
                        if (!addedDrugs.contains(drug.drugType)) {
                            // Collect ALL templates that contain this drug (including disabled ones for cross-template calculations)
                            val allTemplatesForDrug = templates.filter { t ->
                                t.templateType == TemplateType.DECAYING && 
                                t.drugs.any { d -> d.drugType == drug.drugType }
                            }
                            
                            items.add(
                                GraphableItem.Drug(
                                    id = "drug_${drug.drugType}",
                                    name = drug.drugType,
                                    color = colors[colorIndex % colors.size],
                                    isSelected = drug.visible,
                                    drugType = drug.drugType,
                                    templates = allTemplatesForDrug,
                                    unit = drug.unit
                                )
                            )
                            colorIndex++
                            addedDrugs.add(drug.drugType)
                        }
                    }
                }
                
                TemplateType.METRIC -> {
                    val unit = template.metricUnit.ifBlank { "units" }
                    
                    items.add(
                        GraphableItem.MetricTemplate(
                            id = "metric_${template.id}",
                            name = template.name,
                            color = colors[colorIndex % colors.size],
                            isSelected = template.visible,
                            template = template,
                            unit = unit
                        )
                    )
                    colorIndex++
                }
                
                TemplateType.GENERAL -> {
                    items.add(
                        GraphableItem.GeneralTemplate(
                            id = "general_${template.id}",
                            name = template.name,
                            color = colors[colorIndex % colors.size],
                            isSelected = template.visible,
                            template = template
                        )
                    )
                    colorIndex++
                }
            }
        }
        
        return items
    }
}
