package de.ntdote.medicalcalendarlog.ui.screens

import androidx.compose.foundation.layout.*
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Add
import androidx.compose.material.icons.filled.BarChart
import androidx.compose.material.icons.filled.Refresh
import androidx.compose.material.icons.filled.Settings
import androidx.compose.material3.*
import androidx.compose.material.icons.automirrored.filled.Help
import androidx.compose.runtime.*
import kotlinx.coroutines.delay
import android.util.Log
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.focus.FocusRequester
import androidx.compose.ui.focus.focusRequester
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.platform.LocalFocusManager
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.unit.dp
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.compose.LocalLifecycleOwner
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import androidx.lifecycle.repeatOnLifecycle
import androidx.lifecycle.viewmodel.compose.viewModel
import de.ntdote.medicalcalendarlog.R
import de.ntdote.medicalcalendarlog.data.Template
import de.ntdote.medicalcalendarlog.data.TemplateType
import de.ntdote.medicalcalendarlog.repository.CalendarRepository
import de.ntdote.medicalcalendarlog.repository.PreferencesRepository
import de.ntdote.medicalcalendarlog.ui.components.MarkdownDialog
import de.ntdote.medicalcalendarlog.ui.components.TemplateButton
import de.ntdote.medicalcalendarlog.viewmodel.MainViewModel
import kotlinx.coroutines.launch

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun MainScreen(
    onNavigateToConfig: () -> Unit,
    onNavigateToGraph: () -> Unit,
    onNavigateToHelp: () -> Unit = {},
    viewModel: MainViewModel? = null
) {
    val context = LocalContext.current
    val actualViewModel = viewModel ?: viewModel {
        MainViewModel(
            application = context.applicationContext as android.app.Application,
            calendarRepository = CalendarRepository(context.contentResolver, context),
            preferencesRepository = PreferencesRepository(context)
        )
    }
    
    // State that changes every minute to force recomposition of the entire screen
    var refreshTrigger by remember { mutableStateOf(0) }
    
    val uiState by actualViewModel.uiState.collectAsStateWithLifecycle()
    val concentrationData by actualViewModel.concentrationData.collectAsStateWithLifecycle()
    val refreshSignal by actualViewModel.refreshSignal.collectAsStateWithLifecycle()
    
    // Get the lifecycle owner to monitor app state (foreground/background)
    val lifecycleOwner = LocalLifecycleOwner.current
    
    // Auto-refresh data every X seconds ONLY when app is in foreground
    LaunchedEffect(lifecycleOwner) {
        lifecycleOwner.lifecycle.repeatOnLifecycle(Lifecycle.State.RESUMED) {
            // This block only runs when the app is in the foreground (RESUMED state)
            Log.d("MCL", "MainScreen: App entered foreground, starting refresh loop")
            
            // Do an immediate refresh when entering foreground
            actualViewModel.refreshData()
            refreshTrigger += 1
            
            while (true) {
                delay(30000) 
                Log.d("MCL", "MainScreen: Triggering refresh at ${java.util.Date()}")
                actualViewModel.refreshData()
                // Increment the trigger to force recomposition
                refreshTrigger += 1
                Log.d("MCL", "MainScreen: Incremented refreshTrigger to $refreshTrigger")
            }
            
            // When app goes to background, this coroutine is cancelled and the log below is never reached
            // The loop will restart from the beginning when app returns to foreground
        }
        // This line is reached when leaving RESUMED state (going to background)
        Log.d("MCL", "MainScreen: App left foreground, refresh loop stopped")
    }
    
    // Log when the screen is recomposed due to refreshTrigger changes
    LaunchedEffect(refreshTrigger) {
        if (refreshTrigger > 0) {
            Log.d("MCL", "MainScreen: Recomposing due to refreshTrigger change to $refreshTrigger")
        }
    }
    
    var showDosageDialog by remember { mutableStateOf(false) }
    var selectedTemplate by remember { mutableStateOf<Template?>(null) }
    var dosageInput by remember { mutableStateOf("") }
    val focusRequester = remember { FocusRequester() }
    
    // Privacy popup state
    val preferencesRepository = remember { PreferencesRepository(context) }
    val shouldShowPrivacyPopup by preferencesRepository.shouldShowPrivacyPopup.collectAsStateWithLifecycle(initialValue = false)
    var showPrivacyPopup by remember { mutableStateOf(false) }
    var dontShowAgainChecked by remember { mutableStateOf(false) }
    val coroutineScope = rememberCoroutineScope()
    
    // Load popup content
    val popupContent = remember {
        context.resources.openRawResource(R.raw.popup).bufferedReader().use { it.readText() }
    }
    
    // Show privacy popup if should show and user has permissions
    LaunchedEffect(shouldShowPrivacyPopup, uiState.hasCalendarPermission) {
        if (shouldShowPrivacyPopup && uiState.hasCalendarPermission && !uiState.isLoading) {
            showPrivacyPopup = true
        }
    }

    Scaffold(
        topBar = {
            TopAppBar(
                title = { Text("Medical Calendar Log") },
                actions = {
                    IconButton(onClick = onNavigateToHelp) {
                        Icon(Icons.AutoMirrored.Filled.Help, contentDescription = "Help")
                    }
                    IconButton(onClick = onNavigateToGraph) {
                        Icon(Icons.Default.BarChart, contentDescription = "Graph")
                    }
                    IconButton(onClick = onNavigateToConfig) {
                        Icon(Icons.Default.Settings, contentDescription = "Settings")
                    }
                }
            )
        }
    ) { paddingValues ->
        Column(
            modifier = Modifier
                .fillMaxSize()
                .padding(paddingValues)
                .padding(16.dp)
        ) {
            if (uiState.isLoading) {
                Box(
                    modifier = Modifier.fillMaxSize(),
                    contentAlignment = Alignment.Center
                ) {
                    CircularProgressIndicator()
                }
            } else if (!uiState.hasCalendarPermission) {
                Card(
                    modifier = Modifier.fillMaxWidth(),
                    colors = CardDefaults.cardColors(containerColor = MaterialTheme.colorScheme.errorContainer)
                ) {
                    Column(
                        modifier = Modifier.padding(16.dp)
                    ) {
                        Text(
                            text = "Calendar Permission Required",
                            style = MaterialTheme.typography.titleMedium,
                            fontWeight = FontWeight.Bold
                        )
                        Spacer(modifier = Modifier.height(8.dp))
                        Text(
                            text = "This app needs calendar permissions to read and create events. Please grant permissions in your device settings.",
                            style = MaterialTheme.typography.bodyMedium
                        )
                        Spacer(modifier = Modifier.height(8.dp))
                        Button(
                            onClick = { actualViewModel.loadAvailableCalendars() }
                        ) {
                            Text("Retry")
                        }
                    }
                }
            } else if (uiState.templates.isEmpty()) {
                Card(
                    modifier = Modifier.fillMaxWidth()
                ) {
                    Column(
                        modifier = Modifier.padding(16.dp),
                        horizontalAlignment = Alignment.CenterHorizontally
                    ) {
                        Icon(
                            Icons.Default.Add,
                            contentDescription = null,
                            modifier = Modifier.size(48.dp),
                            tint = MaterialTheme.colorScheme.primary
                        )
                        Spacer(modifier = Modifier.height(16.dp))
                        Text(
                            text = "No Templates Configured",
                            style = MaterialTheme.typography.titleMedium,
                            fontWeight = FontWeight.Bold
                        )
                        Spacer(modifier = Modifier.height(8.dp))
                        Text(
                            text = "Add templates in the configuration screen to start logging events.",
                            style = MaterialTheme.typography.bodyMedium
                        )
                        Spacer(modifier = Modifier.height(16.dp))
                        Button(onClick = onNavigateToConfig) {
                            Text("Configure Templates")
                        }
                    }
                }
            } else {
                // Display last refresh timestamp only in debug mode - will update when refreshTrigger or refreshSignal changes
                val currentTime = remember(refreshTrigger, refreshSignal) { java.util.Date() }
                if (uiState.debugModeEnabled) {
                    val formatter = remember { java.text.SimpleDateFormat("HH:mm", java.util.Locale.getDefault()) }
                    val formattedTime = remember(currentTime) { formatter.format(currentTime) }
                    
                    // Format last worker run with type
                    val lastWorkerRun = if (uiState.lastWorkerRunTimestamp > 0) {
                        val time = formatter.format(java.util.Date(uiState.lastWorkerRunTimestamp))
                        val type = if (uiState.lastWorkerRunType.isNotEmpty()) " (${uiState.lastWorkerRunType})" else ""
                        "$time$type"
                    } else {
                        "Never"
                    }
                    
                    // Format second last worker run with type
                    val secondLastWorkerRun = if (uiState.secondLastWorkerRunTimestamp > 0) {
                        val time = formatter.format(java.util.Date(uiState.secondLastWorkerRunTimestamp))
                        val type = if (uiState.secondLastWorkerRunType.isNotEmpty()) " (${uiState.secondLastWorkerRunType})" else ""
                        "$time$type"
                    } else {
                        "N/A"
                    }
                    
                    // Format next reminder timestamp with template name
                    val nextReminder = if (uiState.nextReminderTimestamp > 0) {
                        val time = formatter.format(java.util.Date(uiState.nextReminderTimestamp))
                        if (uiState.nextReminderTemplateName.isNotEmpty()) {
                            "$time (${uiState.nextReminderTemplateName})"
                        } else {
                            time
                        }
                    } else {
                        "None scheduled"
                    }
                    
                    Card(
                        modifier = Modifier
                            .fillMaxWidth()
                            .padding(bottom = 8.dp),
                        colors = CardDefaults.cardColors(containerColor = MaterialTheme.colorScheme.surfaceVariant)
                    ) {
                        Row(
                            modifier = Modifier
                                .fillMaxWidth()
                                .padding(12.dp),
                            horizontalArrangement = Arrangement.SpaceBetween,
                            verticalAlignment = Alignment.CenterVertically
                        ) {
                            // Left side: Debug info (65% width)
                            Column(
                                modifier = Modifier.weight(0.65f)
                            ) {
                                Text(
                                    text = "Debug Info",
                                    style = MaterialTheme.typography.titleSmall,
                                    fontWeight = FontWeight.Bold
                                )
                                Spacer(modifier = Modifier.height(4.dp))
                                Text(
                                    text = "UI Refresh: $formattedTime (#$refreshTrigger)",
                                    style = MaterialTheme.typography.bodySmall
                                )
                                Text(
                                    text = "Last Worker: $lastWorkerRun",
                                    style = MaterialTheme.typography.bodySmall
                                )
                                Text(
                                    text = "2nd Last Worker: $secondLastWorkerRun",
                                    style = MaterialTheme.typography.bodySmall
                                )
                                Text(
                                    text = "Next Reminder: $nextReminder",
                                    style = MaterialTheme.typography.bodySmall
                                )
                            }
                            
                            // Right side: Manual trigger button (35% width)
                            Column(
                                modifier = Modifier.weight(0.35f),
                                horizontalAlignment = Alignment.CenterHorizontally
                            ) {
                                IconButton(
                                    onClick = { actualViewModel.triggerManualWorkerRun() },
                                    modifier = Modifier.size(48.dp)
                                ) {
                                    Icon(
                                        Icons.Default.Refresh,
                                        contentDescription = "Run Worker",
                                        tint = MaterialTheme.colorScheme.primary,
                                        modifier = Modifier.size(32.dp)
                                    )
                                }
                                Text(
                                    text = "Run Worker",
                                    style = MaterialTheme.typography.labelSmall,
                                    color = MaterialTheme.colorScheme.onSurfaceVariant
                                )
                            }
                        }
                    }
                }
                
                LazyColumn(
                    verticalArrangement = Arrangement.spacedBy(12.dp)
                ) {
                    items(uiState.templates.filter { it.enabled }) { template ->
                        val concentration = concentrationData.find { it.templateId == template.id }
                        
                        // Recalculate timeSinceLastEvent whenever refreshTrigger or refreshSignal changes to force recomposition
                        val timeSinceLastEvent = remember(refreshTrigger, refreshSignal) {
                            actualViewModel.getTimeSinceLastEvent(template.id)
                        }
                        
                        TemplateButton(
                            template = template,
                            timeSinceLastEvent = timeSinceLastEvent,
                            reminderThresholdMillis = template.getEffectiveReminderThresholdMillis(),
                            currentConcentration = concentration?.getTotalConcentration(),
                            peakConcentration = concentration?.getTotalPeakConcentration(),
                            latestMetricValue = if (template.templateType == TemplateType.METRIC) actualViewModel.getLatestMetricValue(template.id) else null,
                            concentrationsPerDrug = concentration?.concentrations ?: emptyMap(),
                            peakConcentrationsPerDrug = concentration?.peakConcentrations ?: emptyMap(),
                            onClick = {
                                if (template.templateType == TemplateType.METRIC || template.requiresInput) {
                                    selectedTemplate = template
                                    dosageInput = ""
                                    showDosageDialog = true
                                } else {
                                    actualViewModel.createEventFromTemplate(template)
                                }
                            }
                        )
                    }
                }
            }

            // Error handling
            uiState.error?.let { error ->
                LaunchedEffect(error) {
                    // Show snackbar or handle error
                }
            }
        }
    }

    // Dosage input dialog
    if (showDosageDialog && selectedTemplate != null) {
        AlertDialog(
            onDismissRequest = { 
                showDosageDialog = false
                selectedTemplate = null
            },
            title = { Text(if (selectedTemplate!!.templateType == TemplateType.METRIC) "Enter Value" else "Enter Dosage") },
            text = {
                Column {
                    Text(
                        if (selectedTemplate!!.templateType == TemplateType.METRIC) 
                            "Enter the value for ${selectedTemplate!!.name}:" 
                        else 
                            "Enter the dosage for ${selectedTemplate!!.name}:"
                    )
                    Spacer(modifier = Modifier.height(8.dp))
                    OutlinedTextField(
                        value = dosageInput,
                        onValueChange = { dosageInput = it },
                        label = { Text(if (selectedTemplate!!.templateType == TemplateType.METRIC) "Value" else "Dosage") },
                        keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Decimal),
                        singleLine = true,
                        modifier = Modifier.focusRequester(focusRequester)
                    )
                }
            },
            confirmButton = {
                TextButton(
                    onClick = {
                        val dosage = dosageInput.toDoubleOrNull()
                        if (dosage != null && selectedTemplate != null) {
                            actualViewModel.createEventFromTemplate(selectedTemplate!!, dosage)
                        }
                        showDosageDialog = false
                        selectedTemplate = null
                    },
                    enabled = dosageInput.toDoubleOrNull() != null
                ) {
                    Text("Create Event")
                }
            },
            dismissButton = {
                TextButton(
                    onClick = { 
                        showDosageDialog = false
                        selectedTemplate = null
                    }
                ) {
                    Text("Cancel")
                }
            }
        )
        
        // Auto-focus the text field when dialog opens
        LaunchedEffect(showDosageDialog) {
            if (showDosageDialog) {
                kotlinx.coroutines.delay(100) // Small delay to ensure text field is composed
                focusRequester.requestFocus()
            }
        }
    }
    
    // Privacy popup dialog
    if (showPrivacyPopup) {
        MarkdownDialog(
            markdownContent = popupContent,
            onDismiss = {
                showPrivacyPopup = false
                coroutineScope.launch {
                    // Always dismiss with timestamp - permanently if checkbox is checked
                    preferencesRepository.dismissPrivacyPopup(permanently = dontShowAgainChecked)
                }
            },
            showDontShowAgain = true,
            dontShowAgainChecked = dontShowAgainChecked,
            onDontShowAgainChanged = { dontShowAgainChecked = it },
            title = "Important Information"
        )
    }
}
