package de.ntdote.medicalcalendarlog.ui.screens

import androidx.compose.foundation.layout.*
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material.icons.filled.Visibility
import androidx.compose.material.icons.filled.VisibilityOff
import androidx.compose.material3.*
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.unit.dp
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import de.ntdote.medicalcalendarlog.data.*
import de.ntdote.medicalcalendarlog.ui.components.AdvancedGraphCanvas
import de.ntdote.medicalcalendarlog.ui.components.GraphItemSelector
import de.ntdote.medicalcalendarlog.utils.AdvancedCalculationEngine
import de.ntdote.medicalcalendarlog.utils.GraphUtils
import de.ntdote.medicalcalendarlog.viewmodel.MainViewModel
import java.util.*

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun GraphScreen(
    onNavigateBack: () -> Unit,
    viewModel: MainViewModel,
    modifier: Modifier = Modifier
) {
    val uiState by viewModel.uiState.collectAsStateWithLifecycle()
    val events by viewModel.events.collectAsStateWithLifecycle()
    val showEventText by viewModel.showEventText.collectAsStateWithLifecycle()
    
    // State for graphable items and selection
    var graphableItems by remember { mutableStateOf<List<GraphableItem>>(emptyList()) }
    var graphState by remember { mutableStateOf(GraphState()) }
    
    // Zoom and pan state - use simple state variables
    var zoomLevel by remember { mutableStateOf(1f) }
    var panOffset by remember { mutableStateOf(0f) }
    
    // Force recreation counter
    var canvasRecreationKey by remember { mutableStateOf(0) }
    
    // Initialize graphable items when templates change
    LaunchedEffect(uiState.templates) {
        if (uiState.templates.isNotEmpty()) {
            val colors = GraphUtils.getDrugColors()
            val items = AdvancedCalculationEngine.createGraphableItems(uiState.templates, colors)
            graphableItems = items
            
            // Update graph state with new items
            graphState = graphState.copy(
                items = items,
                daysBack = uiState.daysBack,
                showEventText = showEventText
            )
        }
    }
    
    // Update graph state when settings change
    LaunchedEffect(uiState.daysBack, showEventText) {
        graphState = graphState.copy(
            daysBack = uiState.daysBack,
            showEventText = showEventText,
            startTime = Date(System.currentTimeMillis() - uiState.daysBack * 24 * 60 * 60 * 1000L),
            endTime = Date()
        )
    }
    
    // Generate graph data for selected items
    val selectedItems = graphableItems.filter { it.isSelected }
    val graphDataList = remember(selectedItems, events, graphState.startTime, graphState.endTime) {
        if (selectedItems.isNotEmpty() && events.isNotEmpty()) {
            AdvancedCalculationEngine.generateGraphData(
                items = selectedItems,
                events = events,
                startTime = graphState.startTime,
                endTime = graphState.endTime
            )
        } else {
            emptyList()
        }
    }
    
    
    Scaffold(
        topBar = {
            TopAppBar(
                title = { Text("Graph") },
                navigationIcon = {
                    IconButton(onClick = onNavigateBack) {
                        Icon(Icons.AutoMirrored.Filled.ArrowBack, contentDescription = "Back")
                    }
                },
                actions = {
                    Row(
                        verticalAlignment = Alignment.CenterVertically,
                        horizontalArrangement = Arrangement.spacedBy(4.dp)
                    ) {
                        Icon(
                            imageVector = if (showEventText) Icons.Default.Visibility else Icons.Default.VisibilityOff,
                            contentDescription = if (showEventText) "Hide event text" else "Show event text",
                            tint = MaterialTheme.colorScheme.onSurface.copy(alpha = 0.6f),
                            modifier = Modifier.size(20.dp)
                        )
                        Switch(
                            checked = showEventText,
                            onCheckedChange = { viewModel.toggleEventTextVisibility() },
                            modifier = Modifier.height(24.dp)
                        )
                    }
                }
            )
        },
        modifier = modifier
    ) { paddingValues ->
        Column(
            modifier = Modifier
                .fillMaxSize()
                .padding(paddingValues)
                .padding(16.dp)
        ) {
            // Item selection
            if (graphableItems.isNotEmpty()) {
                Card(
                    modifier = Modifier
                        .fillMaxWidth()
                        .padding(bottom = 8.dp)
                ) {
                    GraphItemSelector(
                        items = graphableItems,
                        modifier = Modifier.padding(12.dp),
                        onItemToggle = { itemId ->
                                // Find the item being toggled
                                val toggledItem = graphableItems.find { it.id == itemId }
                                
                                // Update the UI state immediately
                                graphableItems = graphableItems.map { item ->
                                    if (item.id == itemId) {
                                        when (item) {
                                            is GraphableItem.Drug -> item.copy(isSelected = !item.isSelected)
                                            is GraphableItem.MetricTemplate -> item.copy(isSelected = !item.isSelected)
                                            is GraphableItem.GeneralTemplate -> item.copy(isSelected = !item.isSelected)
                                            is GraphableItem.DecayingTemplate -> item.copy(isSelected = !item.isSelected)
                                        }
                                    } else {
                                        item
                                    }
                                }
                                
                                // Force Canvas recreation by incrementing the key
                                canvasRecreationKey++
                                
                                // Update the persistent template visibility (skip reminder recalculation)
                                toggledItem?.let { item ->
                                    when (item) {
                                        is GraphableItem.Drug -> {
                                            // For drugs, update the drug's visible property in all templates containing this drug
                                            item.templates.forEach { template ->
                                                // Update the specific drug's visible property within the template
                                                val updatedDrugs = template.drugs.map { drug ->
                                                    if (drug.drugType == item.drugType) {
                                                        drug.copy(visible = !item.isSelected)
                                                    } else {
                                                        drug
                                                    }
                                                }
                                                val updatedTemplate = template.copy(drugs = updatedDrugs)
                                                viewModel.updateTemplate(updatedTemplate, recalculateReminders = false)
                                                
                                                // COLLISION CASE: Check if template name matches this drug name
                                                if (template.name == item.drugType) {
                                                    // Auto-sync template visibility based on at least one drug visible
                                                    val anyDrugVisible = updatedDrugs.any { it.visible }
                                                    val finalTemplate = updatedTemplate.copy(visible = anyDrugVisible)
                                                    viewModel.updateTemplate(finalTemplate, recalculateReminders = false)
                                                }
                                            }
                                        }
                                        is GraphableItem.MetricTemplate -> {
                                            val updatedTemplate = item.template.copy(visible = !item.isSelected)
                                            viewModel.updateTemplate(updatedTemplate, recalculateReminders = false)
                                        }
                                        is GraphableItem.GeneralTemplate -> {
                                            val updatedTemplate = item.template.copy(visible = !item.isSelected)
                                            viewModel.updateTemplate(updatedTemplate, recalculateReminders = false)
                                        }
                                        is GraphableItem.DecayingTemplate -> {
                                            val updatedTemplate = item.template.copy(visible = !item.isSelected)
                                            viewModel.updateTemplate(updatedTemplate, recalculateReminders = false)
                                        }
                                    }
                                }
                            }
                        )
                }
            }
        
            // Main graph - fills remaining space
            Card(
                modifier = Modifier
                    .fillMaxWidth()
                    .weight(1f)
                    .padding(bottom = 8.dp)
            ) {
            Column(
                modifier = Modifier
                    .fillMaxSize()
                    .padding(16.dp)
            ) {
                Text(
                    text = "Concentration & Metrics Over Time",
                    style = MaterialTheme.typography.titleMedium,
                    fontWeight = FontWeight.Bold,
                    modifier = Modifier.padding(bottom = 16.dp)
                )
                
                if (graphDataList.isNotEmpty()) {
                    // Force Canvas recreation when selection changes while preserving zoom/pan
                    key(canvasRecreationKey) {
                        AdvancedGraphCanvas(
                            graphDataList = graphDataList,
                            events = events,
                            showEventText = showEventText,
                            startTime = graphState.startTime,
                            endTime = graphState.endTime,
                            zoomLevel = zoomLevel,
                            panOffset = panOffset,
                            onZoomChange = { newZoom, newPan ->
                                zoomLevel = newZoom.coerceIn(0.5f, 20f)
                                panOffset = newPan
                            },
                            modifier = Modifier
                                .fillMaxWidth()
                                .weight(1f)
                        )
                    }
                } else {
                    Box(
                        modifier = Modifier
                            .fillMaxWidth()
                            .weight(1f),
                        contentAlignment = Alignment.Center
                    ) {
                        Column(
                            horizontalAlignment = Alignment.CenterHorizontally,
                            verticalArrangement = Arrangement.spacedBy(8.dp)
                        ) {
                            Text(
                                text = when {
                                    uiState.templates.isEmpty() -> "No templates configured"
                                    graphableItems.isEmpty() -> "No graphable templates found"
                                    selectedItems.isEmpty() -> "No items selected"
                                    events.isEmpty() -> "No events found"
                                    else -> "No data available"
                                },
                                style = MaterialTheme.typography.bodyLarge,
                                color = MaterialTheme.colorScheme.onSurface.copy(alpha = 0.6f)
                            )
                            
                            if (uiState.templates.isEmpty()) {
                                Text(
                                    text = "Configure templates in settings to see graphs",
                                    style = MaterialTheme.typography.bodyMedium,
                                    color = MaterialTheme.colorScheme.onSurface.copy(alpha = 0.4f)
                                )
                            } else if (graphableItems.isEmpty()) {
                                Text(
                                    text = "Create decaying or metric templates to enable graphing",
                                    style = MaterialTheme.typography.bodyMedium,
                                    color = MaterialTheme.colorScheme.onSurface.copy(alpha = 0.4f)
                                )
                            }
                        }
                    }
                }
            }
        }
        
            // Time range info
            Card(
                modifier = Modifier
                    .fillMaxWidth()
            ) {
            Row(
                modifier = Modifier
                    .fillMaxWidth()
                    .padding(16.dp),
                horizontalArrangement = Arrangement.SpaceBetween
            ) {
                Column {
                    Text(
                        text = "Time Range",
                        style = MaterialTheme.typography.titleSmall,
                        fontWeight = FontWeight.Medium
                    )
                    Text(
                        text = "${uiState.daysBack} days back",
                        style = MaterialTheme.typography.bodyMedium,
                        color = MaterialTheme.colorScheme.onSurface.copy(alpha = 0.7f)
                    )
                }
                
                Column(
                    horizontalAlignment = Alignment.End
                ) {
                    Text(
                        text = "Data Points",
                        style = MaterialTheme.typography.titleSmall,
                        fontWeight = FontWeight.Medium
                    )
                    Text(
                        text = "${events.size} events",
                        style = MaterialTheme.typography.bodyMedium,
                        color = MaterialTheme.colorScheme.onSurface.copy(alpha = 0.7f)
                    )
                }
            }
        }
        }
    }
}
