package de.ntdote.medicalcalendarlog.ui.components

import android.util.Log
import androidx.compose.foundation.Canvas
import androidx.compose.foundation.background
import androidx.compose.foundation.gestures.detectTapGestures
import androidx.compose.foundation.gestures.detectTransformGestures
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material3.*
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.geometry.Offset
import androidx.compose.ui.graphics.*
import androidx.compose.ui.graphics.drawscope.DrawScope
import androidx.compose.ui.graphics.drawscope.Stroke
import androidx.compose.ui.graphics.drawscope.drawIntoCanvas
import androidx.compose.ui.graphics.nativeCanvas
import androidx.compose.ui.input.pointer.pointerInput
import androidx.compose.ui.platform.LocalDensity
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.compose.ui.window.Dialog
import kotlinx.coroutines.delay
import de.ntdote.medicalcalendarlog.data.*
import de.ntdote.medicalcalendarlog.utils.AdvancedCalculationEngine
import de.ntdote.medicalcalendarlog.utils.DrugResolution
import java.text.SimpleDateFormat
import java.util.*
import kotlin.math.*

/**
 * Helper function to check if an event matches a template
 */
private fun matchesTemplate(event: CalendarEvent, template: Template): Boolean {
    return event.title == template.name || event.title.startsWith(template.name)
}

data class EventClickData(
    val dosageWithUnit: String,
    val beforeValue: String,
    val afterValue: String,
    val metricName: String,
    val hoursSincePrevious: String,
    val hoursUntilNext: String,
    val averageValue: String
)

/**
 * Stores the rendered position of an event marker for accurate click detection
 */
data class EventMarkerPosition(
    val event: CalendarEvent,
    val x: Float,
    val y: Float,
    val graphData: GraphData
)

@Composable
fun AdvancedGraphCanvas(
    graphDataList: List<GraphData>,
    events: List<CalendarEvent>,
    showEventText: Boolean,
    startTime: Date,
    endTime: Date,
    zoomLevel: Float = 1f,
    panOffset: Float = 0f,
    onZoomChange: ((Float, Float) -> Unit)? = null,
    onEventClick: (CalendarEvent) -> Unit = {},
    modifier: Modifier = Modifier
) {
    var clickedEvent by remember { mutableStateOf<CalendarEvent?>(null) }
    var clickedEventData by remember { mutableStateOf<EventClickData?>(null) }
    
    // Store event marker positions for accurate click detection
    var eventMarkerPositions by remember { mutableStateOf<List<EventMarkerPosition>>(emptyList()) }
    
    // Use internal state for gesture management
    var currentZoom by remember { mutableStateOf(zoomLevel) }
    var currentPan by remember { mutableStateOf(panOffset) }
    
    // Update internal state when external values change
    LaunchedEffect(zoomLevel, panOffset) {
        currentZoom = zoomLevel
        currentPan = panOffset
    }
    
    Box(modifier = modifier) {
        Canvas(
            modifier = Modifier
                .fillMaxSize()
                .pointerInput(Unit) {
                    detectTransformGestures(
                        onGesture = { centroid, pan, zoom, _ ->
                            // Apply zoom with proper bounds (prevent zooming out beyond original view)
                            val newZoom = (currentZoom * zoom).coerceIn(1f, 20f)
                            
                            // Convert pan from screen coordinates to time coordinates
                            val originalTimeRange = endTime.time - startTime.time
                            val screenWidth = size.width - 160f // Account for padding
                            val timePerPixel = originalTimeRange.toFloat() / screenWidth
                            val panInTime = pan.x * timePerPixel / currentZoom
                            
                            // Calculate new pan with proper bounds
                            val zoomedTimeRange = (originalTimeRange / newZoom).toLong()
                            val maxPanTime = (originalTimeRange - zoomedTimeRange).toFloat()
                            val newPan = (currentPan - panInTime).coerceIn(-maxPanTime, maxPanTime)
                            
                            currentZoom = newZoom
                            currentPan = newPan
                            
                            // Notify parent immediately for responsive feedback
                            onZoomChange?.invoke(newZoom, newPan)
                        }
                    )
                }
                .pointerInput(graphDataList, eventMarkerPositions) {
                    detectTapGestures { offset ->
                        // Handle event clicks using actual marker positions
                        val nearestEvent = findNearestEventFromPositions(
                            offset, eventMarkerPositions
                        )
                        nearestEvent?.let { event ->
                            val eventData = calculateEventData(event, graphDataList)
                            clickedEvent = event
                            clickedEventData = eventData
                            onEventClick(event)
                        }
                    }
                }
        ) {
            // Always draw the live graph with current zoom/pan and capture marker positions
            val positions = drawAdvancedGraph(
                graphDataList = graphDataList,
                events = events,
                showEventText = showEventText,
                startTime = startTime,
                endTime = endTime,
                zoomLevel = currentZoom,
                panOffset = currentPan
            )
            eventMarkerPositions = positions
        }
        
        
        // Event details dialog
        clickedEvent?.let { event ->
            EventDetailsDialog(
                event = event,
                eventData = clickedEventData,
                onDismiss = { 
                    clickedEvent = null
                    clickedEventData = null
                }
            )
        }
    }
    
}

@Composable
private fun EventDetailsDialog(
    event: CalendarEvent,
    eventData: EventClickData? = null,
    onDismiss: () -> Unit
) {
    Dialog(onDismissRequest = onDismiss) {
        Card(
            modifier = Modifier
                .fillMaxWidth()
                .padding(16.dp),
            shape = RoundedCornerShape(16.dp)
        ) {
            Column(
                modifier = Modifier.padding(20.dp),
                verticalArrangement = Arrangement.spacedBy(12.dp)
            ) {
                Text(
                    text = "Event Details",
                    style = MaterialTheme.typography.titleLarge,
                    fontWeight = FontWeight.Bold
                )
                
                HorizontalDivider()
                
                DetailRow("Event", event.title)
                DetailRow("Time", SimpleDateFormat("MMM dd, yyyy HH:mm", Locale.getDefault()).format(event.startTime))
                
                if (eventData != null) {
                    // Determine event type based on event title and data
                    val isGeneralEvent = eventData.beforeValue == "N/A" && eventData.afterValue == "N/A"
                    val isMetricEvent = eventData.beforeValue == "N/A" && eventData.afterValue != "N/A"
                    val isDrugEvent = eventData.beforeValue != "N/A" && eventData.afterValue != "N/A"
                    
                    when {
                        isGeneralEvent -> {
                            // General events: only show event and time (no dosage, before, after)
                            // Already shown above
                            if (eventData.hoursSincePrevious != "-") {
                                DetailRow("Since Previous", eventData.hoursSincePrevious)
                            }
                            if (eventData.hoursUntilNext != "-") {
                                DetailRow("Until Next", eventData.hoursUntilNext)
                            }
                            if (eventData.averageValue != "-") {
                                DetailRow("Average Timespan", eventData.averageValue)
                            }
                        }
                        isMetricEvent -> {
                            // Metric events: show value but no before/after event
                            DetailRow("Value", eventData.afterValue)
                            if (eventData.hoursSincePrevious != "-") {
                                DetailRow("Since Previous", eventData.hoursSincePrevious)
                            }
                            if (eventData.hoursUntilNext != "-") {
                                DetailRow("Until Next", eventData.hoursUntilNext)
                            }
                            if (eventData.averageValue != "-") {
                                DetailRow("Average Value", eventData.averageValue)
                            }
                        }
                        isDrugEvent -> {
                            // Drug events: show all information
                            DetailRow("Dosage", eventData.dosageWithUnit)
                            DetailRow("Before Event", "${eventData.metricName}: ${eventData.beforeValue}")
                            DetailRow("After Event", "${eventData.metricName}: ${eventData.afterValue}")
                            if (eventData.hoursSincePrevious != "-") {
                                DetailRow("Since Previous", eventData.hoursSincePrevious)
                            }
                            if (eventData.hoursUntilNext != "-") {
                                DetailRow("Until Next", eventData.hoursUntilNext)
                            }
                            if (eventData.averageValue != "-") {
                                DetailRow("Average", eventData.averageValue)
                            }
                        }
                    }
                } else {
                    // Fallback: show dosage if no eventData available
                    DetailRow("Dosage", "${event.dosage}")
                }
                
                Row(
                    modifier = Modifier.fillMaxWidth(),
                    horizontalArrangement = Arrangement.End
                ) {
                    TextButton(onClick = onDismiss) {
                        Text("Close")
                    }
                }
            }
        }
    }
}

@Composable
private fun DetailRow(label: String, value: String) {
    Row(
        modifier = Modifier.fillMaxWidth(),
        horizontalArrangement = Arrangement.SpaceBetween
    ) {
        Text(
            text = "$label:",
            fontWeight = FontWeight.Medium,
            color = MaterialTheme.colorScheme.onSurface.copy(alpha = 0.7f)
        )
        Text(
            text = value,
            fontWeight = FontWeight.Normal
        )
    }
}

private fun DrawScope.drawAdvancedGraph(
    graphDataList: List<GraphData>,
    events: List<CalendarEvent>,
    showEventText: Boolean,
    startTime: Date,
    endTime: Date,
    zoomLevel: Float,
    panOffset: Float,
    padding: Float = 80f
): List<EventMarkerPosition> {
    if (graphDataList.isEmpty()) return emptyList()
    
    val graphWidth = size.width - 2 * padding
    val graphHeight = size.height - 2 * padding
    val originalTimeRange = endTime.time - startTime.time
    
    // Safety check: ensure we have valid time range
    if (originalTimeRange <= 0) return emptyList()
    
    // Apply zoom and pan transformations with safety checks
    val safeZoomLevel = zoomLevel.coerceIn(1f, 20f) // Prevent zoom out beyond original view
    val zoomedTimeRange = (originalTimeRange / safeZoomLevel).toLong()
    
    // Calculate pan bounds - when zoomed in, we can pan within the extra time
    val maxPanTime = if (safeZoomLevel > 1f) {
        (originalTimeRange - zoomedTimeRange).toFloat()
    } else {
        0f // No panning when at original zoom level
    }
    
    val clampedPanOffset = if (maxPanTime > 0) {
        panOffset.coerceIn(-maxPanTime, maxPanTime)
    } else {
        0f
    }
    
    // Calculate the visible time window based on zoom and pan
    val visibleStartTime = startTime.time + clampedPanOffset.toLong()
    val visibleEndTime = visibleStartTime + zoomedTimeRange
    
    // Additional safety check: ensure visible times are valid
    if (visibleStartTime >= visibleEndTime || zoomedTimeRange <= 0) return emptyList()
    
    // Draw background
    drawRect(
        color = Color.Black.copy(alpha = 0.05f),
        topLeft = Offset(padding, padding),
        size = androidx.compose.ui.geometry.Size(graphWidth, graphHeight)
    )
    
    // Draw time grid and special lines with zoom/pan applied
    drawTimeGrid(padding, graphWidth, graphHeight, visibleStartTime, visibleEndTime, zoomedTimeRange)
    drawMidnightLines(padding, graphWidth, graphHeight, visibleStartTime, visibleEndTime, zoomedTimeRange)
    drawCurrentTimeLine(padding, graphWidth, graphHeight, visibleStartTime, visibleEndTime, zoomedTimeRange)
    
    // Draw each graph line with zoom/pan applied
    graphDataList.forEach { graphData ->
        drawGraphLine(
            graphData = graphData,
            padding = padding,
            graphWidth = graphWidth,
            graphHeight = graphHeight,
            startTime = visibleStartTime,
            timeRange = zoomedTimeRange
        )
    }
    
    // Draw event markers with zoom/pan applied and capture their positions
    val markerPositions = drawEventMarkers(
        events = events,
        graphDataList = graphDataList,
        showEventText = showEventText,
        padding = padding,
        graphWidth = graphWidth,
        graphHeight = graphHeight,
        startTime = visibleStartTime,
        timeRange = zoomedTimeRange
    )
    
    // Draw axes
    drawAxes(padding, graphWidth, graphHeight)
    
    // Draw axis labels with zoom/pan applied
    drawAxisLabels(
        padding = padding,
        graphWidth = graphWidth,
        graphHeight = graphHeight,
        startTime = Date(visibleStartTime),
        endTime = Date(visibleEndTime),
        graphDataList = graphDataList
    )
    
    return markerPositions
}

private fun DrawScope.drawSimplifiedGraph(
    graphDataList: List<GraphData>,
    events: List<CalendarEvent>,
    showEventText: Boolean,
    startTime: Date,
    endTime: Date,
    zoomLevel: Float,
    panOffset: Float,
    padding: Float = 80f
) {
    if (graphDataList.isEmpty()) return
    
    val graphWidth = size.width - 2 * padding
    val graphHeight = size.height - 2 * padding
    val timeRange = endTime.time - startTime.time
    
    // Draw simplified background
    drawRect(
        color = Color.Black.copy(alpha = 0.03f),
        topLeft = Offset(padding, padding),
        size = androidx.compose.ui.geometry.Size(graphWidth, graphHeight)
    )
    
    // Draw only essential grid lines (reduced density for performance)
    drawSimplifiedTimeGrid(padding, graphWidth, graphHeight, startTime.time, endTime.time, timeRange)
    
    // Draw simplified graph lines (fewer points, thicker lines)
    graphDataList.forEach { graphData ->
        drawSimplifiedGraphLine(
            graphData = graphData,
            padding = padding,
            graphWidth = graphWidth,
            graphHeight = graphHeight,
            startTime = startTime.time,
            timeRange = timeRange
        )
    }
    
    // Draw simplified event markers (dots only, no text)
    drawSimplifiedEventMarkers(
        events = events,
        graphDataList = graphDataList,
        padding = padding,
        graphWidth = graphWidth,
        graphHeight = graphHeight,
        startTime = startTime.time,
        timeRange = timeRange
    )
    
    // Draw basic axes
    drawAxes(padding, graphWidth, graphHeight)
}

private fun DrawScope.drawSimplifiedTimeGrid(
    padding: Float,
    graphWidth: Float,
    graphHeight: Float,
    startTime: Long,
    endTime: Long,
    timeRange: Long
) {
    val gridColor = Color.Gray.copy(alpha = 0.1f)
    val daysBack = ((endTime - startTime) / (24 * 60 * 60 * 1000)).toInt()
    
    // Reduced grid density for performance
    val calendar = Calendar.getInstance()
    calendar.timeInMillis = startTime
    
    when {
        daysBack > 7 -> {
            // Only daily markers for long ranges
            calendar.set(Calendar.HOUR_OF_DAY, 0)
            calendar.set(Calendar.MINUTE, 0)
            calendar.set(Calendar.SECOND, 0)
            calendar.add(Calendar.DAY_OF_YEAR, 1)
            
            var count = 0
            while (calendar.timeInMillis <= endTime && count < 10) { // Limit lines
                val x = padding + ((calendar.timeInMillis - startTime).toFloat() / timeRange.toFloat()) * graphWidth
                drawLine(
                    color = gridColor,
                    start = Offset(x, padding),
                    end = Offset(x, size.height - padding),
                    strokeWidth = 0.5f
                )
                calendar.add(Calendar.DAY_OF_YEAR, maxOf(1, daysBack / 8))
                count++
            }
        }
        else -> {
            // Only 6-hour markers for short ranges
            calendar.set(Calendar.MINUTE, 0)
            calendar.set(Calendar.SECOND, 0)
            calendar.add(Calendar.HOUR_OF_DAY, 6)
            
            var count = 0
            while (calendar.timeInMillis <= endTime && count < 8) { // Limit lines
                val x = padding + ((calendar.timeInMillis - startTime).toFloat() / timeRange.toFloat()) * graphWidth
                drawLine(
                    color = gridColor,
                    start = Offset(x, padding),
                    end = Offset(x, size.height - padding),
                    strokeWidth = 0.5f
                )
                calendar.add(Calendar.HOUR_OF_DAY, 6)
                count++
            }
        }
    }
}

private fun DrawScope.drawSimplifiedGraphLine(
    graphData: GraphData,
    padding: Float,
    graphWidth: Float,
    graphHeight: Float,
    startTime: Long,
    timeRange: Long
) {
    val points = graphData.points.filter { it.event == null }
    if (points.size < 2) return
    
    val valueRange = graphData.maxValue - graphData.minValue
    
    // Use fewer points for performance - sample every nth point
    val sampleRate = maxOf(1, points.size / 50) // Max 50 points
    val sampledPoints = points.filterIndexed { index, _ -> index % sampleRate == 0 }
    
    // Draw simplified line segments (thicker for visibility)
    for (i in 0 until sampledPoints.size - 1) {
        val point1 = sampledPoints[i]
        val point2 = sampledPoints[i + 1]
        
        val x1 = padding + ((point1.time.time - startTime).toFloat() / timeRange.toFloat()) * graphWidth
        val y1 = size.height - padding - ((point1.value - graphData.minValue).toFloat() / valueRange.toFloat()) * graphHeight
        
        val x2 = padding + ((point2.time.time - startTime).toFloat() / timeRange.toFloat()) * graphWidth
        val y2 = size.height - padding - ((point2.value - graphData.minValue).toFloat() / valueRange.toFloat()) * graphHeight
        
        drawLine(
            color = graphData.item.color,
            start = Offset(x1, y1),
            end = Offset(x2, y2),
            strokeWidth = 4f // Thicker for gesture visibility
        )
    }
}

private fun DrawScope.drawSimplifiedEventMarkers(
    events: List<CalendarEvent>,
    graphDataList: List<GraphData>,
    padding: Float,
    graphWidth: Float,
    graphHeight: Float,
    startTime: Long,
    timeRange: Long
) {
    val visibleEvents = events.filter { event ->
        event.startTime.time >= startTime && event.startTime.time <= startTime + timeRange
    }
    
    // Only show dots, no text for performance
    visibleEvents.forEach { event ->
        val matchingGraphData = graphDataList.find { graphData ->
            if (!graphData.item.isSelected) return@find false
            
            when (graphData.item) {
                is GraphableItem.Drug -> {
                    graphData.item.templates.any { template -> matchesTemplate(event, template) }
                }
                is GraphableItem.MetricTemplate -> {
                    matchesTemplate(event, graphData.item.template)
                }
                is GraphableItem.GeneralTemplate -> {
                    matchesTemplate(event, graphData.item.template)
                }
                is GraphableItem.DecayingTemplate -> {
                    matchesTemplate(event, graphData.item.template)
                }
            }
        }
        
        if (matchingGraphData != null) {
            val x = padding + ((event.startTime.time - startTime).toFloat() / timeRange.toFloat()) * graphWidth
            val y = size.height - padding - 30f // Simple bottom positioning
            
            // Simple dot marker
            drawCircle(
                color = Color.White,
                radius = 6f,
                center = Offset(x, y)
            )
            drawCircle(
                color = Color.Black,
                radius = 6f,
                center = Offset(x, y),
                style = Stroke(width = 1f)
            )
        }
    }
}

private fun DrawScope.drawTimeGrid(
    padding: Float,
    graphWidth: Float,
    graphHeight: Float,
    startTime: Long,
    endTime: Long,
    timeRange: Long
) {
    val gridColor = Color.Gray.copy(alpha = 0.2f)
    val daysBack = ((endTime - startTime) / (24 * 60 * 60 * 1000)).toInt()
    
    // Adaptive time labels based on range
    val calendar = Calendar.getInstance()
    calendar.timeInMillis = startTime
    
    when {
        daysBack > 30 -> {
            // Weekly markers for long ranges
            calendar.set(Calendar.DAY_OF_WEEK, Calendar.MONDAY)
            calendar.set(Calendar.HOUR_OF_DAY, 0)
            calendar.set(Calendar.MINUTE, 0)
            calendar.set(Calendar.SECOND, 0)
            
            while (calendar.timeInMillis <= endTime) {
                val x = padding + ((calendar.timeInMillis - startTime).toFloat() / timeRange.toFloat()) * graphWidth
                drawLine(
                    color = gridColor,
                    start = Offset(x, padding),
                    end = Offset(x, size.height - padding),
                    strokeWidth = 1f
                )
                calendar.add(Calendar.WEEK_OF_YEAR, 1)
            }
        }
        daysBack > 7 -> {
            // Daily markers for medium ranges
            calendar.set(Calendar.HOUR_OF_DAY, 0)
            calendar.set(Calendar.MINUTE, 0)
            calendar.set(Calendar.SECOND, 0)
            calendar.add(Calendar.DAY_OF_YEAR, 1)
            
            while (calendar.timeInMillis <= endTime) {
                val x = padding + ((calendar.timeInMillis - startTime).toFloat() / timeRange.toFloat()) * graphWidth
                drawLine(
                    color = gridColor,
                    start = Offset(x, padding),
                    end = Offset(x, size.height - padding),
                    strokeWidth = 1f
                )
                calendar.add(Calendar.DAY_OF_YEAR, 1)
            }
        }
        else -> {
            // Hourly markers for short ranges
            calendar.set(Calendar.MINUTE, 0)
            calendar.set(Calendar.SECOND, 0)
            calendar.add(Calendar.HOUR_OF_DAY, 1)
            
            while (calendar.timeInMillis <= endTime) {
                val x = padding + ((calendar.timeInMillis - startTime).toFloat() / timeRange.toFloat()) * graphWidth
                drawLine(
                    color = gridColor,
                    start = Offset(x, padding),
                    end = Offset(x, size.height - padding),
                    strokeWidth = 1f
                )
                calendar.add(Calendar.HOUR_OF_DAY, if (daysBack <= 1) 2 else 6)
            }
        }
    }
}

private fun DrawScope.drawMidnightLines(
    padding: Float,
    graphWidth: Float,
    graphHeight: Float,
    startTime: Long,
    endTime: Long,
    timeRange: Long
) {
    val midnightColor = Color.Blue.copy(alpha = 0.4f)
    val calendar = Calendar.getInstance()
    
    calendar.timeInMillis = startTime
    calendar.set(Calendar.HOUR_OF_DAY, 0)
    calendar.set(Calendar.MINUTE, 0)
    calendar.set(Calendar.SECOND, 0)
    calendar.set(Calendar.MILLISECOND, 0)
    calendar.add(Calendar.DAY_OF_YEAR, 1)
    
    while (calendar.timeInMillis <= endTime) {
        val x = padding + ((calendar.timeInMillis - startTime).toFloat() / timeRange.toFloat()) * graphWidth
        
        drawLine(
            color = midnightColor,
            start = Offset(x, padding),
            end = Offset(x, size.height - padding),
            strokeWidth = 2f
        )
        
        calendar.add(Calendar.DAY_OF_YEAR, 1)
    }
}

private fun DrawScope.drawCurrentTimeLine(
    padding: Float,
    graphWidth: Float,
    graphHeight: Float,
    startTime: Long,
    endTime: Long,
    timeRange: Long
) {
    val currentTime = System.currentTimeMillis()
    
    if (currentTime >= startTime && currentTime <= endTime) {
        val x = padding + ((currentTime - startTime).toFloat() / timeRange.toFloat()) * graphWidth
        val currentTimeColor = Color.Red.copy(alpha = 0.8f)
        
        drawLine(
            color = currentTimeColor,
            start = Offset(x, padding),
            end = Offset(x, size.height - padding),
            strokeWidth = 3f
        )
        
        // Draw "NOW" label
        val paint = Paint().asFrameworkPaint().apply {
            isAntiAlias = true
            textSize = 32f
            color = android.graphics.Color.RED
            isFakeBoldText = true
        }
        
        drawContext.canvas.nativeCanvas.drawText(
            "NOW",
            x - paint.measureText("NOW") / 2,
            padding - 10f,
            paint
        )
    }
}

private fun DrawScope.drawGraphLine(
    graphData: GraphData,
    padding: Float,
    graphWidth: Float,
    graphHeight: Float,
    startTime: Long,
    timeRange: Long
) {
    // Filter points based on GraphableItem type
    val points = when (graphData.item) {
        is GraphableItem.MetricTemplate -> {
            // For metrics: show actual user measurement points (which have events)
            graphData.points.filter { it.event != null }
        }
        is GraphableItem.Drug -> {
            // For drugs: show interpolated curve points (which don't have events)
            graphData.points.filter { it.event == null }
        }
        is GraphableItem.GeneralTemplate -> {
            // For general templates: don't draw lines, only show as event markers
            return
        }
        is GraphableItem.DecayingTemplate -> {
            // For decaying templates: don't draw lines, only show as event markers
            return
        }
    }
    if (points.size < 2) return
    
    val valueRange = graphData.maxValue - graphData.minValue
    
    // Draw line segments
    for (i in 0 until points.size - 1) {
        val point1 = points[i]
        val point2 = points[i + 1]
        
        val x1 = padding + ((point1.time.time - startTime).toFloat() / timeRange.toFloat()) * graphWidth
        val y1 = size.height - padding - ((point1.value - graphData.minValue).toFloat() / valueRange.toFloat()) * graphHeight
        
        val x2 = padding + ((point2.time.time - startTime).toFloat() / timeRange.toFloat()) * graphWidth
        val y2 = size.height - padding - ((point2.value - graphData.minValue).toFloat() / valueRange.toFloat()) * graphHeight
        
        drawLine(
            color = graphData.item.color,
            start = Offset(x1, y1),
            end = Offset(x2, y2),
            strokeWidth = 3f
        )
    }
    
    // Draw data points only for metrics (not for drugs)
    when (graphData.item) {
        is GraphableItem.MetricTemplate -> {
            // Show dots for metric measurements
            points.forEach { point ->
                val x = padding + ((point.time.time - startTime).toFloat() / timeRange.toFloat()) * graphWidth
                val y = size.height - padding - ((point.value - graphData.minValue).toFloat() / valueRange.toFloat()) * graphHeight
                
                drawCircle(
                    color = graphData.item.color,
                    radius = 4f,
                    center = Offset(x, y)
                )
            }
        }
        is GraphableItem.Drug -> {
            // No dots for drug curves - just smooth lines
        }
        is GraphableItem.GeneralTemplate -> {
            // No dots for general templates (they don't draw lines anyway)
        }
        is GraphableItem.DecayingTemplate -> {
            // No dots for decaying templates (they don't draw lines anyway)
        }
    }
}

private fun DrawScope.drawEventMarkers(
    events: List<CalendarEvent>,
    graphDataList: List<GraphData>,
    showEventText: Boolean,
    padding: Float,
    graphWidth: Float,
    graphHeight: Float,
    startTime: Long,
    timeRange: Long
): List<EventMarkerPosition> {
    val markerPositions = mutableListOf<EventMarkerPosition>()
    val paint = Paint().asFrameworkPaint().apply {
        isAntiAlias = true
        textSize = 36f
        color = android.graphics.Color.WHITE
    }
    
    val visibleEvents = events.filter { event ->
        event.startTime.time >= startTime && event.startTime.time <= startTime + timeRange
    }
    
    // Create a map to assign unique vertical lanes for template markers at bottom
    // This includes GeneralTemplate, DecayingTemplate items AND Drug template names (when different from drug name)
    val templateLanes = mutableMapOf<String, Float>()
    var currentLane = 0
    val laneSpacing = 25f // Minimum spacing between lanes

    // Single unified pass: assign lanes in the order items appear in graphDataList
    // Only assign lanes to items that are SELECTED and will actually be drawn
    Log.d("MCL", "=== Lane Assignment Start ===")
    graphDataList.forEach { graphData ->
        when (graphData.item) {
            is GraphableItem.GeneralTemplate -> {
                // Only assign lane if this item is selected (will be drawn)
                if (graphData.item.isSelected) {
                    val templateId = graphData.item.template.id
                    if (!templateLanes.containsKey(templateId)) {
                        templateLanes[templateId] = size.height - padding - 30f - (currentLane * laneSpacing)
                        Log.d("MCL", "Lane $currentLane assigned to GeneralTemplate '${graphData.item.template.name}' - Reason: isSelected=true")
                        currentLane++
                    }
                } else {
                    Log.d("MCL", "GeneralTemplate '${graphData.item.template.name}' NOT assigned lane - Reason: isSelected=false")
                }
            }
            is GraphableItem.DecayingTemplate -> {
                // DecayingTemplate items should NOT get lanes - they don't draw at bottom
                // Only individual Drug items draw at bottom (at their concentration Y level)
                Log.d("MCL", "DecayingTemplate '${graphData.item.template.name}' NOT assigned lane - Reason: DecayingTemplates never get lanes")
            }
            is GraphableItem.Drug -> {
                // For drugs: assign lanes to visible templates that have names different from drug type
                // BUT only if the Drug item itself is selected (otherwise it won't be drawn)
                if (graphData.item.isSelected) {
                    graphData.item.templates.forEach { template ->
                        if (template.visible && template.name != graphData.item.drugType) {
                            // CRITICAL: Check if this template name matches another Drug that is selected
                            // If it does, that Drug will draw its own markers at Y-position, so don't assign a lane
                            val isDrugNamePlotted = graphDataList.any { otherGraphData ->
                                otherGraphData.item is GraphableItem.Drug && 
                                otherGraphData.item.drugType == template.name &&
                                otherGraphData.item.isSelected
                            }
                            
                            if (!isDrugNamePlotted) {
                                val templateId = template.id
                                if (!templateLanes.containsKey(templateId)) {
                                    templateLanes[templateId] = size.height - padding - 30f - (currentLane * laneSpacing)
                                    Log.d("MCL", "Lane $currentLane assigned to Drug template '${template.name}' (ID: $templateId) for drug '${graphData.item.drugType}' - Reason: Drug.isSelected=true, template.visible=true, template.name != drugType, template.name is NOT a plotted drug")
                                    currentLane++
                                }
                            } else {
                                Log.d("MCL", "Drug template '${template.name}' for drug '${graphData.item.drugType}' NOT assigned lane - Reason: template.name matches a selected/plotted Drug")
                            }
                        } else {
                            val reasons = mutableListOf<String>()
                            if (!template.visible) reasons.add("template.visible=false")
                            if (template.name == graphData.item.drugType) reasons.add("template.name == drugType (collision)")
                            Log.d("MCL", "Drug template '${template.name}' for drug '${graphData.item.drugType}' NOT assigned lane - Reason: ${reasons.joinToString(", ")}")
                        }
                    }
                } else {
                    Log.d("MCL", "Drug '${graphData.item.drugType}' templates NOT assigned lanes - Reason: Drug.isSelected=false")
                }
            }
            is GraphableItem.MetricTemplate -> {
                // Metric templates don't appear at bottom
                Log.d("MCL", "MetricTemplate '${graphData.item.template.name}' NOT assigned lane - Reason: MetricTemplates never get lanes")
            }
        }
    }
    Log.d("MCL", "=== Lane Assignment Complete: $currentLane lanes assigned ===")
    
    // Keep track of drug names that were already drawn at Y positions
    val drawnDrugNames = mutableSetOf<String>()
    
    visibleEvents.forEach { event ->
        Log.d("MCL", "=== Processing Event: '${event.title}' at ${SimpleDateFormat("HH:mm:ss", Locale.getDefault()).format(event.startTime)} ===")
        
        val x = padding + ((event.startTime.time - startTime).toFloat() / timeRange.toFloat()) * graphWidth
        
        // Find all matching graph data for this event
        val matchingItems = graphDataList.filter { graphData ->
            if (!graphData.item.isSelected) return@filter false
            
            when (graphData.item) {
                is GraphableItem.Drug -> {
                    graphData.item.templates.any { template -> matchesTemplate(event, template) }
                }
                is GraphableItem.MetricTemplate -> {
                    matchesTemplate(event, graphData.item.template)
                }
                is GraphableItem.GeneralTemplate -> {
                    matchesTemplate(event, graphData.item.template)
                }
                is GraphableItem.DecayingTemplate -> {
                    matchesTemplate(event, graphData.item.template)
                }
            }
        }
        
        Log.d("MCL", "Found ${matchingItems.size} matching items for event '${event.title}'")
        
        matchingItems.forEach { matchingGraphData ->
            when (val item = matchingGraphData.item) {
                is GraphableItem.Drug -> {
                    // ALWAYS draw drug name at its concentration Y position
                    val drugName = item.drugType
                    val displayText = if (drugName.length > 12) drugName.take(9) + "..." else drugName
                    
                    // Find the Y position for this drug
                    val eventPoint = matchingGraphData.points.find { it.event == event }
                    val y = if (eventPoint != null) {
                        val valueRange = matchingGraphData.maxValue - matchingGraphData.minValue
                        if (valueRange > 0) {
                            size.height - padding - ((eventPoint.value - matchingGraphData.minValue).toFloat() / valueRange.toFloat()) * graphHeight
                        } else {
                            size.height - padding - graphHeight / 2
                        }
                    } else {
                        size.height - padding - 30f
                    }
                    
                    // Draw the drug name at Y position
                    if (showEventText) {
                        val textWidth = paint.measureText(displayText)
                        drawContext.canvas.nativeCanvas.drawText(
                            displayText,
                            x - textWidth / 2,
                            y + paint.textSize / 3,
                            paint
                        )
                    } else {
                        drawCircle(
                            color = Color.White,
                            radius = 5f,
                            center = Offset(x, y)
                        )
                        drawCircle(
                            color = Color.Black,
                            radius = 5f,
                            center = Offset(x, y),
                            style = Stroke(width = 1f)
                        )
                    }
                    
                    // Store the marker position for click detection
                    markerPositions.add(EventMarkerPosition(event, x, y, matchingGraphData))
                    
                    // Track that this drug name was drawn
                    drawnDrugNames.add(drugName)
                    
                    // Now handle template names in bottom lanes
                    val sourceTemplate = item.templates.find { template -> matchesTemplate(event, template) }
                    if (sourceTemplate != null && sourceTemplate.visible) {
                        // Only draw template name in bottom lane if it hasn't already been drawn as a drug name
                        if (sourceTemplate.name !in drawnDrugNames) {
                            val templateY = templateLanes[sourceTemplate.id] ?: (size.height - padding - 30f)
                            val templateDisplayText = if (sourceTemplate.name.length > 12) sourceTemplate.name.take(9) + "..." else sourceTemplate.name
                            
                            if (showEventText) {
                                val textWidth = paint.measureText(templateDisplayText)
                                drawContext.canvas.nativeCanvas.drawText(
                                    templateDisplayText,
                                    x - textWidth / 2,
                                    templateY + paint.textSize / 3,
                                    paint
                                )
                            } else {
                                drawCircle(
                                    color = Color.White,
                                    radius = 5f,
                                    center = Offset(x, templateY)
                                )
                                drawCircle(
                                    color = Color.Black,
                                    radius = 5f,
                                    center = Offset(x, templateY),
                                    style = Stroke(width = 1f)
                                )
                            }
                        }
                    }
                }
                
                is GraphableItem.MetricTemplate -> {
                    // Metrics: Show at value position
                    val eventPoint = matchingGraphData.points.find { it.event == event }
                    val y = if (eventPoint != null) {
                        val valueRange = matchingGraphData.maxValue - matchingGraphData.minValue
                        if (valueRange > 0) {
                            size.height - padding - ((eventPoint.value - matchingGraphData.minValue).toFloat() / valueRange.toFloat()) * graphHeight
                        } else {
                            size.height - padding - graphHeight / 2
                        }
                    } else {
                        val latestPoint = matchingGraphData.points.maxByOrNull { it.time.time }
                        if (latestPoint != null) {
                            val valueRange = matchingGraphData.maxValue - matchingGraphData.minValue
                            if (valueRange > 0) {
                                size.height - padding - ((latestPoint.value - matchingGraphData.minValue).toFloat() / valueRange.toFloat()) * graphHeight
                            } else {
                                size.height - padding - graphHeight / 2
                            }
                        } else {
                            size.height - padding - graphHeight / 2
                        }
                    }
                    
                    val cleanText = AdvancedCalculationEngine.cleanEventText(event.title)
                    val displayText = if (cleanText.length > 12) cleanText.take(9) + "..." else cleanText
                    
                    if (showEventText) {
                        val textWidth = paint.measureText(displayText)
                        drawContext.canvas.nativeCanvas.drawText(
                            displayText,
                            x - textWidth / 2,
                            y + paint.textSize / 3,
                            paint
                        )
                    } else {
                        drawCircle(
                            color = Color.White,
                            radius = 5f,
                            center = Offset(x, y)
                        )
                        drawCircle(
                            color = Color.Black,
                            radius = 5f,
                            center = Offset(x, y),
                            style = Stroke(width = 1f)
                        )
                    }
                    
                    // Store the marker position for click detection
                    markerPositions.add(EventMarkerPosition(event, x, y, matchingGraphData))
                }
                
                is GraphableItem.GeneralTemplate -> {
                    // General templates: Show at bottom in assigned lane
                    // Only draw template names that haven't already been drawn as drug names
                    if (item.template.name !in drawnDrugNames) {
                        val y = templateLanes[item.template.id] ?: (size.height - padding - 30f)
                        
                        val cleanText = AdvancedCalculationEngine.cleanEventText(event.title)
                        val displayText = if (cleanText.length > 12) cleanText.take(9) + "..." else cleanText
                        
                        if (showEventText) {
                            val textWidth = paint.measureText(displayText)
                            drawContext.canvas.nativeCanvas.drawText(
                                displayText,
                                x - textWidth / 2,
                                y + paint.textSize / 3,
                                paint
                            )
                        } else {
                            drawCircle(
                                color = Color.White,
                                radius = 5f,
                                center = Offset(x, y)
                            )
                            drawCircle(
                                color = Color.Black,
                                radius = 5f,
                                center = Offset(x, y),
                                style = Stroke(width = 1f)
                            )
                        }
                        
                        // Store the marker position for click detection
                        markerPositions.add(EventMarkerPosition(event, x, y, matchingGraphData))
                    }
                }
                
                is GraphableItem.DecayingTemplate -> {
                    // Only draw if the DecayingTemplate item is selected
                    if (item.isSelected) {
                        // Only draw template names that haven't already been drawn as drug names
                        if (item.template.name !in drawnDrugNames) {
                            val y = templateLanes[item.template.id] ?: (size.height - padding - 30f)
                            
                            val cleanText = AdvancedCalculationEngine.cleanEventText(event.title)
                            val displayText = if (cleanText.length > 12) cleanText.take(9) + "..." else cleanText
                            
                            if (showEventText) {
                                val textWidth = paint.measureText(displayText)
                                drawContext.canvas.nativeCanvas.drawText(
                                    displayText,
                                    x - textWidth / 2,
                                    y + paint.textSize / 3,
                                    paint
                                )
                            } else {
                                drawCircle(
                                    color = Color.White,
                                    radius = 5f,
                                    center = Offset(x, y)
                                )
                                drawCircle(
                                    color = Color.Black,
                                    radius = 5f,
                                    center = Offset(x, y),
                                    style = Stroke(width = 1f)
                                )
                            }
                            
                            // Store the marker position for click detection
                            markerPositions.add(EventMarkerPosition(event, x, y, matchingGraphData))
                        }
                    }
                }
            }
        }
    }
    
    return markerPositions
}

private fun DrawScope.drawAxes(
    padding: Float,
    graphWidth: Float,
    graphHeight: Float
) {
    // X-axis
    drawLine(
        color = Color.Black,
        start = Offset(padding, size.height - padding),
        end = Offset(size.width - padding, size.height - padding),
        strokeWidth = 2f
    )
    
    // Y-axis
    drawLine(
        color = Color.Black,
        start = Offset(padding, padding),
        end = Offset(padding, size.height - padding),
        strokeWidth = 2f
    )
}

private fun DrawScope.drawAxisLabels(
    padding: Float,
    graphWidth: Float,
    graphHeight: Float,
    startTime: Date,
    endTime: Date,
    graphDataList: List<GraphData>
) {
    val paint = Paint().asFrameworkPaint().apply {
        isAntiAlias = true
        textSize = 28f
        color = android.graphics.Color.WHITE
    }
    
    // Time labels (X-axis) - use visible time window for zoom-aware labels
    val visibleTimeRange = endTime.time - startTime.time
    val daysBack = (visibleTimeRange / (24 * 60 * 60 * 1000)).toInt()
    val timeFormat = when {
        daysBack > 30 -> SimpleDateFormat("MMM dd", Locale.getDefault())
        daysBack > 7 -> SimpleDateFormat("MMM dd", Locale.getDefault())
        else -> SimpleDateFormat("HH:mm", Locale.getDefault())
    }
    
    val timeLabels = generateTimeLabels(startTime, endTime, daysBack)
    
    // Draw light-grey vertical grid lines for labels first
    drawLabelGridLines(timeLabels, startTime, endTime, padding, graphWidth, graphHeight)
    
    timeLabels.forEach { time ->
        val x = padding + ((time.time - startTime.time).toFloat() / visibleTimeRange.toFloat()) * graphWidth
        val label = timeFormat.format(time)
        
        drawContext.canvas.nativeCanvas.drawText(
            label,
            x - paint.measureText(label) / 2,
            size.height - padding + 35f,
            paint
        )
    }
    
    // Y-axis labels only for DECAYING and METRIC templates (color-coded)
    val filteredGraphData = graphDataList.filter { graphData ->
        when (graphData.item) {
            is GraphableItem.MetricTemplate -> true // METRIC type
            is GraphableItem.Drug -> {
                // DECAYING type - check if any template is DECAYING
                graphData.item.templates.any { it.templateType == TemplateType.DECAYING }
            }
            is GraphableItem.GeneralTemplate -> false // Skip GENERAL type
            is GraphableItem.DecayingTemplate -> false // Skip DECAYING template items (they don't have their own y-axis)
        }
    }
    
    filteredGraphData.forEachIndexed { index, graphData ->
        paint.color = android.graphics.Color.argb(
            255,
            (graphData.item.color.red * 255).toInt(),
            (graphData.item.color.green * 255).toInt(),
            (graphData.item.color.blue * 255).toInt()
        )
        
        val yLabels = generateYLabels(graphData.minValue, graphData.maxValue)
        val valueRange = graphData.maxValue - graphData.minValue
        
        val axisX = when (index) {
            0 -> padding
            1 -> size.width - padding
            else -> padding + 60f + (index - 2) * 50f
        }
        
        yLabels.forEach { value ->
            val y = size.height - padding - ((value - graphData.minValue).toFloat() / valueRange.toFloat()) * graphHeight
            val label = if (value >= 1) "%.1f".format(value) else "%.2f".format(value)
            
            val labelX = when (index) {
                0 -> axisX - paint.measureText(label) - 10f
                1 -> axisX + 10f
                else -> axisX - paint.measureText(label) / 2
            }
            
            drawContext.canvas.nativeCanvas.drawText(
                label,
                labelX,
                y + paint.textSize / 3,
                paint
            )
        }
        
        // Unit label
        drawContext.canvas.nativeCanvas.save()
        val unitX = when (index) {
            0 -> 20f
            1 -> size.width - 20f
            else -> 20f + (index - 2) * 40f
        }
        drawContext.canvas.nativeCanvas.rotate(-90f, unitX, size.height / 2)
        drawContext.canvas.nativeCanvas.drawText(
            graphData.unit,
            unitX - paint.measureText(graphData.unit) / 2,
            size.height / 2,
            paint
        )
        drawContext.canvas.nativeCanvas.restore()
    }
    
    // Time axis label
    paint.color = android.graphics.Color.WHITE
    drawContext.canvas.nativeCanvas.drawText(
        "Time",
        size.width / 2 - paint.measureText("Time") / 2,
        size.height - 10f,
        paint
    )
}

private fun DrawScope.drawLabelGridLines(
    timeLabels: List<Date>,
    startTime: Date,
    endTime: Date,
    padding: Float,
    graphWidth: Float,
    graphHeight: Float
) {
    val labelGridColor = Color.Gray.copy(alpha = 0.3f) // Brighter than the current blue grid lines
    val visibleTimeRange = endTime.time - startTime.time
    
    timeLabels.forEach { time ->
        val x = padding + ((time.time - startTime.time).toFloat() / visibleTimeRange.toFloat()) * graphWidth
        drawLine(
            color = labelGridColor,
            start = Offset(x, padding),
            end = Offset(x, size.height - padding),
            strokeWidth = 1.5f
        )
    }
}

private fun generateTimeLabels(startTime: Date, endTime: Date, daysBack: Int): List<Date> {
    val labels = mutableListOf<Date>()
    val calendar = Calendar.getInstance()
    val visibleTimeRangeHours = (endTime.time - startTime.time) / (1000.0 * 60.0 * 60.0)
    
    when {
        visibleTimeRangeHours > 72 -> {
            // More than 3 days: Use midnight markers only (stick with days)
            calendar.time = startTime
            calendar.set(Calendar.HOUR_OF_DAY, 0)
            calendar.set(Calendar.MINUTE, 0)
            calendar.set(Calendar.SECOND, 0)
            calendar.set(Calendar.MILLISECOND, 0)
            calendar.add(Calendar.DAY_OF_YEAR, 1) // Start from first midnight after start
            
            // Calculate day interval to get 4-6 evenly distributed labels
            val visibleDays = (visibleTimeRangeHours / 24).toInt()
            val dayInterval = when {
                visibleDays > 30 -> maxOf(1, visibleDays / 5) // For very long ranges
                visibleDays > 14 -> maxOf(1, visibleDays / 4) // For medium ranges
                else -> maxOf(1, visibleDays / 6) // For shorter ranges
            }
            
            while (calendar.time.before(endTime) && labels.size < 6) {
                labels.add(Date(calendar.timeInMillis))
                calendar.add(Calendar.DAY_OF_YEAR, dayInterval)
            }
        }
        else -> {
            // 72 hours or less: Use evenly distributed hour markers
            val totalHours = visibleTimeRangeHours.toInt()
            
            // Choose division that gives us 3-6 labels
            val hourInterval = when {
                totalHours >= 48 -> 12 // Every 12 hours (2 per day)
                totalHours >= 36 -> 8  // Every 8 hours (3 per day)
                totalHours >= 24 -> 6  // Every 6 hours (4 per day)
                totalHours >= 16 -> 4  // Every 4 hours (6 per day)
                totalHours >= 8 -> 3   // Every 3 hours (8 per day)
                totalHours >= 4 -> 2   // Every 2 hours (12 per day)
                else -> 1              // Every hour (24 per day)
            }
            
            // Start from the beginning of the visible time range
            calendar.time = startTime
            calendar.set(Calendar.MINUTE, 0)
            calendar.set(Calendar.SECOND, 0)
            calendar.set(Calendar.MILLISECOND, 0)
            
            // Find the first evenly distributed hour mark
            val startHour = calendar.get(Calendar.HOUR_OF_DAY)
            val firstHour = (startHour / hourInterval) * hourInterval
            calendar.set(Calendar.HOUR_OF_DAY, firstHour)
            
            // If this is before our start time, move to the next interval
            if (calendar.time.before(startTime)) {
                calendar.add(Calendar.HOUR_OF_DAY, hourInterval)
            }
            
            // Generate evenly distributed labels
            while (calendar.time.before(endTime) && labels.size < 6) {
                if (calendar.time.after(startTime)) {
                    labels.add(Date(calendar.timeInMillis))
                }
                calendar.add(Calendar.HOUR_OF_DAY, hourInterval)
            }
            
            // If we still don't have enough labels, try a smaller interval
            if (labels.size < 3 && hourInterval > 1) {
                labels.clear()
                val smallerInterval = maxOf(1, hourInterval / 2)
                
                calendar.time = startTime
                calendar.set(Calendar.MINUTE, 0)
                calendar.set(Calendar.SECOND, 0)
                calendar.set(Calendar.MILLISECOND, 0)
                
                val startHourSmaller = calendar.get(Calendar.HOUR_OF_DAY)
                val firstHourSmaller = (startHourSmaller / smallerInterval) * smallerInterval
                calendar.set(Calendar.HOUR_OF_DAY, firstHourSmaller)
                
                if (calendar.time.before(startTime)) {
                    calendar.add(Calendar.HOUR_OF_DAY, smallerInterval)
                }
                
                while (calendar.time.before(endTime) && labels.size < 6) {
                    if (calendar.time.after(startTime)) {
                        labels.add(Date(calendar.timeInMillis))
                    }
                    calendar.add(Calendar.HOUR_OF_DAY, smallerInterval)
                }
            }
        }
    }
    
    // Ensure we have at least 3 labels
    val sortedLabels = labels.sortedBy { it.time }.distinctBy { it.time }
    
    // If we still have too few labels, add evenly distributed fallback labels
    if (sortedLabels.size < 3) {
        val fallbackLabels = mutableListOf<Date>()
        val timeRange = endTime.time - startTime.time
        
        // Create 4 evenly distributed labels
        for (i in 0..3) {
            val time = startTime.time + (timeRange * i / 3)
            fallbackLabels.add(Date(time))
        }
        
        return fallbackLabels.distinctBy { it.time }.sortedBy { it.time }
    }
    
    return sortedLabels.take(6) // Maximum 6 labels
}

private fun generateYLabels(minValue: Double, maxValue: Double): List<Double> {
    val range = maxValue - minValue
    val stepCount = 5
    val rawStep = range / stepCount
    
    val magnitude = 10.0.pow(floor(log10(rawStep)))
    val normalizedStep = rawStep / magnitude
    
    val niceStep = when {
        normalizedStep <= 1.0 -> 1.0
        normalizedStep <= 2.0 -> 2.0
        normalizedStep <= 5.0 -> 5.0
        else -> 10.0
    } * magnitude
    
    val labels = mutableListOf<Double>()
    var value = ceil(minValue / niceStep) * niceStep
    
    while (value <= maxValue && labels.size < 6) {
        labels.add(value)
        value += niceStep
    }
    
    return labels
}

/**
 * Find the nearest event to a click position using the actual rendered marker positions.
 * This ensures click detection matches exactly where the markers are drawn.
 */
private fun findNearestEventFromPositions(
    clickOffset: Offset,
    markerPositions: List<EventMarkerPosition>,
    clickThreshold: Float = 50f
): CalendarEvent? {
    var nearestEvent: CalendarEvent? = null
    var nearestDistance = Float.MAX_VALUE
    
    markerPositions.forEach { position ->
        val dx = clickOffset.x - position.x
        val dy = clickOffset.y - position.y
        val distance = sqrt(dx * dx + dy * dy)
        
        if (distance < clickThreshold && distance < nearestDistance) {
            nearestDistance = distance
            nearestEvent = position.event
        }
    }
    
    return nearestEvent
}

/**
 * Legacy function - kept for compatibility but no longer used.
 * Click detection now uses findNearestEventFromPositions with actual marker positions.
 */
private fun findNearestEvent(
    clickOffset: Offset,
    graphDataList: List<GraphData>,
    events: List<CalendarEvent>,
    startTime: Date,
    endTime: Date,
    canvasSize: androidx.compose.ui.geometry.Size,
    zoomLevel: Float = 1f,
    panOffset: Float = 0f,
    padding: Float = 80f
): CalendarEvent? {
    val graphWidth = canvasSize.width - 2 * padding
    val graphHeight = canvasSize.height - 2 * padding
    val originalTimeRange = endTime.time - startTime.time
    val clickThreshold = 50f // pixels
    
    // Safety check: ensure we have valid time range
    if (originalTimeRange <= 0) return null
    
    // Apply zoom and pan transformations (same as in drawAdvancedGraph)
    val safeZoomLevel = zoomLevel.coerceIn(1f, 20f)
    val zoomedTimeRange = (originalTimeRange / safeZoomLevel).toLong()
    
    // Calculate pan bounds - when zoomed in, we can pan within the extra time
    val maxPanTime = if (safeZoomLevel > 1f) {
        (originalTimeRange - zoomedTimeRange).toFloat()
    } else {
        0f
    }
    
    val clampedPanOffset = if (maxPanTime > 0) {
        panOffset.coerceIn(-maxPanTime, maxPanTime)
    } else {
        0f
    }
    
    // Calculate the visible time window based on zoom and pan
    val visibleStartTime = startTime.time + clampedPanOffset.toLong()
    val visibleEndTime = visibleStartTime + zoomedTimeRange
    
    // Additional safety check: ensure visible times are valid
    if (visibleStartTime >= visibleEndTime || zoomedTimeRange <= 0) return null
    
    var nearestEvent: CalendarEvent? = null
    var nearestDistance = Float.MAX_VALUE
    
    events.forEach { event ->
        if (event.startTime.time >= visibleStartTime && event.startTime.time <= visibleEndTime) {
            // Find all graph items that match this event
            // Note: graphDataList already contains only selected items, so no need to check isSelected again
            val matchingGraphItems = graphDataList.filter { graphData ->
                when (graphData.item) {
                    is GraphableItem.Drug -> {
                        graphData.item.templates.any { template ->
                            matchesTemplate(event, template)
                        }
                    }
                    is GraphableItem.MetricTemplate -> {
                        matchesTemplate(event, graphData.item.template)
                    }
                    is GraphableItem.GeneralTemplate -> {
                        matchesTemplate(event, graphData.item.template)
                    }
                    is GraphableItem.DecayingTemplate -> {
                        matchesTemplate(event, graphData.item.template)
                    }
                }
            }
            
            // For each matching graph item, calculate the 2D distance to the event marker
            matchingGraphItems.forEach { graphData ->
                val eventX = padding + ((event.startTime.time - visibleStartTime).toFloat() / zoomedTimeRange.toFloat()) * graphWidth
                
                // Calculate the Y position using the same logic as drawEventMarkers
                val eventY = run {
                    // Position at calculated height for this item type
                    val eventPoint = graphData.points.find { it.event == event }
                    if (eventPoint != null) {
                        val valueRange = graphData.maxValue - graphData.minValue
                        if (valueRange > 0) {
                            canvasSize.height - padding - ((eventPoint.value - graphData.minValue).toFloat() / valueRange.toFloat()) * graphHeight
                        } else {
                            // For metric templates with single values, position in middle
                            canvasSize.height - padding - graphHeight / 2
                        }
                    } else {
                        // For general templates or missing data points, position near bottom but not at the very bottom
                        when (graphData.item) {
                            is GraphableItem.MetricTemplate -> {
                                // For metric templates, try to find the latest value and position accordingly
                                val latestPoint = graphData.points.maxByOrNull { it.time.time }
                                if (latestPoint != null) {
                                    val valueRange = graphData.maxValue - graphData.minValue
                                    if (valueRange > 0) {
                                        canvasSize.height - padding - ((latestPoint.value - graphData.minValue).toFloat() / valueRange.toFloat()) * graphHeight
                                    } else {
                                        canvasSize.height - padding - graphHeight / 2
                                    }
                                } else {
                                    canvasSize.height - padding - graphHeight / 2
                                }
                            }
                            is GraphableItem.GeneralTemplate -> {
                                // General templates appear as timeline events at bottom
                                canvasSize.height - padding - 30f
                            }
                            else -> {
                                canvasSize.height - padding - 30f
                            }
                        }
                    }
                }
                
                // Calculate 2D distance from click to event marker
                val dx = clickOffset.x - eventX
                val dy = clickOffset.y - eventY
                val distance = sqrt(dx * dx + dy * dy)
                
                if (distance < clickThreshold && distance < nearestDistance) {
                    nearestDistance = distance
                    nearestEvent = event
                }
            }
        }
    }
    
    return nearestEvent
}

private fun calculateEventData(event: CalendarEvent, graphDataList: List<GraphData>): EventClickData? {
    // Find the matching graph data for this event
    val matchingGraphData = graphDataList.find { graphData ->
        when (graphData.item) {
            is GraphableItem.Drug -> {
                graphData.item.templates.any { template ->
                    matchesTemplate(event, template)
                }
            }
            is GraphableItem.MetricTemplate -> {
                matchesTemplate(event, graphData.item.template)
            }
            is GraphableItem.GeneralTemplate -> {
                matchesTemplate(event, graphData.item.template)
            }
            is GraphableItem.DecayingTemplate -> {
                matchesTemplate(event, graphData.item.template)
            }
        }
    } ?: return null
    
    // Get the template to calculate dosage with unit
    val template = when (matchingGraphData.item) {
        is GraphableItem.Drug -> {
            // Find the specific template that matches this event's title
            matchingGraphData.item.templates.find { template ->
                matchesTemplate(event, template)
            } ?: run {
                // Fallback: if no exact match, try to find by event content
                val eventTitle = event.title.lowercase()
                matchingGraphData.item.templates.find { template ->
                    val templateName = template.name.lowercase()
                    when {
                        eventTitle.contains("beer") && templateName.contains("beer") -> true
                        eventTitle.contains("wine") && templateName.contains("wine") -> true
                        eventTitle.contains("alcohol") && (templateName.contains("beer") || templateName.contains("wine")) -> true
                        else -> false
                    }
                } ?: matchingGraphData.item.templates.firstOrNull() // Last resort: use first template
            }
        }
        is GraphableItem.MetricTemplate -> matchingGraphData.item.template
        is GraphableItem.GeneralTemplate -> matchingGraphData.item.template
        is GraphableItem.DecayingTemplate -> matchingGraphData.item.template
    } ?: return null
    
    // Calculate dosage with unit
    val firstDrug = template.drugs.firstOrNull()
    val factor = firstDrug?.factor ?: 1.0
    val unit = firstDrug?.unit ?: template.metricUnit
    val dosageWithUnit = "${(event.dosage * factor).let { 
        if (it == it.toInt().toDouble()) it.toInt().toString() else "%.1f".format(it) 
    }}$unit"
    
    // Find before and after values
    val eventTime = event.startTime.time
    val points = matchingGraphData.points.sortedBy { it.time.time }
    
    val beforePoint = points.lastOrNull { it.time.time < eventTime }
    
    // Handle different template types appropriately
    val (beforeValue, afterValue) = when (matchingGraphData.item) {
        is GraphableItem.Drug -> {
            // For drugs, calculate before/after concentrations
            val beforeConcentration = beforePoint?.value ?: 0.0
            val beforeVal = "${if (beforeConcentration >= 1) "%.1f".format(beforeConcentration) else "%.2f".format(beforeConcentration)}${matchingGraphData.unit}"
            
            // Calculate the contribution of just this single event and add it to the existing level
            val eventContribution = calculateSingleEventContribution(
                event = event,
                template = template,
                drugType = matchingGraphData.item.drugType,
                templates = matchingGraphData.item.templates
            )
            
            val afterConcentration = beforeConcentration + eventContribution
            val afterVal = "${if (afterConcentration >= 1) "%.1f".format(afterConcentration) else "%.2f".format(afterConcentration)}${matchingGraphData.unit}"
            
            Pair(beforeVal, afterVal)
        }
        is GraphableItem.MetricTemplate -> {
            // For metric templates, show the actual measured value from the event
            val eventPoint = points.find { it.event == event }
            val measuredValue = eventPoint?.value ?: event.dosage
            val templateUnit = template.metricUnit ?: unit ?: ""
            val formattedValue = "${if (measuredValue >= 1) "%.1f".format(measuredValue) else "%.2f".format(measuredValue)}$templateUnit"
            
            // For metrics, "before" and "after" don't make sense - show the measured value
            Pair("N/A", formattedValue)
        }
        is GraphableItem.GeneralTemplate -> {
            // General templates don't have concentration values
            Pair("N/A", "N/A")
        }
        is GraphableItem.DecayingTemplate -> {
            // Decaying templates don't have concentration values (same as general)
            Pair("N/A", "N/A")
        }
    }
    
    // Determine metric name based on template type
    val metricName = when {
        template.name.contains("alcohol", ignoreCase = true) ||
        template.name.contains("beer", ignoreCase = true) ||
        template.name.contains("wine", ignoreCase = true) -> "Alc. concentration"
        template.name.contains("caffeine", ignoreCase = true) ||
        template.name.contains("coffee", ignoreCase = true) -> "Caffeine level"
        template.name.contains("nicotine", ignoreCase = true) -> "Nicotine level"
        else -> template.name
    }
    
    // Find all events of the same type to calculate hours since previous and until next
    val allEvents = getAllEventsFromGraphData(graphDataList)
    val sameTypeEvents = when (matchingGraphData.item) {
        is GraphableItem.Drug -> {
            allEvents.filter { e ->
                matchingGraphData.item.templates.any { template ->
                    matchesTemplate(e, template)
                }
            }
        }
        is GraphableItem.MetricTemplate -> {
            allEvents.filter { e ->
                matchesTemplate(e, matchingGraphData.item.template)
            }
        }
        is GraphableItem.GeneralTemplate -> {
            allEvents.filter { e ->
                matchesTemplate(e, matchingGraphData.item.template)
            }
        }
        is GraphableItem.DecayingTemplate -> {
            allEvents.filter { e ->
                matchesTemplate(e, matchingGraphData.item.template)
            }
        }
    }.sortedBy { it.startTime.time }
    
    // Calculate hours since previous log
    val previousEvent = sameTypeEvents.lastOrNull { e -> e.startTime.time < event.startTime.time }
    val hoursSincePrevious = if (previousEvent != null) {
        val diffMillis = event.startTime.time - previousEvent.startTime.time
        val diffHours = diffMillis / (1000.0 * 60.0 * 60.0)
        "%.1f".format(diffHours) + "h"
    } else {
        "-"
    }
    
    // Calculate hours until next log
    val nextEvent = sameTypeEvents.firstOrNull { e -> e.startTime.time > event.startTime.time }
    val hoursUntilNext = if (nextEvent != null) {
        val diffMillis = nextEvent.startTime.time - event.startTime.time
        val diffHours = diffMillis / (1000.0 * 60.0 * 60.0)
        "%.1f".format(diffHours) + "h"
    } else {
        "-"
    }
    
    // Calculate average value based on item type
    val averageValue = when (matchingGraphData.item) {
        is GraphableItem.Drug -> {
            // For drugs: average concentration in the viewed time window
            val values = points.map { it.value }
            if (values.isNotEmpty()) {
                val avg = values.average()
                "${if (avg >= 1) "%.1f".format(avg) else "%.2f".format(avg)}${matchingGraphData.unit}"
            } else {
                "-"
            }
        }
        is GraphableItem.MetricTemplate -> {
            // For metrics: average metric value in the viewed time window
            val eventPoints = points.filter { it.event != null }
            if (eventPoints.isNotEmpty()) {
                val avg = eventPoints.map { it.value }.average()
                val templateUnit = template.metricUnit ?: unit ?: ""
                "${if (avg >= 1) "%.1f".format(avg) else "%.2f".format(avg)}$templateUnit"
            } else {
                "-"
            }
        }
        is GraphableItem.GeneralTemplate -> {
            // For general events: average time between logs
            if (sameTypeEvents.size >= 2) {
                val timeDiffs = mutableListOf<Long>()
                for (i in 0 until sameTypeEvents.size - 1) {
                    timeDiffs.add(sameTypeEvents[i + 1].startTime.time - sameTypeEvents[i].startTime.time)
                }
                val avgMillis = timeDiffs.average()
                val avgHours = avgMillis / (1000.0 * 60.0 * 60.0)
                "%.1f".format(avgHours) + "h"
            } else {
                "-"
            }
        }
        is GraphableItem.DecayingTemplate -> {
            // For decaying templates: average time between logs (same as general)
            if (sameTypeEvents.size >= 2) {
                val timeDiffs = mutableListOf<Long>()
                for (i in 0 until sameTypeEvents.size - 1) {
                    timeDiffs.add(sameTypeEvents[i + 1].startTime.time - sameTypeEvents[i].startTime.time)
                }
                val avgMillis = timeDiffs.average()
                val avgHours = avgMillis / (1000.0 * 60.0 * 60.0)
                "%.1f".format(avgHours) + "h"
            } else {
                "-"
            }
        }
    }
    
    return EventClickData(
        dosageWithUnit = dosageWithUnit,
        beforeValue = beforeValue,
        afterValue = afterValue,
        metricName = metricName,
        hoursSincePrevious = hoursSincePrevious,
        hoursUntilNext = hoursUntilNext,
        averageValue = averageValue
    )
}

/**
 * Calculate the concentration contribution of a single event
 */
private fun calculateSingleEventContribution(
    event: CalendarEvent,
    template: Template,
    drugType: String,
    templates: List<Template>
): Double {
    // Calculate the effective dose for this single event
    val dosage = event.dosage
    val firstDrug = template.drugs.firstOrNull()
    val factor = firstDrug?.factor ?: 1.0
    val effectiveDose = dosage * factor
    
    // For popup purposes, we just return the immediate contribution
    // (the peak concentration this event adds)
    return effectiveDose
}

/**
 * Helper function to extract all events from graph data
 */
private fun getAllEventsFromGraphData(graphDataList: List<GraphData>): List<CalendarEvent> {
    return graphDataList.flatMap { graphData ->
        graphData.points.mapNotNull { it.event }
    }.distinctBy { "${it.title}_${it.startTime.time}" } // Remove duplicates
}
