package de.ntdote.medicalcalendarlog.ui

import android.app.Activity
import android.appwidget.AppWidgetManager
import android.content.Intent
import android.os.Bundle
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import android.view.LayoutInflater
import android.widget.RemoteViews
import androidx.compose.foundation.background
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.*
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.drawBehind
import androidx.compose.ui.geometry.Offset
import androidx.compose.ui.geometry.Size
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.compose.ui.viewinterop.AndroidView
import androidx.lifecycle.lifecycleScope
import de.ntdote.medicalcalendarlog.R
import de.ntdote.medicalcalendarlog.data.Template
import de.ntdote.medicalcalendarlog.data.TemplateType
import de.ntdote.medicalcalendarlog.data.WidgetConfig
import de.ntdote.medicalcalendarlog.repository.PreferencesRepository
import de.ntdote.medicalcalendarlog.repository.WidgetRepository
import de.ntdote.medicalcalendarlog.service.TemplateWidgetProvider
import de.ntdote.medicalcalendarlog.ui.theme.MedicalCalendarlogTheme
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.launch

/**
 * Activity for configuring widgets - allows user to select which template to display
 */
class WidgetConfigurationActivity : ComponentActivity() {

    private var appWidgetId = AppWidgetManager.INVALID_APPWIDGET_ID
    private var isEditMode = false
    private lateinit var widgetRepository: WidgetRepository
    private lateinit var preferencesRepository: PreferencesRepository

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        // Set the result to CANCELED. This will cause the widget host to cancel
        // out of the widget placement if the user presses the back button.
        setResult(RESULT_CANCELED)

        // Find the widget id from the intent
        appWidgetId = intent?.extras?.getInt(
            AppWidgetManager.EXTRA_APPWIDGET_ID,
            AppWidgetManager.INVALID_APPWIDGET_ID
        ) ?: AppWidgetManager.INVALID_APPWIDGET_ID
        
        // Check if in edit mode
        isEditMode = intent?.getBooleanExtra(WidgetMenuActivity.EXTRA_EDIT_MODE, false) ?: false

        // If this activity was started with an intent without an app widget ID, finish with an error
        if (appWidgetId == AppWidgetManager.INVALID_APPWIDGET_ID) {
            finish()
            return
        }

        widgetRepository = WidgetRepository(this)
        preferencesRepository = PreferencesRepository(this)

        setContent {
            MedicalCalendarlogTheme {
                var selectedTemplate by remember { mutableStateOf<Template?>(null) }
                var existingConfig by remember { mutableStateOf<de.ntdote.medicalcalendarlog.data.WidgetConfig?>(null) }
                
                // Load existing config and pre-select template if in edit mode
                LaunchedEffect(isEditMode) {
                    if (isEditMode) {
                        existingConfig = widgetRepository.getWidgetConfigSync(appWidgetId)
                        if (existingConfig != null) {
                            val templates = preferencesRepository.templates.first()
                            selectedTemplate = templates.find { it.id == existingConfig!!.templateId }
                        }
                    }
                }
                
                if (selectedTemplate == null) {
                    TemplateSelectionScreen(
                        onTemplateSelected = { template ->
                            selectedTemplate = template
                        },
                        onCancel = {
                            finish()
                        }
                    )
                } else {
                    WidgetCustomizationScreen(
                        template = selectedTemplate!!,
                        existingConfig = existingConfig,
                        isEditMode = isEditMode,
                        onConfirm = { config ->
                            configureWidget(config)
                        },
                        onBack = {
                            selectedTemplate = null
                        }
                    )
                }
            }
        }
    }

    @OptIn(ExperimentalMaterial3Api::class)
    @Composable
    private fun TemplateSelectionScreen(
        onTemplateSelected: (Template) -> Unit,
        onCancel: () -> Unit
    ) {
        var templates by remember { mutableStateOf<List<Template>>(emptyList()) }

        LaunchedEffect(Unit) {
            // Filter to only show enabled templates
            templates = preferencesRepository.templates.first().filter { it.enabled }
        }

        Scaffold(
            topBar = {
                TopAppBar(
                    title = { Text("Select Template for Widget") },
                    navigationIcon = {
                        TextButton(onClick = onCancel) {
                            Text("Cancel")
                        }
                    }
                )
            }
        ) { paddingValues ->
            LazyColumn(
                modifier = Modifier
                    .fillMaxSize()
                    .padding(paddingValues)
                    .padding(16.dp),
                verticalArrangement = Arrangement.spacedBy(8.dp)
            ) {
                items(templates) { template ->
                    Card(
                        modifier = Modifier
                            .fillMaxWidth()
                            .clickable { onTemplateSelected(template) },
                        elevation = CardDefaults.cardElevation(defaultElevation = 2.dp)
                    ) {
                        Column(
                            modifier = Modifier
                                .fillMaxWidth()
                                .padding(16.dp)
                        ) {
                            Text(
                                text = template.name,
                                style = MaterialTheme.typography.titleMedium
                            )
                            
                            Spacer(modifier = Modifier.height(4.dp))
                            
                            Text(
                                text = when {
                                    template.templateType == TemplateType.DECAYING && template.drugs.isNotEmpty() -> {
                                        // Display first drug info, or "Multiple drugs" if more than one
                                        if (template.drugs.size == 1) {
                                            "${template.drugs[0].drugType} • ${template.drugs[0].unit}"
                                        } else {
                                            "Multiple drugs (${template.drugs.size})"
                                        }
                                    }
                                    template.templateType == TemplateType.METRIC ->
                                        if (template.metricUnit.isNotEmpty()) "Metric • ${template.metricUnit}" else "Metric"
                                    else -> "General"
                                },
                                style = MaterialTheme.typography.bodySmall,
                                color = MaterialTheme.colorScheme.onSurfaceVariant
                            )
                        }
                    }
                }

                if (templates.isEmpty()) {
                    item {
                        Card(
                            modifier = Modifier.fillMaxWidth(),
                            colors = CardDefaults.cardColors(
                                containerColor = MaterialTheme.colorScheme.errorContainer
                            )
                        ) {
                            Text(
                                text = "No templates available. Please create templates in the app first.",
                                modifier = Modifier.padding(16.dp),
                                color = MaterialTheme.colorScheme.onErrorContainer
                            )
                        }
                    }
                }
            }
        }
    }

    @OptIn(ExperimentalMaterial3Api::class)
    @Composable
    private fun WidgetCustomizationScreen(
        template: Template,
        existingConfig: WidgetConfig? = null,
        isEditMode: Boolean = false,
        onConfirm: (WidgetConfig) -> Unit,
        onBack: () -> Unit
    ) {
        var showTime by remember { mutableStateOf(existingConfig?.showTime ?: true) }
        var showTemplateName by remember { mutableStateOf(existingConfig?.showTemplateName ?: true) }
        var showValue by remember { mutableStateOf(existingConfig?.showValue ?: false) }
        var compactView by remember { mutableStateOf(existingConfig?.compactTime ?: true) }
        var nameClickQuickLog by remember { mutableStateOf(existingConfig?.nameClickQuickLog ?: true) }
        var usePercentage by remember { mutableStateOf(existingConfig?.usePercentage ?: false) }
        var opacity by remember { mutableStateOf(existingConfig?.transparency?.toFloat() ?: 100f) }
        var selectedDrugIndex by remember { mutableStateOf(existingConfig?.selectedDrugIndex ?: 0) }
        var expandedDrugDropdown by remember { mutableStateOf(false) }

        // Determine if value/unit options should be shown based on template type
        val supportsValueDisplay = template.templateType == TemplateType.DECAYING ||
                                   template.templateType == TemplateType.METRIC
        
        // Check if template has multiple drugs
        val hasMultipleDrugs = template.templateType == TemplateType.DECAYING && template.drugs.size > 1
        
        // Determine if percentage option should be shown (only for templates with reminders)
        val hasTimeReminder = template.reminderEnabled && template.reminderThresholdMillis != null
        val hasConcentrationReminder = template.templateType == TemplateType.DECAYING && 
                                       template.drugs.any { it.concentrationReminderEnabled && it.concentrationReminderThreshold != null }
        val supportsPercentage = hasTimeReminder || hasConcentrationReminder

        Scaffold(
            topBar = {
                TopAppBar(
                    title = {
                        Row(
                            modifier = Modifier.fillMaxWidth(),
                            horizontalArrangement = Arrangement.SpaceBetween,
                            verticalAlignment = Alignment.CenterVertically
                        ) {
                            TextButton(onClick = onBack) {
                                Text("Back")
                            }
                            Text(
                                text = "Customize Widget",
                                modifier = Modifier.padding(end = 16.dp)
                            )
                        }
                    }
                )
            }
        ) { paddingValues ->
            Column(
                modifier = Modifier
                    .fillMaxSize()
                    .padding(paddingValues)
                    .verticalScroll(rememberScrollState())
                    .padding(16.dp),
                verticalArrangement = Arrangement.spacedBy(12.dp)
            ) {
                // Preview
                Text(
                    text = "Preview",
                    style = MaterialTheme.typography.titleMedium,
                    fontWeight = FontWeight.Bold
                )
                
                Box(
                    modifier = Modifier
                        .fillMaxWidth()
                        .drawBehind {
                            // Create checkerboard pattern for transparency evaluation
                            val squareSize = 16.dp.toPx()
                            val darkGray = Color(0xFF666666)
                            val lightGray = Color(0xFFAAAAAA)
                            
                            var y = 0f
                            var rowIndex = 0
                            while (y < size.height) {
                                var x = 0f
                                var colIndex = 0
                                while (x < size.width) {
                                    val color = if ((rowIndex + colIndex) % 2 == 0) darkGray else lightGray
                                    drawRect(
                                        color = color,
                                        topLeft = Offset(x, y),
                                        size = Size(squareSize, squareSize)
                                    )
                                    x += squareSize
                                    colIndex++
                                }
                                y += squareSize
                                rowIndex++
                            }
                        }
                        .padding(8.dp)
                ) {
                    // Key usePercentage to force recomposition when it changes
                    key(usePercentage) {
                        WidgetPreview(
                            template = template,
                            showTime = showTime,
                            showTemplateName = showTemplateName,
                            showValue = showValue,
                            opacity = opacity.toInt(),
                            compactView = compactView,
                            usePercentage = usePercentage,
                            selectedDrugIndex = selectedDrugIndex
                        )
                    }
                }
                
                //HorizontalDivider()
                
                // Customization options
                Text(
                    text = "Display Options",
                    style = MaterialTheme.typography.titleMedium,
                    fontWeight = FontWeight.Bold
                )
                
                Column(
                    modifier = Modifier.fillMaxWidth(),
                    verticalArrangement = Arrangement.spacedBy(4.dp)
                ) {
                    Row(
                        modifier = Modifier.fillMaxWidth(),
                        verticalAlignment = Alignment.CenterVertically
                    ) {
                        Checkbox(
                            checked = showTime,
                            onCheckedChange = { showTime = it }
                        )
                        Text("Show time")
                    }
                    
                    Row(
                        modifier = Modifier.fillMaxWidth(),
                        verticalAlignment = Alignment.CenterVertically
                    ) {
                        Checkbox(
                            checked = showTemplateName,
                            onCheckedChange = { showTemplateName = it }
                        )
                        Text("Show template name")
                    }
                    
                    Row(
                        modifier = Modifier.fillMaxWidth(),
                        verticalAlignment = Alignment.CenterVertically
                    ) {
                        Checkbox(
                            checked = nameClickQuickLog,
                            onCheckedChange = { nameClickQuickLog = it },
                            enabled = showTemplateName
                        )
                        Text(
                            text = "Name click = Quick log",
                            color = if (showTemplateName) {
                                MaterialTheme.colorScheme.onSurface
                            } else {
                                MaterialTheme.colorScheme.onSurface.copy(alpha = 0.38f)
                            }
                        )
                    }
                    
                    // Only show value option for templates that support it
                    if (supportsValueDisplay) {
                        Row(
                            modifier = Modifier.fillMaxWidth(),
                            verticalAlignment = Alignment.CenterVertically
                        ) {
                            Checkbox(
                                checked = showValue,
                                onCheckedChange = { showValue = it }
                            )
                            Text("Show concentration/value")
                        }
                        
                        // Drug selector for multi-drug templates
                        if (hasMultipleDrugs && showValue) {
                            Spacer(modifier = Modifier.height(8.dp))
                            ExposedDropdownMenuBox(
                                expanded = expandedDrugDropdown,
                                onExpandedChange = { expandedDrugDropdown = it },
                                modifier = Modifier
                                    .fillMaxWidth()
                                    .padding(start = 48.dp)
                            ) {
                                OutlinedTextField(
                                    value = template.drugs.getOrNull(selectedDrugIndex)?.drugType ?: "Select drug",
                                    onValueChange = {},
                                    readOnly = true,
                                    label = { Text("Drug to display") },
                                    trailingIcon = { ExposedDropdownMenuDefaults.TrailingIcon(expanded = expandedDrugDropdown) },
                                    modifier = Modifier
                                        .menuAnchor(MenuAnchorType.PrimaryNotEditable)
                                        .fillMaxWidth()
                                )
                                ExposedDropdownMenu(
                                    expanded = expandedDrugDropdown,
                                    onDismissRequest = { expandedDrugDropdown = false }
                                ) {
                                    // Individual drug options (0-based index)
                                    template.drugs.forEachIndexed { index, drug ->
                                        DropdownMenuItem(
                                            text = { Text(drug.drugType) },
                                            onClick = {
                                                selectedDrugIndex = index
                                                expandedDrugDropdown = false
                                            }
                                        )
                                    }
                                }
                            }
                        }
                    }
                    
                    Row(
                        modifier = Modifier.fillMaxWidth(),
                        verticalAlignment = Alignment.CenterVertically
                    ) {
                        Checkbox(
                            checked = compactView,
                            onCheckedChange = { compactView = it }
                        )
                        Column(modifier = Modifier.padding(start = 0.dp)) {
                            Text("Compact view")
                            Text(
                                text = "No % sign, no units, compact time format",
                                style = MaterialTheme.typography.bodySmall,
                                color = MaterialTheme.colorScheme.onSurfaceVariant
                            )
                        }
                    }
                    
                    // Only show percentage option for templates with reminders
                    if (supportsPercentage) {
                        //HorizontalDivider(modifier = Modifier.padding(vertical = 4.dp))
                        
                        Row(
                            modifier = Modifier.fillMaxWidth(),
                            verticalAlignment = Alignment.CenterVertically
                        ) {
                            Checkbox(
                                checked = usePercentage,
                                onCheckedChange = { newValue -> 
                                    usePercentage = newValue
                                    // Apply the change immediately to preview widget and actual widget
                                    // This ensures the percentage setting takes immediate effect
                                    if (isEditMode) {
                                        // If in edit mode, update the actual widget immediately
                                        lifecycleScope.launch {
                                            val updatedConfig = WidgetConfig(
                                                templateId = template.id,
                                                showTime = showTime,
                                                showTemplateName = showTemplateName,
                                                showValue = showValue,
                                                showUnit = !compactView,
                                                transparency = opacity.toInt(),
                                                compactTime = compactView,
                                                nameClickQuickLog = nameClickQuickLog,
                                                usePercentage = newValue,
                                                selectedDrugIndex = selectedDrugIndex
                                            )
                                            
                                            // Save the config and update the widget
                                            widgetRepository.saveWidgetConfig(appWidgetId, updatedConfig)
                                            val appWidgetManager = AppWidgetManager.getInstance(this@WidgetConfigurationActivity)
                                            TemplateWidgetProvider.updateWidget(
                                                this@WidgetConfigurationActivity,
                                                appWidgetManager,
                                                appWidgetId
                                            )
                                        }
                                    }
                                }
                            )
                            Column(modifier = Modifier.padding(start = 0.dp)) {
                                Text("Use percentage")
                                Text(
                                    text = "Show % instead of time/concentration values",
                                    style = MaterialTheme.typography.bodySmall,
                                    color = MaterialTheme.colorScheme.onSurfaceVariant
                                )
                            }
                        }
                    }
                }
                
                //HorizontalDivider()
                
                // Opacity slider (compact version)
                Row(
                    modifier = Modifier.fillMaxWidth(),
                    horizontalArrangement = Arrangement.SpaceBetween,
                    verticalAlignment = Alignment.CenterVertically
                ) {
                    Text(
                        text = "Opacity: ${opacity.toInt()}%",
                        style = MaterialTheme.typography.bodyMedium
                    )
                    Text(
                        text = when (opacity.toInt()) {
                            100 -> "Fully visible"
                            0 -> "Invisible"
                            else -> "Semi-transparent"
                        },
                        style = MaterialTheme.typography.bodySmall,
                        color = MaterialTheme.colorScheme.onSurfaceVariant
                    )
                }
                
                Slider(
                    value = opacity,
                    onValueChange = { opacity = it },
                    valueRange = 0f..100f,
                    steps = 19, // 5% increments
                    modifier = Modifier.fillMaxWidth()
                )
                
                Spacer(modifier = Modifier.weight(1f))
                
                // Confirm button
                Button(
                    onClick = {
                        val config = WidgetConfig(
                            templateId = template.id,
                            showTime = showTime,
                            showTemplateName = showTemplateName,
                            showValue = showValue,
                            showUnit = !compactView,  // Inverted: compact view means NO unit
                            transparency = opacity.toInt(),
                            compactTime = compactView,  // Compact view enables compact time
                            nameClickQuickLog = nameClickQuickLog,
                            usePercentage = usePercentage,
                            selectedDrugIndex = selectedDrugIndex
                        )
                        onConfirm(config)
                    },
                    modifier = Modifier.fillMaxWidth()
                ) {
                    Text(if (isEditMode) "Update Widget" else "Add Widget")
                }
            }
        }
    }

    @Composable
    private fun WidgetPreview(
        template: Template,
        showTime: Boolean,
        showTemplateName: Boolean,
        showValue: Boolean,
        opacity: Int,
        compactView: Boolean = false,
        usePercentage: Boolean = false,
        selectedDrugIndex: Int = 0
    ) {
        // Query AppWidgetManager for actual 1x1 widget dimensions
        val appWidgetManager = AppWidgetManager.getInstance(this)
        val options = appWidgetManager.getAppWidgetOptions(appWidgetId)
        
        // Get min/max dimensions (in dp) - these represent portrait and landscape
        val minWidth = options.getInt(AppWidgetManager.OPTION_APPWIDGET_MIN_WIDTH, 40)
        val minHeight = options.getInt(AppWidgetManager.OPTION_APPWIDGET_MIN_HEIGHT, 40)
        val maxWidth = options.getInt(AppWidgetManager.OPTION_APPWIDGET_MAX_WIDTH, 40)
        val maxHeight = options.getInt(AppWidgetManager.OPTION_APPWIDGET_MAX_HEIGHT, 40)
        
        // Portrait: use minWidth x maxHeight
        // Landscape: use maxWidth x minHeight
        val portraitWidth = minWidth
        val portraitHeight = maxHeight
        val landscapeWidth = maxWidth
        val landscapeHeight = minHeight
        
        Row(
            modifier = Modifier.fillMaxWidth(),
            horizontalArrangement = Arrangement.spacedBy(16.dp)
        ) {
                // Portrait column
                Column(
                    modifier = Modifier.weight(1f),
                    horizontalAlignment = Alignment.CenterHorizontally,
                    verticalArrangement = Arrangement.spacedBy(8.dp)
                ) {
                    Surface(
                        color = MaterialTheme.colorScheme.surface.copy(alpha = 0.9f),
                        shape = RoundedCornerShape(4.dp)
                    ) {
                        Text(
                            text = "Portrait",
                            style = MaterialTheme.typography.labelMedium,
                            modifier = Modifier.padding(horizontal = 8.dp, vertical = 2.dp)
                        )
                    }
                    AndroidView(
                        factory = { context ->
                            // Force portrait orientation for portrait preview
                            val config = android.content.res.Configuration(context.resources.configuration)
                            config.orientation = android.content.res.Configuration.ORIENTATION_PORTRAIT
                            val portraitContext = context.createConfigurationContext(config)
                            
                            val view = LayoutInflater.from(portraitContext).inflate(de.ntdote.medicalcalendarlog.R.layout.widget_template, null, false)
                            // Force exact widget dimensions from launcher
                            view.layoutParams = android.widget.LinearLayout.LayoutParams(
                                context.resources.displayMetrics.density.let { (portraitWidth * it).toInt() },
                                context.resources.displayMetrics.density.let { (portraitHeight * it).toInt() }
                            )
                            // Store the context in view tag for use in update
                            view.tag = portraitContext
                            view
                        },
                        update = { view ->
                            val context = view.tag as? android.content.Context ?: applicationContext
                            val remoteViews = RemoteViews(context.packageName, de.ntdote.medicalcalendarlog.R.layout.widget_template)
                            lifecycleScope.launch {
                                populateWidgetPreview(remoteViews, template, showTime, showTemplateName, showValue, !compactView, opacity, compactView, usePercentage, selectedDrugIndex)
                                remoteViews.reapply(context, view)
                            }
                        },
                        modifier = Modifier
                            .width(portraitWidth.dp)
                            .height(portraitHeight.dp)
                    )
                }
            
            // Landscape column
            Column(
                modifier = Modifier.weight(1f),
                horizontalAlignment = Alignment.CenterHorizontally,
                verticalArrangement = Arrangement.spacedBy(8.dp)
            ) {
                Surface(
                    color = MaterialTheme.colorScheme.surface.copy(alpha = 0.9f),
                    shape = RoundedCornerShape(4.dp)
                ) {
                    Text(
                        text = "Landscape",
                        style = MaterialTheme.typography.labelMedium,
                        modifier = Modifier.padding(horizontal = 8.dp, vertical = 2.dp)
                    )
                }
                AndroidView(
                    factory = { context ->
                        // Manually load landscape layout by inflating from layout-land folder
                        val resources = context.resources
                        val layoutId = resources.getIdentifier("widget_template", "layout", context.packageName)
                        
                        // Create a configuration with landscape orientation
                        val config = android.content.res.Configuration(resources.configuration)
                        config.orientation = android.content.res.Configuration.ORIENTATION_LANDSCAPE
                        val landscapeContext = context.createConfigurationContext(config)
                        
                        val view = LayoutInflater.from(landscapeContext).inflate(layoutId, null, false)
                        // Force exact widget dimensions from launcher
                        view.layoutParams = android.widget.LinearLayout.LayoutParams(
                            context.resources.displayMetrics.density.let { (landscapeWidth * it).toInt() },
                            context.resources.displayMetrics.density.let { (landscapeHeight * it).toInt() }
                        )
                        // Store the context in view tag for use in update
                        view.tag = landscapeContext
                        view
                    },
                    update = { view ->
                        val context = view.tag as? android.content.Context ?: applicationContext
                        val remoteViews = RemoteViews(context.packageName, de.ntdote.medicalcalendarlog.R.layout.widget_template)
                        lifecycleScope.launch {
                            populateWidgetPreview(remoteViews, template, showTime, showTemplateName, showValue, !compactView, opacity, compactView, usePercentage, selectedDrugIndex)
                            remoteViews.reapply(context, view)
                        }
                    },
                    modifier = Modifier
                        .width(landscapeWidth.dp)
                        .height(landscapeHeight.dp)
                )
            }
        }
    }
    
    /**
     * Populate widget preview with sample data matching actual widget behavior
     */
    private suspend fun populateWidgetPreview(
        views: RemoteViews,
        template: Template,
        showTime: Boolean,
        showTemplateName: Boolean,
        showValue: Boolean,
        showUnit: Boolean,
        opacity: Int,
        compactTime: Boolean = false,
        usePercentage: Boolean = false,
        selectedDrugIndex: Int = 0
    ) {
        // Apply opacity to all badge backgrounds if supported
        // Note: setAlpha method may not be available through RemoteViews on older APIs (pre-API 29)
        // We need to check API level and skip opacity on older devices
        val alpha = opacity / 100.0f
        
        // Check if setAlpha is supported (available in API 29+)
        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.Q && opacity < 100) {
            // API 29 (Android 10) and above support setAlpha through RemoteViews
            // Only apply if opacity is less than 100% (transparent)
            views.setFloat(de.ntdote.medicalcalendarlog.R.id.widget_template_name, "setAlpha", alpha)
            views.setFloat(de.ntdote.medicalcalendarlog.R.id.widget_time_badge, "setAlpha", alpha)
            views.setFloat(de.ntdote.medicalcalendarlog.R.id.widget_value_badge, "setAlpha", alpha)
        }
        // On older APIs (API 28 and below), opacity feature is disabled
        // Widget preview will show at full opacity
        
        // Template name
        if (showTemplateName) {
            // If this is a DECAYING template with multiple drugs and we're showing concentration,
            // display "templatename:drugname" to clarify which drug the concentration refers to
            val displayName = if (template.templateType == TemplateType.DECAYING && 
                                 template.drugs.size > 1 && 
                                 showValue) {
                // Get the drug based on selectedDrugIndex
                val selectedDrug = template.drugs.getOrNull(selectedDrugIndex)
                if (selectedDrug != null) {
                    "${template.name}:${selectedDrug.drugType}"
                } else {
                    template.name
                }
            } else {
                template.name
            }
            views.setTextViewText(de.ntdote.medicalcalendarlog.R.id.widget_template_name, displayName)
            views.setViewVisibility(de.ntdote.medicalcalendarlog.R.id.widget_template_name, android.view.View.VISIBLE)
        } else {
            views.setViewVisibility(de.ntdote.medicalcalendarlog.R.id.widget_template_name, android.view.View.GONE)
        }
        
        // Hide info line (not used in widget)
        views.setViewVisibility(de.ntdote.medicalcalendarlog.R.id.widget_template_info, android.view.View.GONE)
        
        // Time badge
        if (showTime) {
            views.setViewVisibility(de.ntdote.medicalcalendarlog.R.id.widget_time_badge, android.view.View.VISIBLE)
            
                // Check if template has reminders and percentage mode is enabled
                val hasTimeReminder = template.reminderEnabled && template.reminderThresholdMillis != null
                
                if (usePercentage && hasTimeReminder) {
                    // Calculate an actual percentage based on the template's reminder threshold
                    // This will make the preview more realistic and reflect the actual widget behavior
                    val timePercentage = calculatePreviewTimePercentage(template)
                    val displayText = if (compactTime) "$timePercentage" else "${timePercentage}%"
                    views.setTextViewText(de.ntdote.medicalcalendarlog.R.id.widget_time_badge, displayText)
                    
                    // Color: red when >= 100%, blue otherwise
                    if (timePercentage >= 100) {
                        views.setInt(de.ntdote.medicalcalendarlog.R.id.widget_time_badge, "setBackgroundResource", de.ntdote.medicalcalendarlog.R.drawable.widget_badge_red)
                    } else {
                        views.setInt(de.ntdote.medicalcalendarlog.R.id.widget_time_badge, "setBackgroundResource", de.ntdote.medicalcalendarlog.R.drawable.widget_badge_blue)
                    }
            } else {
                // Show example time that demonstrates compact format difference
                val exampleTime = if (compactTime) "3d12" else "3d 12h"
                views.setTextViewText(de.ntdote.medicalcalendarlog.R.id.widget_time_badge, exampleTime)
                views.setInt(de.ntdote.medicalcalendarlog.R.id.widget_time_badge, "setBackgroundResource", de.ntdote.medicalcalendarlog.R.drawable.widget_badge_blue)
            }
        } else {
            views.setViewVisibility(de.ntdote.medicalcalendarlog.R.id.widget_time_badge, android.view.View.GONE)
        }
        
        // Value badge (for DECAYING and METRIC templates)
        var shouldShowValueBadge = false
        if (showValue && template.templateType != TemplateType.GENERAL) {
            when (template.templateType) {
                TemplateType.DECAYING -> {
                    // Check if selected drug has concentration reminder and percentage mode is enabled
                    val selectedDrug = template.drugs.getOrNull(selectedDrugIndex)
                    val hasConcentrationReminder = selectedDrug?.concentrationReminderEnabled == true && 
                                                   selectedDrug.concentrationReminderThreshold != null &&
                                                   selectedDrug.concentrationReminderThreshold!! > 0
                    
                    if (usePercentage && hasConcentrationReminder) {
                        // Calculate a realistic concentration percentage for preview
                        val concentrationPercentage = calculatePreviewConcentrationPercentage(template, selectedDrugIndex)
                        
                        // Format based on compact mode setting
                        val displayText = if (compactTime) "$concentrationPercentage" else "${concentrationPercentage}%"
                        views.setTextViewText(de.ntdote.medicalcalendarlog.R.id.widget_value_badge, displayText)
                        
                        // Color: red when < 100%, green otherwise (matching the actual widget behavior)
                        if (concentrationPercentage < 100) {
                            views.setInt(de.ntdote.medicalcalendarlog.R.id.widget_value_badge, "setBackgroundResource", de.ntdote.medicalcalendarlog.R.drawable.widget_badge_red)
                        } else {
                            views.setInt(de.ntdote.medicalcalendarlog.R.id.widget_value_badge, "setBackgroundResource", de.ntdote.medicalcalendarlog.R.drawable.widget_badge_green)
                        }
                        shouldShowValueBadge = true
                    } else {
                                        // Show normal concentration format - respect compact mode
                                        val selectedDrug = template.drugs.getOrNull(selectedDrugIndex)
                                        val unit = if (showUnit && selectedDrug != null) selectedDrug.unit else ""
                                        // Use a realistic sample concentration value
                                        val sampleValue = "2.5"
                                        val valueText = if (unit.isNotEmpty() && !compactTime) "$sampleValue $unit" else sampleValue
                                        views.setTextViewText(de.ntdote.medicalcalendarlog.R.id.widget_value_badge, valueText)
                        views.setInt(de.ntdote.medicalcalendarlog.R.id.widget_value_badge, "setBackgroundResource", de.ntdote.medicalcalendarlog.R.drawable.widget_badge_green)
                        shouldShowValueBadge = true
                    }
                }
                TemplateType.METRIC -> {
                    // Metric templates don't support percentage mode - respect compact mode
                    val unit = if (showUnit) template.metricUnit else ""
                    val valueText = if (unit.isNotEmpty() && !compactTime) "120 $unit" else "120"
                    views.setTextViewText(de.ntdote.medicalcalendarlog.R.id.widget_value_badge, valueText)
                    views.setInt(de.ntdote.medicalcalendarlog.R.id.widget_value_badge, "setBackgroundResource", de.ntdote.medicalcalendarlog.R.drawable.widget_badge_blue)
                    shouldShowValueBadge = true
                }
                else -> { }
            }
        }
        
        if (shouldShowValueBadge) {
            views.setViewVisibility(de.ntdote.medicalcalendarlog.R.id.widget_value_badge, android.view.View.VISIBLE)
        } else {
            views.setViewVisibility(de.ntdote.medicalcalendarlog.R.id.widget_value_badge, android.view.View.GONE)
        }
    }

    /**
     * Calculate a realistic time percentage for preview
     * This simulates the logic in TemplateWidgetProvider but uses hardcoded values
     * that look realistic in the preview
     */
    private fun calculatePreviewTimePercentage(template: Template): Int {
        // If template has a reminder threshold, use it to calculate a realistic percentage
        val reminderThresholdMillis = template.reminderThresholdMillis
        
        if (reminderThresholdMillis == null || reminderThresholdMillis <= 0) {
            return 75 // Default fallback
        }
        
        // For preview purposes, show a percentage that makes sense for the template
        // 75% is a good middle value that shows the color is blue (not overdue)
        return 75
    }
    
    /**
     * Calculate a realistic concentration percentage for preview
     * This simulates the logic in TemplateWidgetProvider but uses hardcoded values
     * that look realistic in the preview
     */
    private fun calculatePreviewConcentrationPercentage(template: Template, selectedDrugIndex: Int = 0): Int {
        // Check if the selected drug has a concentration threshold
        val selectedDrug = template.drugs.getOrNull(selectedDrugIndex)
        val concentrationThreshold = selectedDrug?.concentrationReminderThreshold
        
        if (concentrationThreshold == null || concentrationThreshold <= 0.0) {
            return 120 // Default fallback
        }
        
        // For preview purposes, show a percentage that makes sense for the template
        // 120% is a good value that shows the color is green (concentration is good)
        return 120
    }

    private fun configureWidget(config: WidgetConfig) {
        lifecycleScope.launch {
            // Save the widget configuration
            widgetRepository.saveWidgetConfig(appWidgetId, config)

            // Update the widget
            val appWidgetManager = AppWidgetManager.getInstance(this@WidgetConfigurationActivity)
            TemplateWidgetProvider.updateWidget(
                this@WidgetConfigurationActivity,
                appWidgetManager,
                appWidgetId
            )

            // Return success result
            val resultValue = Intent().apply {
                putExtra(AppWidgetManager.EXTRA_APPWIDGET_ID, appWidgetId)
            }
            setResult(Activity.RESULT_OK, resultValue)
            finish()
        }
    }
}
