package de.ntdote.medicalcalendarlog.service

import android.app.NotificationChannel
import android.app.NotificationManager
import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import android.database.ContentObserver
import android.net.Uri
import android.os.Build
import android.os.Handler
import android.os.Looper
import android.provider.CalendarContract
import android.util.Log
import android.widget.Toast
import androidx.core.app.NotificationCompat
import androidx.work.*
import de.ntdote.medicalcalendarlog.MainActivity
import de.ntdote.medicalcalendarlog.R
import de.ntdote.medicalcalendarlog.data.DecayType
import de.ntdote.medicalcalendarlog.data.Template
import de.ntdote.medicalcalendarlog.data.TemplateType
import de.ntdote.medicalcalendarlog.repository.CalendarRepository
import de.ntdote.medicalcalendarlog.repository.PreferencesRepository
import de.ntdote.medicalcalendarlog.utils.AdvancedCalculationEngine
import de.ntdote.medicalcalendarlog.utils.ConcentrationCalculator
import de.ntdote.medicalcalendarlog.utils.EventMatching
import de.ntdote.medicalcalendarlog.utils.NotificationBuilder
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.launch
import java.util.*
import java.util.concurrent.TimeUnit
import kotlin.math.ln
import kotlin.math.pow

class ReminderWorker(
    context: Context,
    params: WorkerParameters
) : CoroutineWorker(context, params) {

    override suspend fun doWork(): Result {
        return try {
            val startTime = System.currentTimeMillis()
            val timeFormatter = java.text.SimpleDateFormat("HH:mm:ss", java.util.Locale.getDefault())
            val currentTimeStr = timeFormatter.format(java.util.Date(startTime))
            
            val reminderType = inputData.getString("reminderType") ?: return Result.failure()
            val templateId = inputData.getString("templateId") ?: return Result.failure()
            val drugType = inputData.getString("drugType")
            
            Log.d("ReminderWorker", "MCL: [$currentTimeStr] Scheduled reminder worker started ($reminderType)")
            
            // Update all widgets before sending notification to ensure consistency
            val preferencesRepository = PreferencesRepository(applicationContext)
            val debugModeEnabled = preferencesRepository.debugModeEnabled.first()
            TemplateWidgetProvider.updateAllWidgets(applicationContext, debugModeEnabled)
            Log.d("ReminderWorker", "MCL: Widgets updated before scheduled notification (debug: $debugModeEnabled)")
            
            // Record worker run with type in PreferencesRepository for UI display
            val workerType = when (reminderType) {
                "time_based" -> "scheduled-time"
                "concentration" -> "scheduled-conc"
                else -> "scheduled"
            }
            preferencesRepository.recordWorkerRun(startTime, workerType)
            
            when (reminderType) {
                "time_based" -> {
                    if (drugType != null) {
                        sendDrugReminderNotification(drugType, templateId)
                    } else {
                        sendTemplateReminderNotification(templateId)
                    }
                }
                "concentration" -> {
                    sendConcentrationReminderNotification(templateId, drugType ?: "")
                }
            }
            
            // Schedule the next reminder in the chain
            ReminderService.recalculateAllReminders(applicationContext, "reminder_fired")
            
            Result.success()
        } catch (e: Exception) {
            Result.retry()
        }
    }
    
    private suspend fun sendDrugReminderNotification(drugType: String, templateId: String) {
        val preferencesRepository = PreferencesRepository(applicationContext)
        val templates = preferencesRepository.templates.first()
        val template = templates.find { it.id == templateId } ?: return
        
        val notificationManager = applicationContext.getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager
        NotificationBuilder.createNotificationChannels(notificationManager)
        
        // Use unified notification builder with mute buttons
        val (notification, notificationId) = NotificationBuilder.buildTimeBasedNotification(
            context = applicationContext,
            template = template,
            drugName = drugType,
            hoursOverdue = 0,
            isOverlooked = false,
            hoursSinceOriginalNotification = 0
        )
        
        // Store notification metadata
        NotificationBuilder.storeNotificationMetadata(
            context = applicationContext,
            notificationId = notificationId,
            templateId = templateId,
            isConcentration = false
        )
        
        notificationManager.notify(notificationId, notification)
    }

    private suspend fun sendTemplateReminderNotification(templateId: String) {
        val preferencesRepository = PreferencesRepository(applicationContext)
        val templates = preferencesRepository.templates.first()
        val template = templates.find { it.id == templateId } ?: return
        
        val notificationManager = applicationContext.getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager
        NotificationBuilder.createNotificationChannels(notificationManager)
        
        // Use unified notification builder with mute buttons
        val (notification, notificationId) = NotificationBuilder.buildTimeBasedNotification(
            context = applicationContext,
            template = template,
            drugName = template.name,
            hoursOverdue = 0,
            isOverlooked = false,
            hoursSinceOriginalNotification = 0
        )
        
        // Store notification metadata
        NotificationBuilder.storeNotificationMetadata(
            context = applicationContext,
            notificationId = notificationId,
            templateId = templateId,
            isConcentration = false
        )
        
        notificationManager.notify(notificationId, notification)
    }
    
    private suspend fun sendConcentrationReminderNotification(templateId: String, drugType: String) {
        val preferencesRepository = PreferencesRepository(applicationContext)
        val templates = preferencesRepository.templates.first()
        val template = templates.find { it.id == templateId } ?: return
        
        val notificationManager = applicationContext.getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager
        NotificationBuilder.createNotificationChannels(notificationManager)
        
        // Calculate current concentration for notification
        val calendarRepository = CalendarRepository(applicationContext.contentResolver, applicationContext)
        val selectedCalendarId = preferencesRepository.selectedCalendarId.first() ?: return
        val events = calendarRepository.getEventsForPeriod(selectedCalendarId, 24 * 7)
        val currentTime = Date()
        
        val currentConcentration = ConcentrationCalculator.calculateCurrentDrugConcentration(
            drugType, templates, events, currentTime
        )
        
        // Use unified notification builder with mute buttons
        val (notification, notificationId) = NotificationBuilder.buildConcentrationNotification(
            context = applicationContext,
            template = template,
            drugName = drugType,
            currentConcentration = currentConcentration,
            hoursSinceOriginalNotification = 0
        )
        
        // Store notification metadata
        NotificationBuilder.storeNotificationMetadata(
            context = applicationContext,
            notificationId = notificationId,
            templateId = templateId,
            isConcentration = true
        )
        
        notificationManager.notify(notificationId, notification)
    }
    
    companion object {
        // No longer needed - using NotificationBuilder
    }
}

class ReminderService {
    companion object {
        private const val PREFS_KEY_CONCENTRATION_NOTIFIED = "concentration_notified_"
        
        // Debouncing variables - now requires external scope to be passed in
        private var debounceJob: Job? = null
        private const val DEBOUNCE_DELAY_MS = 5000L // 3.5 seconds for calendar changes
        
        // Checksum cache for smart calendar change detection
        private var lastChecksum: Long = 0L
        
        fun cleanup(context: Context) {
            // Cancel any pending debounce job
            debounceJob?.cancel()
            debounceJob = null
            
            // Cancel all existing reminders
            WorkManager.getInstance(context).cancelAllWorkByTag("reminder")
        }

        /**
         * Find events relevant to a specific template
         */
        private fun findRelevantEvents(template: Template, events: List<de.ntdote.medicalcalendarlog.data.CalendarEvent>): List<de.ntdote.medicalcalendarlog.data.CalendarEvent> {
            // Use unified matching logic
            return EventMatching.filterEventsForTemplate(events, template)
        }
        
        /**
         * Recalculate all reminders with optional debouncing
         * @param context Application context
         * @param reason Reason for recalculation (affects debouncing strategy)
         * @param coroutineScope Scope to launch debounced jobs in (required for proper cancellation)
         * @param onDebounceComplete Callback invoked after debounce completes (for calendar_change only)
         */
        fun recalculateAllReminders(
            context: Context, 
            reason: String = "unknown",
            coroutineScope: CoroutineScope? = null,
            onDebounceComplete: (() -> Unit)? = null
        ) {
            // Determine if this should be debounced based on the reason
            when (reason) {
                "calendar_change" -> {
                    // Calendar changes MUST be debounced and REQUIRE a scope
                    if (coroutineScope == null) {
                        Log.w("MCL", "calendar_change requires coroutineScope for debouncing - falling back to immediate execution")
                        CoroutineScope(Dispatchers.IO).launch {
                            performRecalculation(context, reason)
                        }
                    } else {
                        debouncedRecalculate(context, reason, DEBOUNCE_DELAY_MS, coroutineScope, onDebounceComplete)
                    }
                }
                "app_start", "templates_imported", "reminder_fired", "app_update", "permissions_granted" -> {
                    // Immediate execution for these critical events (no scope needed)
                    CoroutineScope(Dispatchers.IO).launch {
                        performRecalculation(context, reason)
                    }
                }
                else -> {
                    // Default to debounced for unknown reasons if scope provided
                    if (coroutineScope != null) {
                        debouncedRecalculate(context, reason, DEBOUNCE_DELAY_MS, coroutineScope, onDebounceComplete)
                    } else {
                        Log.w("MCL", "Unknown reason '$reason' without scope - using immediate execution")
                        CoroutineScope(Dispatchers.IO).launch {
                            performRecalculation(context, reason)
                        }
                    }
                }
            }
        }
        
        private fun debouncedRecalculate(
            context: Context, 
            reason: String, 
            delayMs: Long,
            coroutineScope: CoroutineScope,
            onDebounceComplete: (() -> Unit)? = null
        ) {
            // Use synchronized block to prevent race conditions from multiple threads
            synchronized(ReminderService::class.java) {
                // Cancel any existing debounce job
                debounceJob?.cancel()
                
                // Start new debounced job in the PROVIDED scope (critical for proper cancellation!)
                debounceJob = coroutineScope.launch {
                    delay(delayMs)
                    
                    // Invoke callback right after debounce completes, before processing starts
                    onDebounceComplete?.invoke()
                    
                    // Perform the recalculation (can be cancelled by new debounce)
                    performRecalculation(context, reason)
                }
            }
        }
        
        // Cache for the last loaded events to enable smart incremental loading
        private var lastLoadedEvents: List<de.ntdote.medicalcalendarlog.data.CalendarEvent> = emptyList()
        private var lastLoadedDaysBack: Int = 0
        
        private suspend fun performRecalculation(context: Context, reason: String) {
            try {
                val preferencesRepository = PreferencesRepository(context)
                // Pass context to CalendarRepository so it can check permissions
                val calendarRepository = CalendarRepository(context.contentResolver, context)
                
                // Check if calendar permissions are granted
                if (!hasCalendarPermissions(context)) {
                    Log.w("MCL", "Cannot recalculate reminders - calendar permissions not granted (reason: $reason)")
                    return
                }
                
                val templates = preferencesRepository.templates.first()
                val selectedCalendarId = preferencesRepository.selectedCalendarId.first()
                val daysBack = preferencesRepository.daysBack.first()
                
                if (selectedCalendarId == null) return
                
                // Get window start timestamp for checksum comparison
                val windowStart = preferencesRepository.windowStartTimestamp.first()
                
                // Determine loading strategy based on reason
                val events = when (reason) {
                    "calendar_change" -> {
                        // Calendar changed - use checksum to detect if recalculation is needed
                        if (windowStart > 0L) {
                            // We have a valid window - load events from window start and check checksum
                            val windowEvents = calendarRepository.getEventsFromTimestamp(selectedCalendarId, windowStart)
                            val newChecksum = calendarRepository.calculateEventsChecksum(windowEvents)
                            
                            if (newChecksum == lastChecksum) {
                                // No changes detected in the window - skip recalculation
                                Log.d("MCL", "Checksum comparison: No changes detected (checksum=$newChecksum, ${windowEvents.size} events in window)")
                                Log.d("MCL", "  Skipping recalculation - calendar unchanged")
                                return // Exit early - no changes detected
                            } else {
                                // Changes detected - proceed with recalculation
                                Log.d("MCL", "Checksum comparison: Changes detected!")
                                Log.d("MCL", "  Old checksum: $lastChecksum")
                                Log.d("MCL", "  New checksum: $newChecksum")
                                Log.d("MCL", "  Window events: ${windowEvents.size}")
                                
                                // Update checksum cache
                                lastChecksum = newChecksum
                                
                                // Update widgets since calendar changed
                                val debugModeEnabled = preferencesRepository.debugModeEnabled.first()
                                TemplateWidgetProvider.updateAllWidgets(context, debugModeEnabled)
                                Log.d("MCL", "Widgets updated after calendar change detection")
                                
                                // Load full event range for recalculation
                                Log.d("MCL", "Loading full event range for recalculation ($daysBack days)")
                                calendarRepository.getEventsForPeriod(selectedCalendarId, daysBack)
                            }
                        } else {
                            // No window set yet - initial setup, load events and initialize window
                            Log.d("MCL", "No checksum window set - initializing on first calendar change")
                            val fullEvents = calendarRepository.getEventsForPeriod(selectedCalendarId, daysBack)
                            
                            // Initialize window
                            val now = System.currentTimeMillis()
                            preferencesRepository.updateWindowIfNeeded(now, daysBack)
                            
                            // Calculate and store initial checksum
                            val windowEvents = calendarRepository.getEventsFromTimestamp(selectedCalendarId, now)
                            lastChecksum = calendarRepository.calculateEventsChecksum(windowEvents)
                            Log.d("MCL", "Initialized checksum window and calculated initial checksum=$lastChecksum")
                            
                            fullEvents
                        }
                    }
                    "app_start", "app_update" -> {
                        // App start or update - update/initialize window and load full range
                        Log.d("MCL", "App start/update: Updating window and loading events ($daysBack days)")
                        val now = System.currentTimeMillis()
                        val newWindowStart = preferencesRepository.updateWindowIfNeeded(now, daysBack)
                        
                        val fullEvents = calendarRepository.getEventsForPeriod(selectedCalendarId, daysBack)
                        
                        // Calculate and store initial checksum
                        val windowEvents = calendarRepository.getEventsFromTimestamp(selectedCalendarId, newWindowStart)
                        lastChecksum = calendarRepository.calculateEventsChecksum(windowEvents)
                        Log.d("MCL", "Calculated initial checksum=$lastChecksum for ${windowEvents.size} events")
                        
                        fullEvents
                    }
                    else -> {
                        // Template changes, reminder fired, etc. - calendar hasn't changed
                        // Reuse cached events if available and daysBack hasn't changed
                        if (lastLoadedEvents.isNotEmpty() && lastLoadedDaysBack == daysBack) {
                            Log.d("MCL", "Cache reuse: Using cached events for $reason (${lastLoadedEvents.size} events)")
                            lastLoadedEvents
                        } else {
                            // Cache empty or daysBack changed - reload and recalculate checksum
                            Log.d("MCL", "Cache invalid (daysBack changed): Reloading for $reason ($daysBack days)")
                            val fullEvents = calendarRepository.getEventsForPeriod(selectedCalendarId, daysBack)
                            
                            // Recalculate checksum if window is set
                            if (windowStart > 0L) {
                                val windowEvents = calendarRepository.getEventsFromTimestamp(selectedCalendarId, windowStart)
                                lastChecksum = calendarRepository.calculateEventsChecksum(windowEvents)
                                Log.d("MCL", "Recalculated checksum=$lastChecksum after cache reload")
                            }
                            
                            fullEvents
                        }
                    }
                }
                
                // Update cache
                lastLoadedEvents = events
                lastLoadedDaysBack = daysBack
                
                // Cancel all existing reminders
                WorkManager.getInstance(context).cancelAllWorkByTag("reminder")
                
                // Find the next reminder to schedule
                val nextReminder = findEarliestReminder(templates, selectedCalendarId, calendarRepository, context)
                
                if (nextReminder != null) {
                    scheduleReminder(context, nextReminder, reason)
                } else {
                    Log.d("MCL", "No reminders to schedule (reason: $reason)")
                }
                
            } catch (e: Exception) {
                Log.e("MCL", "Error during reminder recalculation (reason: $reason)", e)
            }
        }
        
        /**
         * Check if calendar permissions are granted
         */
        private fun hasCalendarPermissions(context: Context): Boolean {
            val readCalendarPermission = androidx.core.content.ContextCompat.checkSelfPermission(
                context, 
                android.Manifest.permission.READ_CALENDAR
            )
            
            val writeCalendarPermission = androidx.core.content.ContextCompat.checkSelfPermission(
                context, 
                android.Manifest.permission.WRITE_CALENDAR
            )
            
            return readCalendarPermission == android.content.pm.PackageManager.PERMISSION_GRANTED &&
                   writeCalendarPermission == android.content.pm.PackageManager.PERMISSION_GRANTED
        }
        
        private suspend fun findEarliestReminder(
            templates: List<Template>,
            calendarId: Long,
            calendarRepository: CalendarRepository,
            context: Context
        ): ReminderInfo? {
            val currentTime = Date()
            val reminders = mutableListOf<ReminderInfo>()
            
            // Get events for analysis (look back and forward enough to cover all scenarios)
            val events = calendarRepository.getEventsForPeriod(calendarId, 24 * 7) // 7 days
            
            Log.d("MCL", "═══ Reminder Calculation Debug ═══")
            Log.d("MCL", "Current time: $currentTime")
            Log.d("MCL", "Total templates: ${templates.size}, Enabled with time reminders: ${templates.count { it.reminderEnabled && it.reminderThresholdMillis != null && it.enabled }}")
            
            // Check time-based reminders - only process enabled templates
            templates.filter { 
                it.reminderEnabled && 
                it.reminderThresholdMillis != null && 
                it.enabled // Only process enabled templates
            }.forEach { template ->
                val drugOrTemplateName = if (template.templateType == TemplateType.DECAYING && template.drugs.isNotEmpty()) {
                    template.getDrugTypesDisplay()
                } else template.name
                
                val thresholdHours = (template.reminderThresholdMillis ?: 0) / (60.0 * 60.0 * 1000.0)
                Log.d("MCL", "─── Checking template: $drugOrTemplateName (threshold: ${String.format("%.2f", thresholdHours)}h)")
                
                val reminderTime = calculateTimeBasedReminder(template, events, currentTime)
                
                if (reminderTime != null) {
                    val delay = (reminderTime.time - currentTime.time) / (1000 * 60) // minutes
                    Log.d("MCL", "    ✓ Calculated reminder: $reminderTime (in ${delay}m)")
                    
                    if (reminderTime.after(currentTime)) {
                        val drugType = if (template.templateType == TemplateType.DECAYING && template.drugs.isNotEmpty()) {
                            template.drugs.first().drugType // Keep this for ReminderInfo - it needs individual drug type for scheduling
                        } else null
                        
                        reminders.add(ReminderInfo(
                            time = reminderTime,
                            type = "time_based",
                            templateId = template.id,
                            drugType = drugType
                        ))
                        Log.d("MCL", "    ✓ Added to reminder list")
                    } else {
                        Log.d("MCL", "    ✗ Reminder is in the past, skipped")
                    }
                } else {
                    Log.d("MCL", "    ✗ No reminder calculated (no qualifying events or disabled)")
                }
            }
            
            // Check concentration-based reminders - only process enabled templates
            Log.d("MCL", "─── Concentration Reminders ───")
            Log.d("MCL", "Enabled with conc reminders: ${templates.count { 
                it.templateType == TemplateType.DECAYING && 
                it.drugs.any { drug -> drug.concentrationReminderEnabled && drug.concentrationReminderThreshold != null } &&
                it.enabled 
            }}")
            
            templates.filter { 
                it.templateType == TemplateType.DECAYING &&
                it.drugs.any { drug -> drug.concentrationReminderEnabled && drug.concentrationReminderThreshold != null } &&
                it.enabled // Only process enabled templates
            }.forEach { template ->
                // Process each drug in the template that has concentration reminders enabled
                template.drugs.filter { drug -> 
                    drug.concentrationReminderEnabled && drug.concentrationReminderThreshold != null 
                }.forEach { drug ->
                    val drugName = drug.drugType
                    Log.d("MCL", "─── Checking concentration: $drugName (threshold: ${drug.concentrationReminderThreshold})")
                
                    val reminderTime = calculateConcentrationReminderForDrug(template, drug, events, currentTime, context)
                
                    if (reminderTime != null) {
                        val delay = (reminderTime.time - currentTime.time) / (1000 * 60) // minutes
                        Log.d("MCL", "    ✓ Calculated conc reminder: $reminderTime (in ${delay}m)")
                    
                        if (reminderTime.after(currentTime)) {
                            reminders.add(ReminderInfo(
                                time = reminderTime,
                                type = "concentration",
                                templateId = template.id,
                                drugType = drug.drugType
                            ))
                            Log.d("MCL", "    ✓ Added to reminder list")
                        } else {
                            Log.d("MCL", "    ✗ Reminder is in the past, skipped")
                        }
                    } else {
                        Log.d("MCL", "    ✗ No concentration reminder needed")
                    }
                }
            }
            
            // Return the earliest reminder
            Log.d("MCL", "═══ Summary ═══")
            Log.d("MCL", "Total reminders calculated: ${reminders.size}")
            reminders.sortedBy { it.time }.forEach { reminder ->
                val delay = (reminder.time.time - currentTime.time) / (1000 * 60)
                val template = templates.find { it.id == reminder.templateId }
                val displayName = reminder.drugType ?: template?.name ?: reminder.templateId
                Log.d("MCL", "  - $displayName (${reminder.type}): ${reminder.time} (in ${delay}m)")
            }
            
            val earliest = reminders.minByOrNull { it.time }
            if (earliest != null) {
                val delay = (earliest.time.time - currentTime.time) / (1000 * 60)
                val template = templates.find { it.id == earliest.templateId }
                val displayName = earliest.drugType ?: template?.name ?: earliest.templateId
                Log.d("MCL", "Selected earliest: $displayName at ${earliest.time} (in ${delay}m)")
            } else {
                Log.d("MCL", "No reminders to schedule")
            }
            Log.d("MCL", "═══════════════════════════════")
            
            return earliest
        }
        
        private fun calculateTimeBasedReminder(template: Template, events: List<de.ntdote.medicalcalendarlog.data.CalendarEvent>, currentTime: Date): Date? {
            val thresholdMillis = template.getEffectiveReminderThresholdMillis() ?: return null
            
            // Find relevant events using unified matching logic
            val relevantEvents = EventMatching.filterEventsForTemplate(events, template)
            
            val lastEvent = relevantEvents.maxByOrNull { it.startTime }
            
            return if (lastEvent != null) {
                Date(lastEvent.startTime.time + thresholdMillis)
            } else {
                // No events found - don't schedule a reminder for unused templates
                null
            }
        }
        
        private suspend fun calculateConcentrationReminderForDrug(
            template: Template,
            drug: de.ntdote.medicalcalendarlog.data.Drug,
            events: List<de.ntdote.medicalcalendarlog.data.CalendarEvent>, 
            currentTime: Date,
            context: Context
        ): Date? {
            val threshold = drug.concentrationReminderThreshold ?: return null
            val drugType = drug.drugType
            
            // Check if we've already notified for this threshold crossing (use template + drug combination)
            val prefs = context.getSharedPreferences("reminder_prefs", Context.MODE_PRIVATE)
            val lastNotifiedKey = PREFS_KEY_CONCENTRATION_NOTIFIED + template.id + "_" + drugType
            val lastNotifiedTime = prefs.getLong(lastNotifiedKey, 0L)
            
            // Get templates for concentration calculation
            val preferencesRepository = PreferencesRepository(context)
            val templates = preferencesRepository.templates.first()
            
            // Get current concentration
            val currentConcentration = ConcentrationCalculator.calculateCurrentDrugConcentration(
                drugType, templates, events, currentTime
            )
            
            // If concentration is above threshold, reset notification state
            if (currentConcentration > threshold) {
                prefs.edit().remove(lastNotifiedKey).apply()
                
                // Calculate when it will cross the threshold
                return calculateThresholdCrossingTimeForDrug(drug, templates, events, currentTime, threshold)
            } else {
                // Below threshold - check if we should notify
                val shouldNotify = lastNotifiedTime == 0L || 
                    (currentTime.time - lastNotifiedTime) > (24 * 60 * 60 * 1000L) // 24 hours since last notification
                
                if (shouldNotify) {
                    // Mark as notified
                    prefs.edit().putLong(lastNotifiedKey, currentTime.time).apply()
                    return currentTime // Notify immediately
                }
            }
            
            return null
        }
        
        private fun calculateThresholdCrossingTimeForDrug(
            drug: de.ntdote.medicalcalendarlog.data.Drug,
            templates: List<Template>,
            events: List<de.ntdote.medicalcalendarlog.data.CalendarEvent>,
            currentTime: Date,
            threshold: Double
        ): Date? {
            val drugType = drug.drugType
            
            val currentConcentration = ConcentrationCalculator.calculateCurrentDrugConcentration(
                drugType, templates, events, currentTime
            )
            
            if (currentConcentration <= threshold) return null // Already below threshold
            
            val hoursToThreshold = when (drug.decayType) {
                DecayType.HALF_LIFE -> {
                    val halfLife = drug.halfLifeHours ?: return null
                    // threshold = currentConcentration * (0.5)^(hours / halfLife)
                    // Solve for hours: hours = halfLife * log2(currentConcentration / threshold)
                    halfLife * (ln(currentConcentration / threshold) / ln(2.0))
                }
                DecayType.CONSTANT -> {
                    val decayRate = drug.hourlyDecayRate ?: return null
                    // threshold = currentConcentration - (decayRate * hours)
                    // Solve for hours: hours = (currentConcentration - threshold) / decayRate
                    if (decayRate <= 0) return null
                    (currentConcentration - threshold) / decayRate
                }
                null -> {
                    // Derived drug - cannot calculate threshold crossing without decay parameters
                    return null
                }
            }
            
            if (hoursToThreshold <= 0) return null
            
            return Date(currentTime.time + (hoursToThreshold * 60 * 60 * 1000L).toLong())
        }
        
        private suspend fun scheduleReminder(context: Context, reminderInfo: ReminderInfo, reason: String = "unknown") {
            val delay = reminderInfo.time.time - System.currentTimeMillis()
            if (delay <= 0) return // Don't schedule past reminders
            
            // Debug logging and user notification
            val preferencesRepository = PreferencesRepository(context)
            
            // Get template name for display
            val templates = preferencesRepository.templates.first()
            val template = templates.find { it.id == reminderInfo.templateId }
            val templateName = template?.name ?: "Unknown template"
            
            val drugOrEvent = reminderInfo.drugType ?: templateName
            val logMessage = "setting next reminder to ${reminderInfo.time} for $drugOrEvent (reason: $reason)"
            Log.d("MCL", logMessage)
            
            // Check if debug mode is enabled (which includes toast notifications)
            val debugModeEnabled = preferencesRepository.debugModeEnabled.first()
            
            if (debugModeEnabled) {
                // Show toast notification to user on main thread
                Handler(Looper.getMainLooper()).post {
                    val cleanName = drugOrEvent
                    val reminderTypeText = when (reminderInfo.type) {
                        "time_based" -> "Time-based"
                        "concentration" -> "Concentration"
                        else -> "Unknown"
                    }
                    val reasonText = when (reason) {
                        "app_start" -> "App Start"
                        "calendar_change" -> "Calendar Change"
                        "reminder_fired" -> "Previous Reminder"
                        "templates_imported" -> "Templates Updated"
                        else -> "System"
                    }
                    val timeFormatted = java.text.SimpleDateFormat("MMM dd, HH:mm", java.util.Locale.getDefault()).format(reminderInfo.time)
                    val toastMessage = "Next reminder: $cleanName at $timeFormatted\n($reminderTypeText • $reasonText)"
                    Toast.makeText(context, toastMessage, Toast.LENGTH_LONG).show()
                }
            }
            
            val inputData = Data.Builder()
                .putString("reminderType", reminderInfo.type)
                .putString("templateId", reminderInfo.templateId)
                .putString("drugType", reminderInfo.drugType)
                .build()
            
            val reminderWork = OneTimeWorkRequestBuilder<ReminderWorker>()
                .setInitialDelay(delay, TimeUnit.MILLISECONDS)
                .setInputData(inputData)
                .addTag("reminder")
                .build()
            
            WorkManager.getInstance(context).enqueue(reminderWork)
            
            // Store the next reminder time in PreferencesRepository for UI display
            CoroutineScope(Dispatchers.IO).launch {
                val cleanName = drugOrEvent
                preferencesRepository.setNextReminderInfo(reminderInfo.time.time, cleanName)
            }
        }
    }
}

data class ReminderInfo(
    val time: Date,
    val type: String, // "time_based" or "concentration"
    val templateId: String,
    val drugType: String?
)
