package de.ntdote.medicalcalendarlog.service

import android.app.NotificationManager
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.os.Handler
import android.util.Log
import androidx.core.app.RemoteInput
import de.ntdote.medicalcalendarlog.repository.CalendarRepository
import de.ntdote.medicalcalendarlog.repository.PreferencesRepository
import de.ntdote.medicalcalendarlog.utils.EventMatching
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.launch

/**
 * Receiver for handling notification action buttons (Mute 24h, Mute this time)
 */
class NotificationActionReceiver : BroadcastReceiver() {

    companion object {
        const val ACTION_MUTE_24H = "de.ntdote.medicalcalendarlog.MUTE_24H"
        const val ACTION_MUTE_UNTIL_RELOG = "de.ntdote.medicalcalendarlog.MUTE_UNTIL_RELOG"
        const val ACTION_LOG = "de.ntdote.medicalcalendarlog.ACTION_LOG"
        const val ACTION_DISMISS = "de.ntdote.medicalcalendarlog.ACTION_DISMISS"
        const val EXTRA_TEMPLATE_ID = "template_id"
        const val EXTRA_TEMPLATE_NAME = "template_name"
        const val EXTRA_NOTIFICATION_ID = "notification_id"
        const val EXTRA_IS_CONCENTRATION = "is_concentration"
        const val EXTRA_REQUIRES_INPUT = "requires_input"
        const val KEY_TEXT_REPLY = "key_text_reply"
        
        private const val TAG = "MCL"
        
        // SharedPreferences keys
        private const val PREFS_NAME = "notification_mute_prefs"
        private const val KEY_MUTE_24H_UNTIL = "mute_24h_until_"
        private const val KEY_MUTE_UNTIL_RELOG = "mute_until_relog_"
        private const val KEY_MUTE_LAST_EVENT_TIME = "mute_last_event_time_"
        
        /**
         * Check if a template is muted (24h or until relog)
         * Returns true if the template should not show notifications
         */
        fun isTemplateMuted(context: Context, templateId: String, isConcentration: Boolean = false): Boolean {
            val prefs = context.getSharedPreferences(PREFS_NAME, Context.MODE_PRIVATE)
            val currentTime = System.currentTimeMillis()
            
            val muteKey = if (isConcentration) "${templateId}_concentration" else templateId
            
            // Check 24h mute
            val mute24hUntil = prefs.getLong("$KEY_MUTE_24H_UNTIL$muteKey", 0L)
            if (mute24hUntil > currentTime) {
                Log.d(TAG, "Template $muteKey is muted (24h) until ${mute24hUntil - currentTime}ms")
                return true
            }
            
            // Check mute until relog
            val muteUntilRelog = prefs.getBoolean("$KEY_MUTE_UNTIL_RELOG$muteKey", false)
            if (muteUntilRelog) {
                Log.d(TAG, "Template $muteKey is muted (until relog)")
                return true
            }
            
            return false
        }
        
        /**
         * Clear "mute until relog" if a new event has been logged for this template
         * This should be called when checking for new events
         */
        suspend fun checkAndClearMuteOnNewEvent(
            context: Context,
            templateId: String,
            isConcentration: Boolean = false,
            calendarRepository: CalendarRepository,
            preferencesRepository: PreferencesRepository
        ) {
            val prefs = context.getSharedPreferences(PREFS_NAME, Context.MODE_PRIVATE)
            val muteKey = if (isConcentration) "${templateId}_concentration" else templateId
            
            val muteUntilRelog = prefs.getBoolean("$KEY_MUTE_UNTIL_RELOG$muteKey", false)
            if (!muteUntilRelog) return
            
            // Get the last event time when mute was set
            val storedLastEventTime = prefs.getLong("$KEY_MUTE_LAST_EVENT_TIME$muteKey", 0L)
            
            // Get current last event time
            val templates = preferencesRepository.templates.first()
            val template = templates.find { it.id == templateId } ?: return
            val selectedCalendarId = preferencesRepository.selectedCalendarId.first() ?: return
            val events = calendarRepository.getEventsForPeriod(selectedCalendarId, 24 * 7)
            
            // Use unified matching logic
            val relevantEvents = EventMatching.filterEventsForTemplate(events, template)
            
            val currentLastEvent = relevantEvents.maxByOrNull { it.startTime }
            val currentLastEventTime = currentLastEvent?.startTime?.time ?: 0L
            
            // If there's a new event since the mute was set, clear the mute
            if (currentLastEventTime > storedLastEventTime) {
                Log.d(TAG, "New event detected for $muteKey, clearing mute until relog")
                prefs.edit()
                    .remove("$KEY_MUTE_UNTIL_RELOG$muteKey")
                    .remove("$KEY_MUTE_LAST_EVENT_TIME$muteKey")
                    .apply()
            }
        }
    }

    override fun onReceive(context: Context, intent: Intent) {
        val templateId = intent.getStringExtra(EXTRA_TEMPLATE_ID) ?: return
        val notificationId = intent.getIntExtra(EXTRA_NOTIFICATION_ID, -1)
        val isConcentration = intent.getBooleanExtra(EXTRA_IS_CONCENTRATION, false)
        
        val muteKey = if (isConcentration) "${templateId}_concentration" else templateId
        
        when (intent.action) {
            ACTION_MUTE_24H -> {
                // Mute for 24 hours
                val muteUntil = System.currentTimeMillis() + (24 * 60 * 60 * 1000L)
                val prefs = context.getSharedPreferences(PREFS_NAME, Context.MODE_PRIVATE)
                prefs.edit().putLong("$KEY_MUTE_24H_UNTIL$muteKey", muteUntil).apply()
                
                Log.d(TAG, "Muted template $muteKey for 24h")
                
                // Cancel the notification
                if (notificationId != -1) {
                    val notificationManager = context.getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager
                    notificationManager.cancel(notificationId)
                }
            }
            
            ACTION_MUTE_UNTIL_RELOG -> {
                // Mute until next log entry
                val prefs = context.getSharedPreferences(PREFS_NAME, Context.MODE_PRIVATE)
                
                // Store current last event time so we can detect new events
                CoroutineScope(Dispatchers.IO + SupervisorJob()).launch {
                    try {
                        val preferencesRepository = PreferencesRepository(context)
                        val calendarRepository = CalendarRepository(context.contentResolver, context)
                        
                        val templates = preferencesRepository.templates.first()
                        val template = templates.find { it.id == templateId }
                        
                        if (template != null) {
                            val selectedCalendarId = preferencesRepository.selectedCalendarId.first()
                            if (selectedCalendarId != null) {
                                val events = calendarRepository.getEventsForPeriod(selectedCalendarId, 24 * 7)
                                
                                // Use unified matching logic
                                val relevantEvents = EventMatching.filterEventsForTemplate(events, template)
                                
                                val lastEvent = relevantEvents.maxByOrNull { it.startTime }
                                val lastEventTime = lastEvent?.startTime?.time ?: 0L
                                
                                prefs.edit()
                                    .putBoolean("$KEY_MUTE_UNTIL_RELOG$muteKey", true)
                                    .putLong("$KEY_MUTE_LAST_EVENT_TIME$muteKey", lastEventTime)
                                    .apply()
                                
                                Log.d(TAG, "Muted template $muteKey until relog (last event: $lastEventTime)")
                            }
                        }
                    } catch (e: Exception) {
                        Log.e(TAG, "Error setting mute until relog", e)
                    }
                }
                
                // Cancel the notification
                if (notificationId != -1) {
                    val notificationManager = context.getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager
                    notificationManager.cancel(notificationId)
                }
            }
            
            ACTION_LOG -> {
                val requiresInput = intent.getBooleanExtra(EXTRA_REQUIRES_INPUT, false)
                val templateName = intent.getStringExtra(EXTRA_TEMPLATE_NAME) ?: "medication"
                
                if (requiresInput) {
                    // Extract RemoteInput value
                    val remoteInput = RemoteInput.getResultsFromIntent(intent)
                    val inputText = remoteInput?.getCharSequence(KEY_TEXT_REPLY)?.toString()
                    
                    if (inputText != null) {
                        // Parse the input value and log
                        val value = inputText.toDoubleOrNull()
                        if (value != null) {
                            CoroutineScope(Dispatchers.IO + SupervisorJob()).launch {
                                try {
                                    val preferencesRepository = PreferencesRepository(context)
                                    val calendarRepository = CalendarRepository(context.contentResolver, context)
                                    
                                    val templates = preferencesRepository.templates.first()
                                    val template = templates.find { it.id == templateId }
                                    
                                    if (template != null) {
                                        val selectedCalendarId = preferencesRepository.selectedCalendarId.first()
                                        if (selectedCalendarId != null) {
                                            // Create event with the input value
                                            val eventId = calendarRepository.createEventFromTemplate(selectedCalendarId, template, value)
                                            
                                            if (eventId != null) {
                                                Log.d(TAG, "Successfully logged template ${template.name} with value $value from notification")
                                                
                                                // Update the original notification to show confirmation with Dismiss button
                                                if (notificationId != -1) {
                                                    val notificationManager = context.getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager
                                                    
                                                    // Create dismiss action for the SAME notification ID
                                                    val dismissIntent = Intent(context, NotificationActionReceiver::class.java).apply {
                                                        action = ACTION_DISMISS
                                                        putExtra(EXTRA_NOTIFICATION_ID, notificationId)
                                                    }
                                                    val dismissPendingIntent = android.app.PendingIntent.getBroadcast(
                                                        context,
                                                        notificationId,
                                                        dismissIntent,
                                                        android.app.PendingIntent.FLAG_UPDATE_CURRENT or android.app.PendingIntent.FLAG_IMMUTABLE
                                                    )
                                                    
                                                    // Build UPDATED notification (replaces the original)
                                                    val channelId = if (isConcentration) "medication_reminders_updates" else "medication_reminders_initial"
                                                    val updatedNotification = androidx.core.app.NotificationCompat.Builder(context, channelId)
                                                        .setSmallIcon(de.ntdote.medicalcalendarlog.R.drawable.ic_launcher_foreground)
                                                        .setContentTitle(context.getString(de.ntdote.medicalcalendarlog.R.string.notification_logged_title, templateName))
                                                        .setContentText(context.getString(de.ntdote.medicalcalendarlog.R.string.notification_logged_value, value.toString()))
                                                        .setAutoCancel(true)
                                                        .setOnlyAlertOnce(true)
                                                        .addAction(
                                                            de.ntdote.medicalcalendarlog.R.drawable.ic_launcher_foreground,
                                                            context.getString(de.ntdote.medicalcalendarlog.R.string.notification_dismiss),
                                                            dismissPendingIntent
                                                        )
                                                        .build()
                                                    
                                                    // Update the existing notification (same ID)
                                                    notificationManager.notify(notificationId, updatedNotification)
                                                    
                                                    // Schedule auto-dismiss after 0.5 seconds
                                                    android.os.Handler(context.mainLooper).postDelayed({
                                                        notificationManager.cancel(notificationId)
                                                        Log.d(TAG, "Auto-dismissed notification $notificationId")
                                                    }, 500)
                                                }
                                            } else {
                                                Log.e(TAG, "Failed to log template ${template.name} from notification")
                                            }
                                        }
                                    }
                                } catch (e: Exception) {
                                    Log.e(TAG, "Error logging from notification with input", e)
                                }
                            }
                        } else {
                            Log.e(TAG, "Invalid input value: $inputText")
                        }
                    } else {
                        Log.e(TAG, "No RemoteInput text found")
                    }
                } else {
                    // Directly log for templates without input
                    CoroutineScope(Dispatchers.IO + SupervisorJob()).launch {
                        try {
                            val preferencesRepository = PreferencesRepository(context)
                            val calendarRepository = CalendarRepository(context.contentResolver, context)
                            
                            val templates = preferencesRepository.templates.first()
                            val template = templates.find { it.id == templateId }
                            
                            if (template != null) {
                                val selectedCalendarId = preferencesRepository.selectedCalendarId.first()
                                if (selectedCalendarId != null) {
                                    // Directly create event without input
                                    val eventId = calendarRepository.createEventFromTemplate(selectedCalendarId, template)
                                    
                                    if (eventId != null) {
                                        Log.d(TAG, "Successfully logged template ${template.name} from notification")
                                        
                                        // Cancel the notification after logging
                                        if (notificationId != -1) {
                                            val notificationManager = context.getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager
                                            notificationManager.cancel(notificationId)
                                        }
                                    } else {
                                        Log.e(TAG, "Failed to log template ${template.name} from notification")
                                    }
                                }
                            }
                        } catch (e: Exception) {
                            Log.e(TAG, "Error logging from notification", e)
                        }
                    }
                }
            }
            
            ACTION_DISMISS -> {
                // Simply dismiss the notification
                if (notificationId != -1) {
                    val notificationManager = context.getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager
                    notificationManager.cancel(notificationId)
                    Log.d(TAG, "Dismissed notification $notificationId")
                }
            }
        }
    }
}
