package de.ntdote.medicalcalendarlog.service

import android.Manifest
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.pm.PackageManager
import android.os.Build
import android.util.Log
import androidx.core.content.ContextCompat
import androidx.work.ExistingPeriodicWorkPolicy
import androidx.work.PeriodicWorkRequestBuilder
import androidx.work.WorkManager
import de.ntdote.medicalcalendarlog.repository.PreferencesRepository
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.launch
import java.util.concurrent.TimeUnit

/**
 * BroadcastReceiver that triggers on app install/update and device boot.
 * 
 * Responsibilities:
 * - Run data migrations immediately after app update (even if user doesn't open the app)
 * - Reschedule workers and reminders after device boot
 * - Ensures notifications and widgets continue working after automatic updates
 */
class AppUpdateReceiver : BroadcastReceiver() {
    
    companion object {
        private const val TAG = "MCL"
    }
    
    // Use a background coroutine scope for async work
    private val scope = CoroutineScope(SupervisorJob() + Dispatchers.IO)
    
    override fun onReceive(context: Context, intent: Intent) {
        when (intent.action) {
            Intent.ACTION_MY_PACKAGE_REPLACED -> {
                Log.i(TAG, "App was updated - running migrations and rescheduling services")
                runMigrationsAndReschedule(context, "app_update")
            }
            Intent.ACTION_BOOT_COMPLETED -> {
                Log.i(TAG, "Device booted - verifying migrations and rescheduling services")
                runMigrationsAndReschedule(context, "boot")
            }
        }
    }
    
    /**
     * Check if calendar permissions are granted
     */
    private fun hasCalendarPermissions(context: Context): Boolean {
        val readCalendarPermission = ContextCompat.checkSelfPermission(
            context, 
            Manifest.permission.READ_CALENDAR
        )
        
        val writeCalendarPermission = ContextCompat.checkSelfPermission(
            context, 
            Manifest.permission.WRITE_CALENDAR
        )
        
        return readCalendarPermission == PackageManager.PERMISSION_GRANTED &&
               writeCalendarPermission == PackageManager.PERMISSION_GRANTED
    }
    
    /**
     * Run data migrations and reschedule all services
     */
    private fun runMigrationsAndReschedule(context: Context, reason: String) {
        // Use goAsync() to allow asynchronous work in BroadcastReceiver
        val pendingResult = goAsync()
        
        scope.launch {
            try {
                // Step 1: Run data migrations
                val preferencesRepository = PreferencesRepository(context)
                val currentVersionCode = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) {
                    context.packageManager.getPackageInfo(context.packageName, 0).longVersionCode.toInt()
                } else {
                    @Suppress("DEPRECATION")
                    context.packageManager.getPackageInfo(context.packageName, 0).versionCode
                }
                
                Log.d(TAG, "Running migrations for version $currentVersionCode (reason: $reason)")
                preferencesRepository.migrateTemplatesOnVersionUpgrade(currentVersionCode)
                
                // Step 2: Check permissions before rescheduling services
                if (!hasCalendarPermissions(context)) {
                    Log.w(TAG, "Cannot reschedule services - calendar permissions not granted")
                    return@launch
                }
                
                // Step 3: Reschedule hourly worker
                scheduleHourlyWorker(context)
                
                // Step 4: Trigger reminder recalculation (ContentObserver is already registered in Application class)
                ReminderService.recalculateAllReminders(context, reason)
                
                Log.i(TAG, "Migrations and service rescheduling completed (reason: $reason)")
            } catch (e: Exception) {
                Log.e(TAG, "Error during migration/rescheduling", e)
            } finally {
                // Signal that async work is complete
                pendingResult.finish()
            }
        }
    }
    
    /**
     * Schedule the hourly worker to run periodically
     */
    private fun scheduleHourlyWorker(context: Context) {
        val hourlyWorkRequest = PeriodicWorkRequestBuilder<HourlyReminderWorker>(
            1, TimeUnit.HOURS, // Run every hour
            15, TimeUnit.MINUTES // Flex period of 15 minutes
        )
            .addTag("hourly_reminder_worker")
            .build()
        
        // Enqueue the work request, replacing any existing one
        WorkManager.getInstance(context).enqueueUniquePeriodicWork(
            "hourly_reminder_worker",
            ExistingPeriodicWorkPolicy.REPLACE,
            hourlyWorkRequest
        )
        
        Log.d(TAG, "Hourly reminder worker scheduled")
    }
}
