package de.ntdote.medicalcalendarlog.repository

import android.content.Context
import androidx.datastore.core.DataStore
import androidx.datastore.preferences.core.Preferences
import androidx.datastore.preferences.core.edit
import androidx.datastore.preferences.core.stringPreferencesKey
import androidx.datastore.preferences.preferencesDataStore
import de.ntdote.medicalcalendarlog.data.WidgetConfig
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.map
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json

private val Context.widgetDataStore: DataStore<Preferences> by preferencesDataStore(name = "widget_settings")

/**
 * Repository for managing widget configurations
 * Stores mapping of widget IDs to widget configurations
 */
class WidgetRepository(private val context: Context) {
    
    private val json = Json { ignoreUnknownKeys = true }
    
    /**
     * Save widget configuration
     * @param widgetId The unique ID of the widget instance
     * @param config The widget configuration
     */
    suspend fun saveWidgetConfig(widgetId: Int, config: WidgetConfig) {
        val key = stringPreferencesKey("widget_$widgetId")
        context.widgetDataStore.edit { preferences ->
            preferences[key] = json.encodeToString(config)
        }
    }
    
    /**
     * Get widget configuration
     * @param widgetId The widget ID
     * @return The widget configuration or null if not configured
     */
    fun getWidgetConfig(widgetId: Int): Flow<WidgetConfig?> {
        val key = stringPreferencesKey("widget_$widgetId")
        return context.widgetDataStore.data.map { preferences ->
            preferences[key]?.let { configJson ->
                try {
                    json.decodeFromString<WidgetConfig>(configJson)
                } catch (e: Exception) {
                    // Handle legacy format (just templateId string)
                    WidgetConfig(templateId = configJson)
                }
            }
        }
    }
    
    /**
     * Get widget configuration synchronously
     */
    suspend fun getWidgetConfigSync(widgetId: Int): WidgetConfig? {
        val key = stringPreferencesKey("widget_$widgetId")
        var config: WidgetConfig? = null
        context.widgetDataStore.edit { preferences ->
            config = preferences[key]?.let { configJson ->
                try {
                    json.decodeFromString<WidgetConfig>(configJson)
                } catch (e: Exception) {
                    // Handle legacy format (just templateId string)
                    WidgetConfig(templateId = configJson)
                }
            }
        }
        return config
    }
    
    /**
     * Get template ID for a widget (for backwards compatibility)
     * @param widgetId The widget ID
     * @return The template ID or null if not configured
     */
    fun getWidgetTemplateId(widgetId: Int): Flow<String?> {
        return getWidgetConfig(widgetId).map { it?.templateId }
    }
    
    /**
     * Get template ID for a widget synchronously (for backwards compatibility)
     */
    suspend fun getWidgetTemplateIdSync(widgetId: Int): String? {
        return getWidgetConfigSync(widgetId)?.templateId
    }
    
    /**
     * Remove widget configuration when widget is deleted
     * @param widgetId The widget ID to remove
     */
    suspend fun removeWidgetConfig(widgetId: Int) {
        val key = stringPreferencesKey("widget_$widgetId")
        context.widgetDataStore.edit { preferences ->
            preferences.remove(key)
        }
    }
    
    /**
     * Get all configured widget IDs
     */
    fun getAllWidgetIds(): Flow<List<Int>> {
        return context.widgetDataStore.data.map { preferences ->
            preferences.asMap().keys
                .mapNotNull { key ->
                    if (key.name.startsWith("widget_")) {
                        key.name.removePrefix("widget_").toIntOrNull()
                    } else null
                }
        }
    }
}
