package de.ntdote.medicalcalendarlog.repository

import android.content.Context
import androidx.datastore.core.DataStore
import androidx.datastore.preferences.core.Preferences
import androidx.datastore.preferences.core.booleanPreferencesKey
import androidx.datastore.preferences.core.edit
import androidx.datastore.preferences.core.intPreferencesKey
import androidx.datastore.preferences.core.longPreferencesKey
import androidx.datastore.preferences.core.stringPreferencesKey
import androidx.datastore.preferences.preferencesDataStore
import de.ntdote.medicalcalendarlog.data.Template
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.map
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json

val Context.dataStore: DataStore<Preferences> by preferencesDataStore(name = "settings")

class PreferencesRepository(private val context: Context) {
    
    companion object {
        private val SELECTED_CALENDAR_ID = longPreferencesKey("selected_calendar_id")
        private val DAYS_BACK = intPreferencesKey("days_back")
        private val TEMPLATES = stringPreferencesKey("templates")
        // Keeping the same key name for backward compatibility but renaming the variable
        private val DEBUG_MODE_ENABLED = booleanPreferencesKey("reminder_toasts_enabled")
        private val PRIVACY_POPUP_DISMISS_TIMESTAMP = longPreferencesKey("privacy_popup_dismiss_timestamp")
        private val LAST_MIGRATED_VERSION = intPreferencesKey("last_migrated_version")
        private val LAST_WORKER_RUN_TIMESTAMP = longPreferencesKey("last_worker_run_timestamp")
        private val LAST_WORKER_RUN_TYPE = stringPreferencesKey("last_worker_run_type")
        private val SECOND_LAST_WORKER_RUN_TIMESTAMP = longPreferencesKey("second_last_worker_run_timestamp")
        private val SECOND_LAST_WORKER_RUN_TYPE = stringPreferencesKey("second_last_worker_run_type")
        private val NEXT_REMINDER_TIMESTAMP = longPreferencesKey("next_reminder_timestamp")
        private val NEXT_REMINDER_TEMPLATE_NAME = stringPreferencesKey("next_reminder_template_name")
        private val WINDOW_START_TIMESTAMP = longPreferencesKey("window_start_timestamp")
        private val WINDOW_LAST_UPDATED = longPreferencesKey("window_last_updated")
        
        // 24 hours in milliseconds
        private const val DISMISS_DURATION_MS = 24 * 60 * 60 * 1000L
        // 10 days in milliseconds - window advance interval
        private const val WINDOW_ADVANCE_INTERVAL_MS = 10 * 24 * 60 * 60 * 1000L
    }

    val selectedCalendarId: Flow<Long?> = context.dataStore.data.map { preferences ->
        preferences[SELECTED_CALENDAR_ID]
    }

    val daysBack: Flow<Int> = context.dataStore.data.map { preferences ->
        preferences[DAYS_BACK] ?: 90
    }

    val templates: Flow<List<Template>> = context.dataStore.data.map { preferences ->
        val templatesJson = preferences[TEMPLATES] ?: "[]"
        try {
            Json.decodeFromString<List<Template>>(templatesJson)
        } catch (e: Exception) {
            emptyList()
        }
    }

    val debugModeEnabled: Flow<Boolean> = context.dataStore.data.map { preferences ->
        preferences[DEBUG_MODE_ENABLED] ?: false // Default: disabled
    }

    val shouldShowPrivacyPopup: Flow<Boolean> = context.dataStore.data.map { preferences ->
        val dismissTimestamp = preferences[PRIVACY_POPUP_DISMISS_TIMESTAMP] ?: 0L
        
        // If timestamp is 0, popup has never been dismissed - show it
        if (dismissTimestamp == 0L) {
            return@map true
        }
        
        // If timestamp is Long.MAX_VALUE, user chose "don't show again" - don't show
        if (dismissTimestamp == Long.MAX_VALUE) {
            return@map false
        }
        
        // Check if 24 hours have passed since dismissal
        val currentTime = System.currentTimeMillis()
        val timeSinceDismiss = currentTime - dismissTimestamp
        
        // Show popup if more than 24 hours have passed
        timeSinceDismiss >= DISMISS_DURATION_MS
    }
    
    val lastWorkerRunTimestamp: Flow<Long> = context.dataStore.data.map { preferences ->
        preferences[LAST_WORKER_RUN_TIMESTAMP] ?: 0L
    }
    
    val lastWorkerRunType: Flow<String> = context.dataStore.data.map { preferences ->
        preferences[LAST_WORKER_RUN_TYPE] ?: ""
    }
    
    val secondLastWorkerRunTimestamp: Flow<Long> = context.dataStore.data.map { preferences ->
        preferences[SECOND_LAST_WORKER_RUN_TIMESTAMP] ?: 0L
    }
    
    val secondLastWorkerRunType: Flow<String> = context.dataStore.data.map { preferences ->
        preferences[SECOND_LAST_WORKER_RUN_TYPE] ?: ""
    }
    
    val nextReminderTimestamp: Flow<Long> = context.dataStore.data.map { preferences ->
        preferences[NEXT_REMINDER_TIMESTAMP] ?: 0L
    }
    
    val nextReminderTemplateName: Flow<String> = context.dataStore.data.map { preferences ->
        preferences[NEXT_REMINDER_TEMPLATE_NAME] ?: ""
    }
    
    val windowStartTimestamp: Flow<Long> = context.dataStore.data.map { preferences ->
        preferences[WINDOW_START_TIMESTAMP] ?: 0L
    }
    
    val windowLastUpdated: Flow<Long> = context.dataStore.data.map { preferences ->
        preferences[WINDOW_LAST_UPDATED] ?: 0L
    }

    suspend fun setSelectedCalendarId(calendarId: Long) {
        context.dataStore.edit { preferences ->
            preferences[SELECTED_CALENDAR_ID] = calendarId
        }
    }

    suspend fun setDaysBack(days: Int) {
        context.dataStore.edit { preferences ->
            preferences[DAYS_BACK] = days
            // Reset window when daysBack changes to force recalculation
            preferences[WINDOW_START_TIMESTAMP] = 0L
            preferences[WINDOW_LAST_UPDATED] = 0L
            android.util.Log.d("MCL", "PreferencesRepository: daysBack changed to $days, reset window timestamps")
        }
    }


    suspend fun setDebugModeEnabled(enabled: Boolean) {
        context.dataStore.edit { preferences ->
            preferences[DEBUG_MODE_ENABLED] = enabled
        }
    }

    suspend fun dismissPrivacyPopup(permanently: Boolean = false) {
        context.dataStore.edit { preferences ->
            if (permanently) {
                // Store Long.MAX_VALUE to indicate permanent dismissal
                preferences[PRIVACY_POPUP_DISMISS_TIMESTAMP] = Long.MAX_VALUE
            } else {
                // Store current timestamp for 24-hour dismissal
                preferences[PRIVACY_POPUP_DISMISS_TIMESTAMP] = System.currentTimeMillis()
            }
        }
    }

    /**
     * Reset privacy popup timestamp to force users to see it again.
     * Used during migrations when important disclaimer content is updated.
     */
    suspend fun resetPrivacyPopupForMigration() {
        context.dataStore.edit { preferences ->
            preferences[PRIVACY_POPUP_DISMISS_TIMESTAMP] = 0L
            android.util.Log.d("MCL", "PreferencesRepository: Privacy popup reset for migration - users will see updated disclaimer")
        }
    }

    suspend fun saveTemplates(templates: List<Template>) {
        context.dataStore.edit { preferences ->
            preferences[TEMPLATES] = Json.encodeToString(templates)
        }
    }

    suspend fun addTemplate(template: Template) {
        context.dataStore.edit { preferences ->
            val currentTemplatesJson = preferences[TEMPLATES] ?: "[]"
            val currentTemplates = try {
                Json.decodeFromString<List<Template>>(currentTemplatesJson).toMutableList()
            } catch (e: Exception) {
                mutableListOf()
            }
            
            currentTemplates.add(template)
            preferences[TEMPLATES] = Json.encodeToString(currentTemplates)
        }
    }

    suspend fun removeTemplate(templateId: String) {
        context.dataStore.edit { preferences ->
            val currentTemplatesJson = preferences[TEMPLATES] ?: "[]"
            val currentTemplates = try {
                Json.decodeFromString<List<Template>>(currentTemplatesJson)
            } catch (e: Exception) {
                emptyList()
            }
            
            val updatedTemplates = currentTemplates.filter { it.id != templateId }
            preferences[TEMPLATES] = Json.encodeToString(updatedTemplates)
        }
    }

    suspend fun updateTemplate(template: Template) {
        context.dataStore.edit { preferences ->
            val currentTemplatesJson = preferences[TEMPLATES] ?: "[]"
            val currentTemplates = try {
                Json.decodeFromString<List<Template>>(currentTemplatesJson).toMutableList()
            } catch (e: Exception) {
                mutableListOf()
            }
            
            val index = currentTemplates.indexOfFirst { it.id == template.id }
            if (index != -1) {
                currentTemplates[index] = template
            } else {
                currentTemplates.add(template)
            }
            
            preferences[TEMPLATES] = Json.encodeToString(currentTemplates)
        }
    }

    suspend fun replaceAllTemplates(templates: List<Template>) {
        context.dataStore.edit { preferences ->
            preferences[TEMPLATES] = Json.encodeToString(templates)
        }
    }
    
    suspend fun setLastWorkerRunTimestamp(timestamp: Long) {
        context.dataStore.edit { preferences ->
            preferences[LAST_WORKER_RUN_TIMESTAMP] = timestamp
        }
    }
    
    /**
     * Record a worker run, shifting the previous last run to second-last
     */
    suspend fun recordWorkerRun(timestamp: Long, type: String) {
        context.dataStore.edit { preferences ->
            // Shift current last run to second-last
            val currentLastTimestamp = preferences[LAST_WORKER_RUN_TIMESTAMP] ?: 0L
            val currentLastType = preferences[LAST_WORKER_RUN_TYPE] ?: ""
            
            if (currentLastTimestamp > 0) {
                preferences[SECOND_LAST_WORKER_RUN_TIMESTAMP] = currentLastTimestamp
                preferences[SECOND_LAST_WORKER_RUN_TYPE] = currentLastType
            }
            
            // Store new last run
            preferences[LAST_WORKER_RUN_TIMESTAMP] = timestamp
            preferences[LAST_WORKER_RUN_TYPE] = type
        }
    }
    
    suspend fun setNextReminderTimestamp(timestamp: Long) {
        context.dataStore.edit { preferences ->
            preferences[NEXT_REMINDER_TIMESTAMP] = timestamp
        }
    }
    
    suspend fun setNextReminderInfo(timestamp: Long, templateName: String) {
        context.dataStore.edit { preferences ->
            preferences[NEXT_REMINDER_TIMESTAMP] = timestamp
            preferences[NEXT_REMINDER_TEMPLATE_NAME] = templateName
        }
    }
    
    /**
     * Migrate templates on app version upgrade using centralized DataMigrations framework
     */
    suspend fun migrateTemplatesOnVersionUpgrade(currentVersionCode: Int) {
        var shouldResetPopup = false
        var lastMigratedVersion = 0
        
        context.dataStore.edit { preferences ->
            lastMigratedVersion = preferences[LAST_MIGRATED_VERSION] ?: 0
            
            android.util.Log.d("MCL", "PreferencesRepository: migrateTemplatesOnVersionUpgrade called - lastMigrated=$lastMigratedVersion, current=$currentVersionCode")
            
            // Skip if already migrated to current or newer version
            if (lastMigratedVersion >= currentVersionCode) {
                android.util.Log.d("MCL", "PreferencesRepository: Skipping migration - already at version $lastMigratedVersion (current=$currentVersionCode)")
                return@edit
            }
            
            val currentTemplatesJson = preferences[TEMPLATES] ?: "[]"
            val currentTemplates = try {
                Json.decodeFromString<List<Template>>(currentTemplatesJson)
            } catch (e: Exception) {
                emptyList()
            }
            
            android.util.Log.d("MCL", "PreferencesRepository: Starting migration with ${currentTemplates.size} templates")
            
            // Apply migrations using centralized framework
            val migratedTemplates = de.ntdote.medicalcalendarlog.utils.DataMigrations.migrateTemplates(
                templates = currentTemplates,
                sourceVersion = lastMigratedVersion,
                targetVersion = currentVersionCode
            )
            
            // Save migrated templates if changes were made
            if (migratedTemplates != currentTemplates) {
                android.util.Log.d("MCL", "PreferencesRepository: Templates changed after migration, saving ${migratedTemplates.size} templates")
                preferences[TEMPLATES] = Json.encodeToString(migratedTemplates)
            } else {
                android.util.Log.d("MCL", "PreferencesRepository: No template changes after migration")
            }
            
            // Check if we need to reset privacy popup (migrating to version 38+)
            if (lastMigratedVersion < 38 && currentVersionCode >= 38) {
                shouldResetPopup = true
            }
            
            // Update last migrated version
            preferences[LAST_MIGRATED_VERSION] = currentVersionCode
            android.util.Log.d("MCL", "PreferencesRepository: Updated last migrated version to $currentVersionCode")
        }
        
        // Reset privacy popup if needed (must be done outside the edit block)
        if (shouldResetPopup) {
            android.util.Log.d("MCL", "PreferencesRepository: Migration to version 38+ detected, resetting privacy popup")
            resetPrivacyPopupForMigration()
        }
    }
    
    /**
     * Initialize or update the window start timestamp.
     * Returns the current window start timestamp (newly set or existing).
     */
    suspend fun updateWindowIfNeeded(currentTime: Long, daysBack: Int): Long {
        var resultTimestamp = 0L
        context.dataStore.edit { preferences ->
            val windowStart = preferences[WINDOW_START_TIMESTAMP] ?: 0L
            val windowLastUpdated = preferences[WINDOW_LAST_UPDATED] ?: 0L
            
            // Initialize window if not set (first run or after reset)
            if (windowStart == 0L) {
                val newWindowStart = currentTime - (daysBack * 24 * 60 * 60 * 1000L)
                preferences[WINDOW_START_TIMESTAMP] = newWindowStart
                preferences[WINDOW_LAST_UPDATED] = currentTime
                resultTimestamp = newWindowStart
                android.util.Log.d("MCL", "PreferencesRepository: Initialized window - start=${java.util.Date(newWindowStart)}, daysBack=$daysBack")
                return@edit
            }
            
            // Check if 10 days have passed since last update
            val timeSinceUpdate = currentTime - windowLastUpdated
            if (timeSinceUpdate >= WINDOW_ADVANCE_INTERVAL_MS) {
                // Advance window by 10 days
                val newWindowStart = windowStart + WINDOW_ADVANCE_INTERVAL_MS
                preferences[WINDOW_START_TIMESTAMP] = newWindowStart
                preferences[WINDOW_LAST_UPDATED] = currentTime
                resultTimestamp = newWindowStart
                android.util.Log.d("MCL", "PreferencesRepository: Advanced window - old=${java.util.Date(windowStart)}, new=${java.util.Date(newWindowStart)}, elapsed=${timeSinceUpdate}ms")
            } else {
                // Window is still valid
                resultTimestamp = windowStart
                android.util.Log.d("MCL", "PreferencesRepository: Window still valid - start=${java.util.Date(windowStart)}, lastUpdated=${java.util.Date(windowLastUpdated)}, elapsed=${timeSinceUpdate}ms")
            }
        }
        return resultTimestamp
    }
    
    /**
     * Reset the window (e.g., on reboot or config import).
     * This forces a full recalculation on the next run.
     */
    suspend fun resetWindow() {
        context.dataStore.edit { preferences ->
            preferences[WINDOW_START_TIMESTAMP] = 0L
            preferences[WINDOW_LAST_UPDATED] = 0L
            android.util.Log.d("MCL", "PreferencesRepository: Window reset - will reinitialize on next update")
        }
    }
}
