package de.ntdote.medicalcalendarlog.navigation

import androidx.compose.runtime.Composable
import androidx.compose.ui.platform.LocalContext
import androidx.lifecycle.viewmodel.compose.viewModel
import androidx.navigation.NavHostController
import androidx.navigation.compose.NavHost
import androidx.navigation.compose.composable
import de.ntdote.medicalcalendarlog.repository.CalendarRepository
import de.ntdote.medicalcalendarlog.repository.PreferencesRepository
import de.ntdote.medicalcalendarlog.ui.screens.ConfigurationScreen
import de.ntdote.medicalcalendarlog.ui.screens.GraphScreen
import de.ntdote.medicalcalendarlog.ui.screens.HelpScreen
import de.ntdote.medicalcalendarlog.ui.screens.MainScreen
import de.ntdote.medicalcalendarlog.viewmodel.MainViewModel

sealed class Screen(val route: String) {
    object Main : Screen("main")
    object Configuration : Screen("configuration")
    object Graph : Screen("graph")
    object Help : Screen("help")
}

@Composable
fun MedicalCalendarNavigation(navController: NavHostController) {
    val context = LocalContext.current
    val viewModel: MainViewModel = viewModel {
        MainViewModel(
            application = context.applicationContext as android.app.Application,
            calendarRepository = CalendarRepository(context.contentResolver, context),
            preferencesRepository = PreferencesRepository(context)
        )
    }
    
    NavHost(
        navController = navController,
        startDestination = Screen.Main.route
    ) {
        composable(Screen.Main.route) {
            MainScreen(
                onNavigateToConfig = {
                    navController.navigate(Screen.Configuration.route)
                },
                onNavigateToGraph = {
                    navController.navigate(Screen.Graph.route)
                },
                onNavigateToHelp = {
                    navController.navigate(Screen.Help.route)
                }
            )
        }
        
        composable(Screen.Configuration.route) {
            ConfigurationScreen(
                onNavigateBack = {
                    navController.popBackStack()
                }
            )
        }
        
        composable(Screen.Graph.route) {
            GraphScreen(
                onNavigateBack = {
                    navController.popBackStack()
                },
                viewModel = viewModel
            )
        }
        
        composable(Screen.Help.route) {
            HelpScreen(
                onNavigateBack = {
                    navController.popBackStack()
                }
            )
        }
    }
}
