package de.ntdote.medicalcalendarlog

import android.app.Application
import android.content.pm.PackageManager
import android.database.ContentObserver
import android.net.Uri
import android.os.Build
import android.os.Handler
import android.os.Looper
import android.provider.CalendarContract
import android.util.Log
import androidx.work.ExistingPeriodicWorkPolicy
import androidx.work.PeriodicWorkRequestBuilder
import androidx.work.WorkManager
import de.ntdote.medicalcalendarlog.repository.PreferencesRepository
import de.ntdote.medicalcalendarlog.service.HourlyReminderWorker
import de.ntdote.medicalcalendarlog.service.ReminderService
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.launch
import java.util.concurrent.TimeUnit

class MedicalCalendarLogApplication : Application() {
    
    // Application-scoped coroutine scope
    private val applicationScope = CoroutineScope(SupervisorJob() + Dispatchers.IO)
    
    private var calendarObserver: ContentObserver? = null
    
    // Debounce tracking variables
    private var debounceEventCount = 0
    private var debounceFirstEventTime = 0L
    private var noRescheduleUntil = 0L  // Optimization: don't reschedule within first 2s
    
    override fun onCreate() {
        super.onCreate()
        
        val processId = android.os.Process.myPid()
        Log.d("MCL", "*** APPLICATION CREATED - PID: $processId ***")
        
        // Run data migrations on app startup
        runStartupMigrations()
        
        // Register calendar observer if permissions are available
        // Note: The observer will only work if permissions are granted
        registerCalendarObserver()
        
        // Initial reminder calculation is now handled by MainActivity.initializeServices()
        // after permissions are confirmed - this avoids duplicate calculations
    }
    
    /**
     * Run data migrations on app startup when version changes
     */
    private fun runStartupMigrations() {
        applicationScope.launch {
            try {
                val preferencesRepository = PreferencesRepository(this@MedicalCalendarLogApplication)
                val currentVersionCode = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) {
                    packageManager.getPackageInfo(packageName, 0).longVersionCode.toInt()
                } else {
                    @Suppress("DEPRECATION")
                    packageManager.getPackageInfo(packageName, 0).versionCode
                }
                
                Log.d("MCL", "Running startup migrations - current version: $currentVersionCode")
                preferencesRepository.migrateTemplatesOnVersionUpgrade(currentVersionCode)
                Log.d("MCL", "Startup migrations completed")
            } catch (e: Exception) {
                Log.e("MCL", "Error during startup migration", e)
            }
        }
    }

    
    override fun onTerminate() {
        super.onTerminate()
        Log.d("MCL", "Application.onTerminate() - Cleaning up")
        unregisterCalendarObserver()
    }
    
    private fun registerCalendarObserver() {
        // Check if permissions are granted
        if (!hasCalendarPermissions()) {
            Log.w("MCL", "Application: Cannot register calendar observer - permissions not granted")
            return
        }
        
        // Unregister existing observer if any
        unregisterCalendarObserver()
        
        calendarObserver = object : ContentObserver(Handler(Looper.getMainLooper())) {
            override fun onChange(selfChange: Boolean, uri: Uri?) {
                val now = System.currentTimeMillis()
                var shouldReschedule = false
                
                // Track debounce events
                synchronized(this@MedicalCalendarLogApplication) {
                    if (debounceEventCount == 0) {
                        // First event in this debounce batch
                        debounceFirstEventTime = now
                        noRescheduleUntil = now + 2000L  // Don't reschedule for 2 seconds
                        shouldReschedule = true
                        Log.d("MCL", "Calendar change detected (event #1) - starting debounce")
                    } else {
                        // Subsequent events: check if we're past the no-reschedule window
                        if (now >= noRescheduleUntil) {
                            // Past the 2s window - allow rescheduling and reset the window
                            shouldReschedule = true
                            noRescheduleUntil = now + 2000L  // Reset no-reschedule window
                        }
                    }
                    debounceEventCount++
                }
                
                // Only trigger recalculation if we should reschedule
                if (shouldReschedule) {
                    // Calendar events changed - trigger recalculation
                    // Use applicationScope for proper debouncing (all jobs in same scope can cancel each other)
                    applicationScope.launch {
                        ReminderService.recalculateAllReminders(
                            applicationContext, 
                            "calendar_change",
                            applicationScope,  // Pass the scope for debouncing
                            onDebounceComplete = { 
                                // This callback runs after debounce completes
                                handleDebounceComplete()
                            }
                        )
                    }
                }
            }
        }
        
        try {
            contentResolver.registerContentObserver(
                CalendarContract.Events.CONTENT_URI,
                true,
                calendarObserver!!
            )
            Log.d("MCL", "Application: Calendar observer registered successfully")
        } catch (e: SecurityException) {
            Log.e("MCL", "Application: Security exception when registering calendar observer", e)
        } catch (e: Exception) {
            Log.e("MCL", "Application: Error registering calendar observer", e)
        }
    }
    
    private fun unregisterCalendarObserver() {
        calendarObserver?.let { observer ->
            try {
                contentResolver.unregisterContentObserver(observer)
                Log.d("MCL", "Application: Calendar observer unregistered")
            } catch (e: Exception) {
                Log.e("MCL", "Application: Error unregistering calendar observer", e)
            }
            calendarObserver = null
        }
    }
    
    /**
     * Called after debounce completes to log statistics and reset counters
     */
    private fun handleDebounceComplete() {
        synchronized(this) {
            if (debounceEventCount > 0) {
                val totalDuration = System.currentTimeMillis() - debounceFirstEventTime
                // Subtract the debounce delay (5000ms) to show only the actual event spread time
                val eventSpreadDuration = (totalDuration - 5000L).coerceAtLeast(0L)
                
                Log.d("MCL", "Debounce batch completed - processed $debounceEventCount calendar changes spread over ${eventSpreadDuration}ms")
                
                // Reset counters for next batch
                debounceEventCount = 0
                debounceFirstEventTime = 0L
                noRescheduleUntil = 0L  // Reset no-reschedule window
            }
        }
    }
    
    private fun hasCalendarPermissions(): Boolean {
        val readCalendarPermission = androidx.core.content.ContextCompat.checkSelfPermission(
            this,
            android.Manifest.permission.READ_CALENDAR
        )
        
        val writeCalendarPermission = androidx.core.content.ContextCompat.checkSelfPermission(
            this,
            android.Manifest.permission.WRITE_CALENDAR
        )
        
        return readCalendarPermission == android.content.pm.PackageManager.PERMISSION_GRANTED &&
               writeCalendarPermission == android.content.pm.PackageManager.PERMISSION_GRANTED
    }
    
    /**
     * Schedule the hourly worker to run periodically
     * This is called from MainActivity after permissions are granted
     */
    fun scheduleHourlyWorker() {
        // Check permissions before scheduling
        if (!hasCalendarPermissions()) {
            Log.d("MCL", "Cannot schedule hourly worker - calendar permissions not granted")
            return
        }
        
        val hourlyWorkRequest = PeriodicWorkRequestBuilder<HourlyReminderWorker>(
            1, TimeUnit.HOURS, // Run every hour
            15, TimeUnit.MINUTES // Flex period of 15 minutes
        )
            .addTag("hourly_reminder_worker")
            .build()
        
        // Enqueue the work request, replacing any existing one
        WorkManager.getInstance(this).enqueueUniquePeriodicWork(
            "hourly_reminder_worker",
            ExistingPeriodicWorkPolicy.REPLACE, // Replace existing work if any
            hourlyWorkRequest
        )
        
        Log.d("MCL", "Hourly reminder worker scheduled")
    }
}
