package de.noisruker.openPasskeyAuth.utils.preferences

import androidx.datastore.core.DataStore
import androidx.datastore.core.IOException
import androidx.datastore.preferences.core.Preferences
import androidx.datastore.preferences.core.booleanPreferencesKey
import androidx.datastore.preferences.core.edit
import androidx.datastore.preferences.core.emptyPreferences
import androidx.datastore.preferences.core.stringPreferencesKey
import de.noisruker.openPasskeyAuth.utils.OpenPasskeyAuthUtils
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.catch
import kotlinx.coroutines.flow.distinctUntilChanged
import kotlinx.coroutines.flow.map
import kotlin.coroutines.coroutineContext

class UserManager(private val dataStore: DataStore<Preferences>): IUserPreferenceRepository {
    // Create keys to store and retrieve the data
    companion object Keys {
        val scramblePin = booleanPreferencesKey("scramble_pin")
        val allowScreenshots = booleanPreferencesKey("allow_screenshots")
        val allowKeyboard = booleanPreferencesKey("allow_keyboard")
        val experimentalHardwareKeyCredentialCreation = booleanPreferencesKey("experimental_hardware_key_creation")
        val strongBoxKey = booleanPreferencesKey("strong_box_key")
        val askForStrongBoxOnKeyCreation = booleanPreferencesKey("ask_for_strong_box_on_key_creation")
        val appLock = booleanPreferencesKey("app_lock")
        val appUnlockSecret = stringPreferencesKey("app_unlock_secret")
        val appUnlockIV = stringPreferencesKey("app_unlock_iv")
    }

    private inline val Preferences.scramblePin
        get() = this[Keys.scramblePin] != false

    private inline val Preferences.allowScreenshots
        get() = this[Keys.allowScreenshots] == true

    private inline val Preferences.allowKeyboard
        get() = this[Keys.allowKeyboard] == true

    private inline val Preferences.strongBoxKey
        get() = this[Keys.strongBoxKey] != false

    private inline val Preferences.appLock
        get() = this[Keys.appLock] == true

    private inline val Preferences.appUnlockSecret
        get() = this[Keys.appUnlockSecret]

    private inline val Preferences.askForStrongBoxOnKeyCreation
        get() = this[Keys.askForStrongBoxOnKeyCreation] == true

    private inline val Preferences.appUnlockIV get() = this[Keys.appUnlockIV]

    override val userPreferences: Flow<UserPreferences> = dataStore.data
        .catch {
            if (it is IOException) {
                emit(emptyPreferences())
            } else {
                throw it
            }
        }
        .map { preferences ->
            UserPreferences(
                scramblePin = preferences.scramblePin,
                allowScreenshots = preferences.allowScreenshots,
                askForStrongBoxOnKeyCreation = preferences.askForStrongBoxOnKeyCreation,
                allowKeyboard = preferences.allowKeyboard,
                strongBoxKey = preferences.strongBoxKey,
                appLock = preferences.appLock,
                appUnlockSecret = preferences.appUnlockSecret,
                appUnlockIV = preferences.appUnlockIV,
            )
        }
        .distinctUntilChanged()

    override suspend fun enableScramblePinLayout(enabled: Boolean) {
        dataStore.edit {
            it[Keys.scramblePin] = enabled
        }
    }

    override suspend fun enableScreenshots(enabled: Boolean) {
        dataStore.edit {
            it[Keys.allowScreenshots] = enabled
        }
    }

    override suspend fun allowKeyboard(enabled: Boolean) {
        dataStore.edit {
            it[Keys.allowKeyboard] = enabled
        }
    }

    override suspend fun enableStrongBoxKeys(enabled: Boolean) {
        dataStore.edit {
            it[Keys.strongBoxKey] = enabled
        }
    }

    override suspend fun enableScreenLock(enabled: Boolean) {
        dataStore.edit {
            it[Keys.appLock] = enabled
        }
    }

    override suspend fun setAppUnlockSecret(secret: String) {
        dataStore.edit {
            it[Keys.appUnlockSecret] = secret
        }
    }

    override suspend fun askForStrongBoxOnKeyCreation(enabled: Boolean) {
        dataStore.edit {
            it[Keys.askForStrongBoxOnKeyCreation] = enabled
        }
    }

    override suspend fun appUnlockIV(iv: String) {
        dataStore.edit {
            it[Keys.appUnlockIV] = iv
        }
    }
}