package de.noisruker.openPasskeyAuth.utils.hwkeys

import at.asitplus.catching
import at.asitplus.signum.indispensable.cosef.CoseKey
import at.asitplus.signum.indispensable.cosef.io.coseCompliantSerializer
import com.google.android.gms.fido.fido2.api.common.PublicKeyCredential
import kotlinx.serialization.ExperimentalSerializationApi
import kotlinx.serialization.Serializable
import kotlinx.serialization.cbor.Cbor
import kotlinx.serialization.decodeFromByteArray
import kotlinx.serialization.json.JsonObject
import org.microg.gms.fido.core.protocol.AuthenticatorData
import kotlin.io.encoding.Base64
import kotlin.io.encoding.ExperimentalEncodingApi

@Serializable
data class PublicKeyCredentialData (
    val id: String,
    val type: String,
    val rawId: String,
    val response: AuthenticatorResponse,
    val authenticatorAttachment: String?,
    val clientExtensionResults: JsonObject?,
) {
    companion object {
        @OptIn(ExperimentalSerializationApi::class)
        fun parseMicroGPublicKeyCredentialData(base64: Base64, cbor: Cbor?, publicKeyCreds: PublicKeyCredential): PublicKeyCredentialData? {
            return PublicKeyCredentialData(
                id = publicKeyCreds.id,
                type = publicKeyCreds.type,
                rawId = base64.encode(publicKeyCreds.rawId),
                authenticatorAttachment = publicKeyCreds.authenticatorAttachment,
                response = AuthenticatorResponse.parseMicroGAuthenticatorResponse(base64, cbor, publicKeyCreds.response) ?: return null,
                clientExtensionResults = JsonObject(emptyMap())
            )
        }
    }
}

@Serializable
sealed class AuthenticatorResponse {
    @Serializable
    class AuthenticatorAttestationResponse(
        val keyHandle: String,
        val clientDataJSON: String,
        val attestationObject: String,
        val transports: List<String>,
        val authenticatorData: String? = null,
        val publicKeyAlgorithm: Int? = null,
        val publicKey: String? = null,
    ): AuthenticatorResponse()

    @Serializable
    class AuthenticatorAssertionResponse(
        val clientDataJSON: String,
        val authenticatorData: String,
        val signature: String,
        val userHandle: String?,
    ): AuthenticatorResponse()

    companion object {
        @OptIn(ExperimentalEncodingApi::class, ExperimentalSerializationApi::class)
        fun parseMicroGAuthenticatorResponse(base64: Base64, cbor: Cbor?, res: com.google.android.gms.fido.fido2.api.common.AuthenticatorResponse): AuthenticatorResponse? {
            return when (res) {
                is com.google.android.gms.fido.fido2.api.common.AuthenticatorAttestationResponse -> {
                    val attestation = runCatching { cbor?.decodeFromByteArray<AttestationObjectData>(res.attestationObject) }.getOrNull()
                    val base64AuthData = attestation?.authData?.let { base64.encode(it) }
                    val authData = attestation?.authData?.let { AuthenticatorData.decode(it) }
                    val coseKey = authData?.attestedCredentialData?.publicKey?.let { catching { coseCompliantSerializer.decodeFromByteArray<CoseKey>(it) }.getOrNull() }

                    val pubKeyAlgo = coseKey?.algorithm?.coseValue
                    val pubKey = coseKey?.toCryptoPublicKey()?.getOrNull()?.encodeToDer()?.let { base64.encode(it) }

                    AuthenticatorAttestationResponse(
                        keyHandle = base64.encode(res.keyHandle),
                        clientDataJSON = base64.encode(res.clientDataJSON),
                        attestationObject = base64.encode(res.attestationObject),
                        transports = res.transports.asList(),
                        authenticatorData = base64AuthData,
                        publicKeyAlgorithm = pubKeyAlgo,
                        publicKey = pubKey,
                    )
                }
                is com.google.android.gms.fido.fido2.api.common.AuthenticatorAssertionResponse -> {
                    AuthenticatorAssertionResponse(
                        clientDataJSON = base64.encode(res.clientDataJSON),
                        authenticatorData = base64.encode(res.authenticatorData),
                        signature = base64.encode(res.signature),
                        userHandle = res.userHandle?.let { base64.encode(it) }
                    )
                }
                else -> { null }
            }
        }
    }
}