package de.noisruker.openPasskeyAuth.utils

import android.annotation.SuppressLint
import android.util.Log
import kotlinx.serialization.ExperimentalSerializationApi
import kotlinx.serialization.Serializable
import kotlinx.serialization.cbor.ByteString
import kotlinx.serialization.cbor.Cbor.Default.CoseCompliant
import kotlinx.serialization.cbor.CborLabel
import org.json.JSONObject
import java.nio.ByteBuffer
import java.nio.ByteOrder
import kotlinx.serialization.decodeFromByteArray
import kotlinx.serialization.json.Json

@SuppressLint("RestrictedApi")
@OptIn(ExperimentalSerializationApi::class)
class HardwareKeyCreationResponse(val clientData: ByteArray, private val attestationObject: ByteArray, private val authenticatorData: ByteArray, private val rawId: String, private val authenticatorAttachment: String, private val typeProperty: String) {

    fun formatPublicKeyDer(publicKeyDer: ByteArray): String {
        if (publicKeyDer.isEmpty()) {
            return ""
        }

        // 0.metadata(26byte)
        val metaHeader = hexStringToByteArray("3059301306072a8648ce3d020106082a8648ce3d030107034200")
        val tmp = mutableListOf<Byte>()
        tmp.addAll(metaHeader.toList())
        tmp.addAll(publicKeyDer.toList())

        // 1.encode Base64
        val base64Str = OpenPasskeyAuthUtils.b64Encode(tmp.toByteArray())

        // 2. \n every 64 characters
        val pemBase = StringBuilder()
        var counter = 0
        for (c in base64Str) {
            pemBase.append(c)
            if (counter == 63) {
                pemBase.appendLine()
                counter = 0
            } else {
                counter++
            }
        }
        pemBase.appendLine() // Add a final newline

        // 3. Header and footer
        return "-----BEGIN PUBLIC KEY-----\n$pemBase-----END PUBLIC KEY-----"
    }


    // Helper function to convert hex string to byte array
    private fun hexStringToByteArray(hexString: String): ByteArray {
        val bytes = ByteArray(hexString.length / 2)
        for (i in hexString.indices step 2) {
            val hexValue = hexString.substring(i, i + 2)
            bytes[i / 2] = hexValue.toInt(16).toByte()
        }
        return bytes
    }

    @Serializable
    data class PublicKey(
        @CborLabel(1)
        val keyTypeValue: Int,
        @CborLabel(3)
        val algorithm: Int,
        @CborLabel(-1)
        val curves: Int,
        @CborLabel(-2)
        @ByteString
        val keyBytes1: ByteArray,
        @CborLabel(-3)
        @ByteString
        val keyBytes2: ByteArray
    ) {
        override fun equals(other: Any?): Boolean {
            if (this === other) return true
            if (javaClass != other?.javaClass) return false

            other as PublicKey

            if (keyTypeValue != other.keyTypeValue) return false
            if (algorithm != other.algorithm) return false
            if (curves != other.curves) return false
            if (!keyBytes1.contentEquals(other.keyBytes1)) return false
            if (!keyBytes2.contentEquals(other.keyBytes2)) return false

            return true
        }

        override fun hashCode(): Int {
            var result = keyTypeValue
            result = 31 * result + algorithm
            result = 31 * result + curves
            result = 31 * result + keyBytes1.contentHashCode()
            result = 31 * result + keyBytes2.contentHashCode()
            return result
        }

        @OptIn(ExperimentalStdlibApi::class)
        override fun toString(): String {
            val json = JSONObject()
            json.put("1", keyTypeValue)
            json.put("3", algorithm)
            json.put("-1", curves)
            json.put("-2", keyBytes1.toHexString())
            json.put("-3", keyBytes2.toHexString())
            return json.toString()
        }

        fun toPublicKeyDer(): ByteArray {
            val pubKey = ArrayList<Byte>()
            if (keyTypeValue == 1) {
                pubKey.addAll(keyBytes1.toList())
            } else if (keyTypeValue == 2) {
                pubKey.add(0x04)
                pubKey.addAll(keyBytes1.toList())
                pubKey.addAll(keyBytes2.toList())
            }
            return pubKey.toByteArray()
        }
    }
    

    @OptIn(ExperimentalSerializationApi::class)
    private fun getPublicKeyFromAuthData(): PublicKey {
        var index = 0
        // First 32 Bytes are rpIdHash
        index += 32
        // 1 Byte Flags
        index += 1
        // 4 Bytes Sign Count
        index += 4
        // 16 Bytes AAGuid
        index += 16
        // 2 Bytes CredentialIdLength
        val rdr = ByteBuffer.wrap(authenticatorData.sliceArray(index until index + 2))
        rdr.order(ByteOrder.BIG_ENDIAN)
        val len = rdr.short.toInt()
        index += 2

        // Len Bytes of credential id
        index += len

        // Get Key
        val slice = authenticatorData.sliceArray(index until authenticatorData.size)

        val key = CoseCompliant.decodeFromByteArray<PublicKey>(slice)

        Log.d("OPA", "Decoded Data: $key")

        return key
    }
    @Serializable
    private data class CreatePublicKeyCredentialResponseJson(
        //RegistrationResponseJSON
        val id:String,
        val rawId: String,
        val response: Response,
        val authenticatorAttachment: String?,
        val clientExtensionResults: EmptyClass = EmptyClass(),
        val type: String,
    ) {
        @Serializable
        data class Response(
            //AuthenticatorAttestationResponseJSON
            val clientDataJSON: String? = null,
            var authenticatorData: String? = null,
            val transports: List<String>? = arrayOf("internal").toList(),
            var publicKey: String? = null, // easy accessors fields
            var publicKeyAlgorithm: Long? = null, // easy accessors fields
            val attestationObject: String? // easy accessors fields
        )
        @Serializable
        class EmptyClass
    }

    fun json(): String {


        val pubKey = getPublicKeyFromAuthData()

        val response = CreatePublicKeyCredentialResponseJson.Response(
            clientDataJSON = OpenPasskeyAuthUtils.b64Encode(clientData),
            authenticatorData = OpenPasskeyAuthUtils.b64Encode(authenticatorData),
            transports = listOf("internal", "hybrid"),
            publicKey = OpenPasskeyAuthUtils.b64Encode(pubKey.toPublicKeyDer()),
            publicKeyAlgorithm = pubKey.algorithm.toLong(),
            attestationObject = OpenPasskeyAuthUtils.b64Encode(attestationObject),
        )

        val fullResponse = CreatePublicKeyCredentialResponseJson(
            //RegistrationResponseJSON
            id = rawId,
            rawId,
            response,
            authenticatorAttachment,
            type = typeProperty,
        )
        /*response.put("publicKey", OpenPasskeyAuthUtils.b64Encode(pubKey.toPublicKeyDer()))
        response.put("clientDataJSON", OpenPasskeyAuthUtils.b64Encode(clientData))
        response.put("authenticatorData", OpenPasskeyAuthUtils.b64Encode(authenticatorData))
        response.put("transports", JSONArray(listOf("internal", "hybrid")))
        response.put("publicKeyAlgorithm", pubKey.algorithm)
        response.put("attestationObject", OpenPasskeyAuthUtils.b64Encode(attestationObject))*/

        return Json.encodeToString(fullResponse)
    }
}