package de.noisruker.openPasskeyAuth.utils

import android.app.Activity
import android.content.Context
import android.content.SharedPreferences
import android.security.keystore.KeyProperties
import android.util.Log
import de.noisruker.openPasskeyAuth.Locator
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.Json
import java.security.KeyStore
import javax.crypto.Cipher
import androidx.core.content.edit
import java.security.SecureRandom
import javax.crypto.spec.IvParameterSpec


object CredentialDataManager {

    fun save(context: Context, credential: Credential) {
        val credSet = loadCredSet(context)

        // convert from Credential to SerializedCred
        //val keyPair:String =
        //    OpenPasskeyAuthUtils.b64Encode(credential.keyPair!!.toByteArray())

        val cred = CredSet.SerializedCred(
            credential.rpid,
            credential.serviceName,
            credential.credentialId,
            credential.userHandle,
            credential.displayName,
            credential.description,
            credential.securityLevel
        )
        credSet.list.add(cred)

        // save dataset
        saveCredSet(context, credSet)
    }

    fun moveCredPos(fromIndex: Int, toIndex: Int, context: Context) {
        val credSet = loadCredSet(context)

        credSet.list.add(toIndex, credSet.list.removeAt(fromIndex))

        saveCredSet(context, credSet)
    }

    fun load(context: Context, rpid: String): MutableList<Credential> {
        val credSet = loadCredSet(context)
        val list:MutableList<Credential> = mutableListOf()
        credSet.list.forEach {
            if (rpid.endsWith(it.rpid)) list.add(convertCred(it))
        }

        return list
    }

    fun loadAll(context: Context): MutableList<Credential> {
        val credSet = loadCredSet(context)
        val list:MutableList<Credential> = mutableListOf()
        credSet.list.forEach {
            list.add(convertCred(it))
        }
        return list
    }

    fun load(context: Context, rpid: String, credentialId: ByteArray): Credential? {
        val credSet = loadCredSet(context)
        credSet.list.forEach {
            if (it.rpid == rpid && it.credentialId.contentEquals(credentialId)) return convertCred(it)
        }
        return null
    }

    fun delete(context: Context, rpid: String, credentialId: ByteArray) {
        val credSet = loadCredSet(context)
        val newCredSet = CredSet()

        credSet.list.forEach {
            if (it.rpid == rpid && it.credentialId.contentEquals(credentialId))  {
                Log.d("OPA", "Credential to delete found. Deleting it from the android keystore as well")

                val keyStore = KeyStore.getInstance("AndroidKeyStore").apply {
                    load(null)
                }

                keyStore.deleteEntry("opa-" + OpenPasskeyAuthUtils.b64Encode(it.credentialId))
            } else {
                newCredSet.list.add(it)
            }
        }

        saveCredSet(context, newCredSet)
    }

    fun updateEncrypt(context: Context) {
        val PREF_KEY = "PREF_CREDENTIAL_SET"
        val PREF_IV = "PREF_CREDENTIAL_DECRYPTION_IV"
        // load dataset
        val sharedPref = context.getSharedPreferences(context.packageName, Activity.MODE_PRIVATE)
        var sharedIV = sharedPref.getString(PREF_IV, "") ?: ""
        val setjson = sharedPref.getString(PREF_KEY, "") ?: ""

        if (sharedIV.isBlank()) {
            val iv = ByteArray(16)
            val secureRandom = SecureRandom()
            secureRandom.nextBytes(iv)
            sharedIV = OpenPasskeyAuthUtils.b64Encode(IvParameterSpec(iv).iv)
        }

        if (!setjson.isEmpty()) {
            val cipher = Cipher.getInstance("AES/CBC/PKCS7PADDING")
            cipher.init(Cipher.ENCRYPT_MODE, Locator.getSecretKey(), OpenPasskeyAuthUtils.getIV(sharedIV))
            val encrypted = cipher.doFinal(setjson.toByteArray())
            sharedPref.edit {
                putString(PREF_KEY, OpenPasskeyAuthUtils.b64Encode(encrypted))
                putString(PREF_IV, sharedIV)
                apply()
            }
        }
        Log.d("OPA", "Encrypted Credentials using shared secret")
    }

    private fun loadCredSet(context: Context): CredSet {
        val PREF_KEY = "PREF_CREDENTIAL_SET"
        val PREF_IV = "PREF_CREDENTIAL_DECRYPTION_IV"

        // load dataset
        val credSet: CredSet
        val sharedPref = context.getSharedPreferences(context.packageName, Activity.MODE_PRIVATE)
        val sharedIV = sharedPref.getString(PREF_IV, "") ?: ""
        val setjson = sharedPref.getString(PREF_KEY, "") ?: ""

        credSet = if (setjson.isEmpty()) {
            CredSet()
        } else {
            val cipher = Cipher.getInstance("AES/CBC/PKCS7PADDING")

            if (Locator.getSecretKey() == null)
                return CredSet()

            cipher.init(Cipher.DECRYPT_MODE, Locator.getSecretKey(), OpenPasskeyAuthUtils.getIV(sharedIV))

            val decrypt = cipher.doFinal(OpenPasskeyAuthUtils.b64Decode(setjson))

            try {
                Json.decodeFromString<CredSet>(String(decrypt))
            } catch (_: Exception) {
                Log.d("OPA", "Failed to read json data: ${String(decrypt)}")
                CredSet()
            }
        }

        return credSet
    }

    private fun convertCred(cred: CredSet.SerializedCred): Credential {
        return Credential(
            cred.rpid,
            cred.serviceName,
            cred.credentialId,
            cred.userHandle,
            cred.displayName,
            cred.description,
            securityLevel = cred.securityLevel,
        )
    }

    private fun saveCredSet(context: Context, credSet: CredSet){
        val PREF_KEY = "PREF_CREDENTIAL_SET"
        val PREF_IV = "PREF_CREDENTIAL_DECRYPTION_IV"

        // save dataset
        val sharedPref = context.getSharedPreferences(context.packageName, Activity.MODE_PRIVATE)
        val prefsEditor: SharedPreferences.Editor = sharedPref.edit()
        var sharedIV = sharedPref.getString(PREF_IV, "") ?: ""
        val tojson = Json.encodeToString(credSet)

        if (sharedIV.isBlank()) {
            val iv = ByteArray(16)
            val secureRandom = SecureRandom()
            secureRandom.nextBytes(iv)
            sharedIV = OpenPasskeyAuthUtils.b64Encode(IvParameterSpec(iv).iv)
        }

        val cipher = Cipher.getInstance("AES/CBC/PKCS7PADDING")
        cipher.init(Cipher.ENCRYPT_MODE, Locator.getSecretKey(), OpenPasskeyAuthUtils.getIV(sharedIV))
        val encrypted = cipher.doFinal(tojson.toByteArray())

        prefsEditor.putString(PREF_KEY, OpenPasskeyAuthUtils.b64Encode(encrypted))
        prefsEditor.putString(PREF_IV, sharedIV)
        prefsEditor.apply()
        Log.d("OPA", "SharedPref saved: $tojson")
    }

    fun changeCredentialData(context: Context, rpid: String, credentialId: ByteArray, serviceName: String, displayName: String, description: String) {
        val credSet = loadCredSet(context)
        val newCredSet = CredSet()

        credSet.list.forEach {
            if (it.rpid == rpid && it.credentialId.contentEquals(credentialId))  {
                newCredSet.list.add(CredSet.SerializedCred(
                    rpid = it.rpid,
                    serviceName = serviceName,
                    credentialId = it.credentialId,
                    userHandle = it.userHandle,
                    displayName = displayName,
                    description = description,
                    securityLevel = it.securityLevel
                ))
            } else {
                newCredSet.list.add(it)
            }
        }

        saveCredSet(context, newCredSet)
    }

    data class Credential(
        val rpid: String,
        val serviceName: String = rpid,
        val credentialId: ByteArray,
        val userHandle: ByteArray = byteArrayOf(),
        val displayName: String = "",
        val description: String = "",
        val securityLevel: Int = KeyProperties.SECURITY_LEVEL_UNKNOWN,
    ) {
        override fun equals(other: Any?): Boolean {
            if (this === other) return true
            if (javaClass != other?.javaClass) return false

            other as Credential

            if (securityLevel != other.securityLevel) return false
            if (rpid != other.rpid) return false
            if (serviceName != other.serviceName) return false
            if (!credentialId.contentEquals(other.credentialId)) return false
            if (!userHandle.contentEquals(other.userHandle)) return false
            if (displayName != other.displayName) return false
            if (description != other.description) return false

            return true
        }

        override fun hashCode(): Int {
            var result = securityLevel
            result = 31 * result + rpid.hashCode()
            result = 31 * result + serviceName.hashCode()
            result = 31 * result + credentialId.contentHashCode()
            result = 31 * result + userHandle.contentHashCode()
            result = 31 * result + displayName.hashCode()
            result = 31 * result + description.hashCode()
            return result
        }
    }

    @Serializable
    data class CredSet(
        var list: MutableList<SerializedCred> = mutableListOf()
    ){
        @Serializable
        data class SerializedCred(
            val rpid: String,
            val serviceName: String = rpid,
            val credentialId: ByteArray,
            val userHandle: ByteArray = byteArrayOf(),
            val displayName: String = "",
            val description: String = "",
            val securityLevel: Int = KeyProperties.SECURITY_LEVEL_UNKNOWN,
        ) {
            override fun equals(other: Any?): Boolean {
                if (this === other) return true
                if (javaClass != other?.javaClass) return false

                other as SerializedCred

                if (rpid != other.rpid) return false
                if (serviceName != other.serviceName) return false
                if (!credentialId.contentEquals(other.credentialId)) return false
                if (!userHandle.contentEquals(other.userHandle)) return false
                if (displayName != other.displayName) return false
                if (description != other.description) return false
                if (securityLevel != other.securityLevel) return false

                return true
            }

            override fun hashCode(): Int {
                var result = rpid.hashCode()
                result = 31 * result + serviceName.hashCode()
                result = 31 * result + credentialId.contentHashCode()
                result = 31 * result + userHandle.contentHashCode()
                result = 31 * result + displayName.hashCode()
                result = 31 * result + description.hashCode()
                result = 31 * result + securityLevel.hashCode()
                return result
            }
        }
    }

}