package de.noisruker.openPasskeyAuth.ui.settings


import android.os.Bundle
import android.util.Log
import androidx.activity.compose.setContent
import androidx.activity.viewModels
import androidx.annotation.StringRes
import androidx.biometric.BiometricManager
import androidx.biometric.BiometricPrompt
import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.ColumnScope
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.defaultMinSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.selection.toggleable
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material.icons.filled.Check
import androidx.compose.material.icons.outlined.Close
import androidx.compose.material.icons.outlined.Lock
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Surface
import androidx.compose.material3.Switch
import androidx.compose.material3.SwitchDefaults
import androidx.compose.material3.Text
import androidx.compose.material3.TopAppBar
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.alpha
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.semantics.Role
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.dp
import de.noisruker.openPasskeyAuth.Locator
import de.noisruker.openPasskeyAuth.R
import de.noisruker.openPasskeyAuth.ui.BaseActivity
import de.noisruker.openPasskeyAuth.ui.BiometricUnlock
import de.noisruker.openPasskeyAuth.ui.BiometricUnlock.IAppUnlockSuccessHandler
import de.noisruker.openPasskeyAuth.ui.InfoActivity
import de.noisruker.openPasskeyAuth.ui.theme.OpenPasskeyAuthTheme
import de.noisruker.openPasskeyAuth.utils.OpenPasskeyAuthUtils
import kotlinx.coroutines.runBlocking
import java.security.KeyStore

class SettingsActivity : BaseActivity() {

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
    }

    @OptIn(ExperimentalMaterial3Api::class)
    override fun onResume() {
        super.onResume()

        setContent {
            val preferences by viewModel.userPreferences.collectAsState()
            OpenPasskeyAuthTheme {
                Scaffold(
                    topBar = {
                        TopAppBar(
                            title = { Text(text = stringResource(R.string.title_activity_settings)) },
                            navigationIcon = {
                                IconButton(onClick = {
                                    finish()
                                }) {
                                    Icon(
                                        Icons.AutoMirrored.Filled.ArrowBack,
                                        contentDescription = stringResource(R.string.back)
                                    )
                                }
                            },
                            actions = {

                                if (preferences?.appLock == true) {
                                    IconButton(onClick = {
                                        super.lockApp()
                                        recreate()
                                    }) {
                                        Icon(
                                            Icons.Outlined.Lock,
                                            contentDescription = stringResource(R.string.lock)
                                        )
                                    }
                                }
                            }
                        )
                    }
                ) { paddingValues ->
                    SettingsView(paddingValues)
                }
            }
        }
    }

    private val viewModel by viewModels<SettingsViewModel>(factoryProducer = { Locator.settingsViewModelFactory })

    @Composable
    private fun SettingsView(padding: PaddingValues) {
        val preferences by viewModel.userPreferences.collectAsState()

        LazyColumn(
            modifier = Modifier.padding(padding)
        ) {
            item {
                SettingsGroup(stringResource(R.string.security_settings_title)) {
                    SettingSwitchItem(
                        checked = preferences?.strongBoxKey == true,
                        onCheckedChange = { viewModel.updateStrongBoxKey(it) },
                        title = R.string.strong_box_key,
                        description = R.string.strong_box_key_description,
                        enabled = OpenPasskeyAuthUtils.checkStrongBoxAvailability(applicationContext),
                    )

                    HorizontalDivider()

                    AnimatedVisibility(preferences?.strongBoxKey == true) {
                        SettingSwitchItem(
                            checked = preferences?.askForStrongBoxOnKeyCreation == true,
                            onCheckedChange = { viewModel.updateAskForStrongBoxKeyCreation(it) },
                            title = R.string.ask_for_strong_box_key,
                            description = R.string.ask_for_strong_box_key_description,
                            enabled = OpenPasskeyAuthUtils.checkStrongBoxAvailability(
                                applicationContext
                            ),
                        )
                    }

                    HorizontalDivider()

                    SettingSwitchItem(
                        checked = preferences?.appLock == true,
                        onCheckedChange = { enabled ->
                            changeAppLock(enabled)
                        },
                        title = R.string.app_lock_title,
                        description = R.string.app_lock_description,
                        enabled = (BiometricManager.from(applicationContext)
                            .canAuthenticate(BiometricManager.Authenticators.BIOMETRIC_STRONG) == BiometricManager.BIOMETRIC_SUCCESS)
                    )

                    HorizontalDivider()

                    SettingSwitchItem(
                        checked = preferences?.allowScreenshots == true,
                        onCheckedChange = { viewModel.updateEnableScreenshots(it) },
                        title = R.string.allow_screenshot,
                        description = R.string.allow_screenshot_explanation,
                        enabled = true,
                    )

                    /* TODO: REACTIVATE IF REIMPLEMENTED
                    HorizontalDivider()

                    SettingSwitchItem(
                        checked = preferences?.scramblePin != false,
                        onCheckedChange = { viewModel.updateScramblePinLayout(it) },
                        title = R.string.scramble_pin,
                        description = R.string.scramble_pin_explanation,
                        enabled = true,
                    )*/
                }
            }

            item {
                SettingsGroup(stringResource(R.string.setup)) {
                    InfoActivity().SetupOPA(this@SettingsActivity)
                }

            }

        }
    }

    companion object {

        @Composable
        fun SettingsGroup(title: String? = null, content: @Composable ColumnScope.() -> Unit) {
            Column(modifier = Modifier.padding(horizontal = 16.dp, vertical = 8.dp)) {
                if (title != null) {
                    Text(title, style = MaterialTheme.typography.labelLarge, modifier = Modifier.padding(bottom = 8.dp))
                }

                Surface(
                    shape = MaterialTheme.shapes.medium,
                    tonalElevation = 1.dp,
                    modifier = Modifier.fillMaxWidth()
                ) {
                    Column {
                        content()
                    }
                }
            }
        }

        @Composable
        fun SettingSwitchItem(
            modifier: Modifier = Modifier,
            checked: Boolean,
            onCheckedChange: (Boolean) -> Unit,
            @StringRes title: Int,
            @StringRes description: Int?,
            enabled: Boolean = true,
        ) {
            Row(
                modifier = modifier
                    .fillMaxWidth()
                    .defaultMinSize(Dp.Unspecified, 50.dp)
                    .toggleable(
                        value = checked,
                        enabled = enabled,
                        role = Role.Switch,
                        onValueChange = onCheckedChange
                    ),
                verticalAlignment = Alignment.CenterVertically
            ) {
                Row (modifier = Modifier.padding(16.dp), verticalAlignment = Alignment.CenterVertically){
                    Column(
                        modifier = Modifier.weight(1.0f),
                        verticalArrangement = Arrangement.spacedBy(8.dp)
                    ) {
                        val contentAlpha = if (enabled) 1.0f else 0.5f

                        Text(
                            text = stringResource(id = title),
                            style = MaterialTheme.typography.titleSmall,
                            maxLines = 1,
                            modifier = Modifier.alpha(contentAlpha),
                            color = MaterialTheme.colorScheme.primary
                        )
                        if (description != null) {
                            Text(
                                text = stringResource(id = description),
                                style = MaterialTheme.typography.labelMedium,
                                modifier = Modifier.alpha(contentAlpha),
                                color = MaterialTheme.colorScheme.primary
                            )
                        }
                    }

                    Switch(
                        checked = checked,
                        onCheckedChange = null,
                        enabled = enabled,
                        thumbContent = {
                            if (checked) {
                                Icon(
                                    imageVector = Icons.Filled.Check,
                                    contentDescription = null,
                                    modifier = Modifier.size(SwitchDefaults.IconSize),
                                    tint = MaterialTheme.colorScheme.onPrimary
                                )
                            } else {
                                Icon(
                                    imageVector = Icons.Outlined.Close,
                                    contentDescription = null,
                                    modifier = Modifier.size(SwitchDefaults.IconSize),
                                    tint = MaterialTheme.colorScheme.onPrimary
                                )
                            }
                        },
                        colors = SwitchDefaults.colors(
                            checkedThumbColor = MaterialTheme.colorScheme.primary,
                            checkedTrackColor = MaterialTheme.colorScheme.primaryContainer,
                            uncheckedThumbColor = MaterialTheme.colorScheme.secondary,
                            uncheckedTrackColor = MaterialTheme.colorScheme.secondaryContainer,
                        )
                    )
                }
            }
        }
    }

    @Preview
    @Composable
    fun SimpleComposablePreview() {
        Locator.initWith(this.application)
        SettingsView(PaddingValues(0.dp))
    }

    private fun changeAppLock(enabled: Boolean) {
        if (enabled) {
            super.generateAppLockKey(KeyStore.getInstance("AndroidKeyStore").apply {
                load(null)
            })

            Log.d("OPA", "Start enabling app lock")

            val successHandler = IAppUnlockSuccessHandler { encryptedSecret, iv ->
                if (encryptedSecret != null && iv != null) {
                    runBlocking {
                        Locator.userPreferencesRepository.enableScreenLock(true)
                        Locator.userPreferencesRepository.setAppUnlockSecret(encryptedSecret)
                        Locator.userPreferencesRepository.appUnlockIV(OpenPasskeyAuthUtils.b64Encode(iv))
                    }
                }
            }

            val cipher = BiometricUnlock.encryptAppSecretCypher()

            val promptInfo = BiometricUnlock.biometricPromptInfo(this.application)
            val biometricPrompt =
                BiometricUnlock.unlockAppSecretPromt(successHandler, mainExecutor, this)

            // Log.d("OPA", "Try enabling app lock: $cipher; $promptInfo $biometricPrompt")

            if (cipher == null) return

            biometricPrompt.authenticate(promptInfo, BiometricPrompt.CryptoObject(cipher))
        } else {
            val successHandler = IAppUnlockSuccessHandler { success, _ ->
                if (success != null) {
                    Locator.appSecret = success

                    runBlocking {
                        Locator.userPreferencesRepository.enableScreenLock(false)
                        Locator.userPreferencesRepository.setAppUnlockSecret(success)
                        Locator.userPreferencesRepository.appUnlockIV("")
                    }
                }
            }

            val cipher = BiometricUnlock.unlockAppSecretCypher()
            val promptInfo = BiometricUnlock.biometricPromptInfo(this.application)
            val biometricPrompt =
                BiometricUnlock.unlockAppSecretPromt(successHandler, mainExecutor, this)

            if (cipher == null) return

            biometricPrompt.authenticate(promptInfo, BiometricPrompt.CryptoObject(cipher))
        }
    }
}