package de.noisruker.openPasskeyAuth.ui

import android.annotation.SuppressLint
import android.content.Intent
import android.content.pm.PackageManager
import android.os.Bundle
import android.security.keystore.KeyGenParameterSpec
import android.security.keystore.KeyInfo
import android.security.keystore.KeyProperties
import android.security.keystore.StrongBoxUnavailableException

import android.util.Log
import android.widget.Toast
import androidx.activity.compose.setContent
import android.content.IntentSender
import androidx.activity.result.IntentSenderRequest
import androidx.activity.result.contract.ActivityResultContracts
import androidx.appcompat.app.AppCompatActivity
import androidx.biometric.BiometricManager
import androidx.biometric.BiometricPrompt
import androidx.biometric.BiometricPrompt.CryptoObject
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Lock
import androidx.compose.material.icons.outlined.Cancel
import androidx.compose.material.icons.outlined.SyncLock
import androidx.compose.material3.AlertDialog
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.ModalBottomSheet
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.material3.TopAppBar
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.core.net.toUri
import androidx.credentials.CreatePublicKeyCredentialRequest
import androidx.credentials.CreatePublicKeyCredentialResponse
import androidx.credentials.GetCredentialResponse
import androidx.credentials.GetPublicKeyCredentialOption
import androidx.credentials.PublicKeyCredential
import androidx.credentials.provider.CallingAppInfo
import androidx.credentials.provider.PendingIntentHandler
import androidx.credentials.provider.ProviderGetCredentialRequest
import androidx.credentials.webauthn.AuthenticatorAssertionResponse
import androidx.credentials.webauthn.AuthenticatorAttestationResponse
import androidx.credentials.webauthn.FidoPublicKeyCredential
import androidx.credentials.webauthn.PublicKeyCredentialCreationOptions
import androidx.credentials.webauthn.PublicKeyCredentialRequestOptions
import com.google.android.gms.fido.Fido
import com.google.android.gms.fido.common.Transport
import com.google.android.gms.fido.fido2.api.common.AttestationConveyancePreference
import com.google.android.gms.fido.fido2.api.common.AuthenticationExtensions
import com.google.android.gms.fido.fido2.api.common.AuthenticatorErrorResponse
import com.google.android.gms.fido.fido2.api.common.AuthenticatorSelectionCriteria
import com.google.android.gms.fido.fido2.api.common.BrowserPublicKeyCredentialCreationOptions
import com.google.android.gms.fido.fido2.api.common.BrowserPublicKeyCredentialRequestOptions
import com.google.android.gms.fido.fido2.api.common.FidoAppIdExtension
import com.google.android.gms.fido.fido2.api.common.PublicKeyCredentialDescriptor
import com.google.android.gms.fido.fido2.api.common.PublicKeyCredentialParameters
import com.google.android.gms.fido.fido2.api.common.PublicKeyCredentialRpEntity
import com.google.android.gms.fido.fido2.api.common.PublicKeyCredentialType
import com.google.android.gms.fido.fido2.api.common.ResidentKeyRequirement
import de.noisruker.openPasskeyAuth.Locator
import de.noisruker.openPasskeyAuth.OpenPasskeyAuthService
import de.noisruker.openPasskeyAuth.R
import de.noisruker.openPasskeyAuth.ui.theme.OpenPasskeyAuthTheme
import de.noisruker.openPasskeyAuth.utils.CredentialDataManager
import de.noisruker.openPasskeyAuth.utils.DeviceKeyAuthenticationResponse
import de.noisruker.openPasskeyAuth.utils.OpenPasskeyAuthUtils
import de.noisruker.openPasskeyAuth.utils.hwkeys.PublicKeyCredentialData
import de.noisruker.openPasskeyAuth.utils.hwkeys.parseCreateCredentialRequest
import de.noisruker.openPasskeyAuth.utils.hwkeys.parseGetCredentialRequest
import de.noisruker.openPasskeyAuth.utils.preferences.UserPreferences
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.runBlocking
import kotlinx.serialization.ExperimentalSerializationApi
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.ClassDiscriminatorMode
import kotlinx.serialization.json.Json
import java.math.BigInteger
import java.security.KeyFactory
import java.security.KeyPair
import java.security.KeyPairGenerator
import java.security.KeyStore
import java.security.MessageDigest
import java.security.SecureRandom
import java.security.Signature
import java.security.interfaces.ECPublicKey
import java.security.spec.ECGenParameterSpec
import java.security.spec.ECPoint
import java.security.spec.InvalidKeySpecException
import kotlin.io.encoding.Base64

class OpenPasskeyAuthActivity: AppCompatActivity() {

    @OptIn(ExperimentalMaterial3Api::class)
    @SuppressLint("RestrictedApi")
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        setContent {
            OpenPasskeyAuthTheme {
                Scaffold(
                    topBar = @Composable {
                        TopAppBar(
                            title = { Text(text = stringResource(R.string.title_activity_edit_credential)) },
                        )
                    },
                    content = @Composable { padding ->
                        Column(modifier = Modifier.padding(padding)) {
                            Column(
                                modifier = Modifier
                                    .fillMaxWidth()
                                    .fillMaxHeight()
                                    .padding(horizontal = 16.dp, vertical = 8.dp),
                                verticalArrangement = Arrangement.Center,
                            ) {
                                Column(
                                    modifier = Modifier.verticalScroll(
                                        rememberScrollState()
                                        ),
                                    verticalArrangement = Arrangement.spacedBy(128.dp)
                                ) {
                                    TextButton(
                                        onClick = {
                                            recreate()
                                        },
                                        modifier = Modifier
                                            .fillMaxWidth()
                                    ) {
                                        Column(
                                            verticalArrangement = Arrangement.spacedBy(30.dp)
                                        ) {
                                            Icon(
                                                Icons.Outlined.SyncLock,
                                                contentDescription = null,
                                                modifier = Modifier
                                                    .align(Alignment.CenterHorizontally)
                                                    .size(80.dp)
                                            )
                                            Text(
                                                text = stringResource(R.string.retry),
                                                fontSize = 60.sp,
                                                fontWeight = FontWeight.Bold
                                            )
                                        }

                                    }

                                    TextButton(
                                        onClick = {
                                            setResult(RESULT_CANCELED)
                                            finish()
                                        },
                                        modifier = Modifier
                                            .fillMaxWidth()
                                    ) {
                                        Column(
                                            verticalArrangement = Arrangement.spacedBy(30.dp)
                                        ) {
                                            Icon(
                                                Icons.Outlined.Cancel,
                                                contentDescription = null,
                                                modifier = Modifier
                                                    .align(Alignment.CenterHorizontally)
                                                    .size(80.dp)
                                            )
                                            Text(
                                                text = stringResource(R.string.cancel),
                                                fontSize = 60.sp,
                                                fontWeight = FontWeight.Bold
                                            )
                                        }

                                    }
                                }
                            }
                        }
                    }
                )
            }


            if (intent == null) {
                setResult(RESULT_CANCELED)
                finish()
                return@setContent
            }

            if (intent.action == OpenPasskeyAuthService.GET_PASSKEY_INTENT_ACTION) {
                val getRequest = PendingIntentHandler.retrieveProviderGetCredentialRequest(intent)!!

                val extras = intent.getBundleExtra(OpenPasskeyAuthService.CREDENTIAL_DATA_EXTRA)
                if (extras != null) {
                    handleKeyGetRequest(getRequest, extras)
                } else {
                    HandleHardwareKeyGetRequest(getRequest)
                }
            } else if (intent.action == OpenPasskeyAuthService.CREATE_PASSKEY_INTENT_ACTION) {
                val request =
                    PendingIntentHandler.retrieveProviderCreateCredentialRequest(intent)

                val accountId =
                    intent.getBundleExtra(OpenPasskeyAuthService.CREDENTIAL_DATA_EXTRA)?.getString(
                        OpenPasskeyAuthService.ACCOUNT_ID
                    )

                if (request != null && request.callingRequest is CreatePublicKeyCredentialRequest) {
                    val publicKeyRequest: CreatePublicKeyCredentialRequest =
                        request.callingRequest as CreatePublicKeyCredentialRequest

                    if (accountId == OpenPasskeyAuthService.DEVICE_ACCOUNT) {
                        CreateDevicePasskeyStrongBoxDialog(
                            publicKeyRequest.requestJson,
                            request.callingAppInfo,
                            publicKeyRequest.clientDataHash
                        )
                    } else {
                        createHardwareKeyPasskey(
                            publicKeyRequest.requestJson,
                            request.callingAppInfo,
                            publicKeyRequest.clientDataHash
                        )
                    }
                }
            }
        }
    }

    @Composable
    private fun CreateDevicePasskeyStrongBoxDialog(
        requestJson: String,
        callingAppInfo: CallingAppInfo,
        clientDataHash: ByteArray?,
    ) {
        var prefs = UserPreferences(true, false, false, false, false)

        runBlocking {
            try {
                prefs =
                    Locator.userPreferencesRepository.userPreferences.first() // Suspend until value is available

                Log.d("OPA", "Load prefs: $prefs")
            } catch (e: Exception) {
                // Handle errors (e.g., repository not initialized, network issue)
                Log.e("OPA", "Error loading prefs: ${e.message}", e)
            }
        }

        if (prefs.strongBoxKey) {
            if (OpenPasskeyAuthUtils.checkStrongBoxAvailability(applicationContext)) {
                if (!prefs.askForStrongBoxOnKeyCreation) {
                    createDevicePasskey(
                        requestJson,
                        callingAppInfo,
                        clientDataHash,
                        true,
                        prefs
                    )
                } else {

                        val openDialog = remember { mutableStateOf(true) }
                        if (openDialog.value) {
                            AlertDialog(
                                modifier = Modifier.fillMaxWidth(),
                                onDismissRequest = {
                                    // Dismiss the dialog when the user clicks outside the dialog or on the back
                                    // button. If you want to disable that functionality, simply use an empty
                                    // onDismissRequest.
                                },
                                icon = { Icon(Icons.Default.Lock, contentDescription = null) },
                                title = {
                                    Text(text = stringResource(R.string.create_strong_box_key))
                                },
                                text = {
                                    Text(
                                        text = stringResource(R.string.create_strong_box_key_description),
                                    )
                                },
                                confirmButton = {
                                    TextButton(
                                        onClick = {
                                            openDialog.value = false

                                            createDevicePasskey(
                                                requestJson,
                                                callingAppInfo,
                                                clientDataHash,
                                                true,
                                                prefs
                                            )
                                        },
                                    ) {
                                        Text(stringResource(R.string.create_in_strongbox))
                                    }
                                },
                                dismissButton = {
                                    TextButton(
                                        onClick = {
                                            openDialog.value = false

                                            createDevicePasskey(
                                                requestJson,
                                                callingAppInfo,
                                                clientDataHash,
                                                false,
                                                prefs
                                            )
                                        },
                                    ) {
                                        Text(stringResource(R.string.create_in_keystore))
                                    }
                                }
                            )
                        }
                    }

            } else {

                    val openDialog = remember { mutableStateOf(true) }
                    if (openDialog.value) {
                        AlertDialog(
                            modifier = Modifier.fillMaxWidth(),
                            onDismissRequest = {
                                // Dismiss the dialog when the user clicks outside the dialog or on the back
                                // button. If you want to disable that functionality, simply use an empty
                                // onDismissRequest.
                            },
                            icon = { Icon(Icons.Default.Lock, contentDescription = null) },
                            title = {
                                Text(text = stringResource(R.string.create_strong_box_not_available))
                            },
                            text = {
                                Text(
                                    text = stringResource(R.string.create_strong_box_not_available_description),
                                )
                            },
                            confirmButton = {
                                TextButton(
                                    onClick = {
                                        openDialog.value = false

                                        createDevicePasskey(
                                            requestJson,
                                            callingAppInfo,
                                            clientDataHash,
                                            true,
                                            prefs
                                        )
                                    },
                                ) {
                                    Text(stringResource(R.string.create_in_keystore))
                                }
                            },
                            dismissButton = {
                                TextButton(
                                    onClick = {
                                        openDialog.value = false

                                        finish()
                                    },
                                ) {
                                    Text(stringResource(R.string.cancel))
                                }
                            }
                        )

                }
            }
        } else {
            createDevicePasskey(requestJson, callingAppInfo, clientDataHash, false, prefs)
        }
    }

    @SuppressLint("RestrictedApi")
    private fun createDevicePasskey(
        requestJson: String,
        callingAppInfo: CallingAppInfo,
        clientDataHash: ByteArray?,
        putInStrongBox: Boolean,
        prefs: UserPreferences
    ) {
        val request = PublicKeyCredentialCreationOptions(requestJson)

        val biometricPrompt = BiometricPrompt(
            this,
            this.mainExecutor,
            object: BiometricPrompt.AuthenticationCallback() {
                override fun onAuthenticationFailed() {
                    super.onAuthenticationFailed()
                }

                override fun onAuthenticationError(errorCode: Int, errString: CharSequence) {
                    super.onAuthenticationError(errorCode, errString)
                }

                override fun onAuthenticationSucceeded(authResult: BiometricPrompt.AuthenticationResult) {
                    super.onAuthenticationSucceeded(authResult)

                    if (prefs.appLock) {
                        val cipher = authResult.cryptoObject?.cipher

                        if (cipher == null) {
                            setResult(RESULT_CANCELED)
                            finish()
                            return
                        }

                        val data = cipher.doFinal(OpenPasskeyAuthUtils.b64Decode(prefs.appUnlockSecret))

                        Locator.appSecret = OpenPasskeyAuthUtils.b64Encode(data)
                    }

                    // Generate a credential ID
                    val credentialId = ByteArray(32)
                    SecureRandom().nextBytes(credentialId)

                    val spec = KeyGenParameterSpec.Builder(
                        "opa-" + OpenPasskeyAuthUtils.b64Encode(credentialId),
                        KeyProperties.PURPOSE_SIGN
                    ).run {
                        setDigests(KeyProperties.DIGEST_SHA256)
                        setAlgorithmParameterSpec(ECGenParameterSpec("secp256r1"))
                        if (applicationContext.packageManager.hasSystemFeature(PackageManager.FEATURE_STRONGBOX_KEYSTORE)) {
                            setIsStrongBoxBacked(putInStrongBox)
                        }
                        setUserAuthenticationRequired(true)
                        setUserAuthenticationParameters(0, KeyProperties.AUTH_BIOMETRIC_STRONG)
                        setUnlockedDeviceRequired(true)
                        setInvalidatedByBiometricEnrollment(false)
                        build()
                    }
                    val keyPairGen = KeyPairGenerator.getInstance("EC", "AndroidKeyStore")

                    try {
                        keyPairGen.initialize(spec)
                    } catch (_: StrongBoxUnavailableException) {
                        Toast.makeText(applicationContext, R.string.error_in_passkey_creation, Toast.LENGTH_LONG).show()

                        finish()
                    }
                    val keyPair = keyPairGen.generateKeyPair()

                    // Check if rpId is valid
                    val rpId =
                        OpenPasskeyAuthUtils.validateRpId(applicationContext, callingAppInfo, request.rp.id)

                    var securityLevel = KeyProperties.SECURITY_LEVEL_UNKNOWN
                    val keyStore = KeyStore.getInstance("AndroidKeyStore").apply {
                        load(null)
                    }
                    val entry: KeyStore.Entry = keyStore.getEntry("opa-" + OpenPasskeyAuthUtils.b64Encode(credentialId), null)

                    if (entry is KeyStore.PrivateKeyEntry) {
                        val factory = KeyFactory.getInstance(entry.privateKey.algorithm, "AndroidKeyStore")
                        try {
                            securityLevel = factory.getKeySpec(entry.privateKey, KeyInfo::class.java).securityLevel
                        } catch (_: InvalidKeySpecException) {

                        }
                    }

                    Log.d("OPA", "Save new credential data ${Locator.appSecret}")

                    CredentialDataManager.save(
                        this@OpenPasskeyAuthActivity, CredentialDataManager.Credential(
                            rpId,
                            serviceName = request.rp.name,
                            credentialId,
                            displayName = request.user.displayName,
                            description = "",
                            userHandle = request.user.id,
                            securityLevel = securityLevel,
                        )
                    )

                    // Generate response

                    val response = AuthenticatorAttestationResponse(
                        requestOptions = request,
                        credentialId,
                        credentialPublicKey = getPublicKeyFromKeyPair(keyPair),
                        origin = OpenPasskeyAuthUtils.appInfoToOrigin(applicationContext, callingAppInfo),
                        up = true,
                        uv = true,
                        be = true,
                        bs = true,
                        packageName = callingAppInfo.packageName,
                        clientDataHash,
                    )

                    val credential = FidoPublicKeyCredential(
                        rawId = credentialId, response, authenticatorAttachment = "platform"
                    )

                    val credentialJson = populateEasyAccessorFields(credential.json(), rpId, keyPair, credentialId)

                    val result = Intent()

                    Log.d("OPA", "RESPONSE: $credentialJson")

                    val createPublicKeyCredentialResponse = CreatePublicKeyCredentialResponse(credentialJson)

                    PendingIntentHandler.setCreateCredentialResponse(result, createPublicKeyCredentialResponse)

                    setResult(RESULT_OK, result)
                    finish()
                }
            }
        )

        val promptInfo = BiometricPrompt.PromptInfo.Builder()
            .setTitle(application.getString(R.string.use_screen_lock))
            .setSubtitle("${application.getString(R.string.create_passkey_for)} ${request.rp.name}")
            .setAllowedAuthenticators(BiometricManager.Authenticators.BIOMETRIC_STRONG)
            .setNegativeButtonText(application.getString(R.string.cancel))
            .build()

        if (prefs.appLock) {
            val cipher = BiometricUnlock.unlockAppSecretCypher()

            if (cipher == null) {
                setResult(RESULT_CANCELED)
                finish()
                return
            }

            biometricPrompt.authenticate(promptInfo, CryptoObject(cipher))
        } else {
            if (Locator.appSecret == null) {
                Locator.appSecret = prefs.appUnlockSecret
            }
            biometricPrompt.authenticate(promptInfo)
        }
    }

    @Serializable
    private data class CreatePublicKeyCredentialResponseJson(
        //RegistrationResponseJSON
        val id:String,
        val rawId: String,
        val response: Response,
        val authenticatorAttachment: String?,
        val clientExtensionResults: EmptyClass = EmptyClass(),
        val type: String,
    ) {
        @Serializable
        data class Response(
            //AuthenticatorAttestationResponseJSON
            val clientDataJSON: String? = null,
            var authenticatorData: String? = null,
            val transports: List<String>? = arrayOf("internal").toList(),
            var publicKey: String? = null, // easy accessors fields
            var publicKeyAlgorithm: Long? = null, // easy accessors fields
            val attestationObject: String? // easy accessors fields
        )
        @Serializable
        class EmptyClass
    }

    private fun populateEasyAccessorFields(json: String, rpid: String , keyPair: KeyPair, credentialId: ByteArray):String{
        val response = Json.decodeFromString<CreatePublicKeyCredentialResponseJson>(json)

        response.response.publicKeyAlgorithm = -7 // ES256
        response.response.publicKey =
            OpenPasskeyAuthUtils.b64Encode(keyPair.public.encoded)
        response.response.authenticatorData = getAuthData(rpid, credentialId, keyPair)

        Log.d("OPA", "PUBLIC KEY: ${response.response.publicKey}")

        Log.d("OPA","=== populateEasyAccessorFields AFTER === "+ Json.encodeToString(response))
        return Json.encodeToString(response)
    }

    private fun getAuthData(rpid: String, credentialRawId: ByteArray, keyPair: KeyPair ):String{
        val AAGUID = "00000000000000000000000000000000" // TODO: INVESTIGATE!
        check(AAGUID.length % 2 == 0) { "AAGUID Must have an even length" }

        val rpIdHash:ByteArray = MessageDigest.getInstance("SHA-256")
            .digest(rpid.toByteArray())

        val flags: ByteArray = byteArrayOf(0x5d.toByte())
        val signCount:ByteArray = byteArrayOf(0x00, 0x00, 0x00, 0x00)
        val aaguid = AAGUID.chunked(2)
            .map { it.toInt(16).toByte() }
            .toByteArray()

        val credentialIdLength: ByteArray = byteArrayOf(0x00, credentialRawId.size.toByte()) // = 20 bytes
        val credentialPublicKey: ByteArray = getPublicKeyFromKeyPair(keyPair)

        val retVal = rpIdHash + flags + signCount + aaguid + credentialIdLength + credentialRawId + credentialPublicKey
        return OpenPasskeyAuthUtils.b64Encode(retVal)
    }

    private fun getPublicKeyFromKeyPair(keyPair: KeyPair?): ByteArray {
        // credentialPublicKey CBOR
        if (keyPair == null)
            return ByteArray(0)
        if (keyPair.public !is ECPublicKey)
            return ByteArray(0)

        val ecPubKey = keyPair.public as ECPublicKey
        val ecPoint: ECPoint = ecPubKey.w

        // for now, only covers ES256
        if (ecPoint.affineX.bitLength() > 256 || ecPoint.affineY.bitLength() > 256)
            return ByteArray(0)

        val byteX = bigIntToByteArray32(ecPoint.affineX)
        val byteY = bigIntToByteArray32(ecPoint.affineY)

        // refer to RFC9052 Section 7 for details
        return "A5010203262001215820".chunked(2).map { it.toInt(16).toByte() }.toByteArray() +
                byteX+
                "225820".chunked(2).map { it.toInt(16).toByte() }.toByteArray() +
                byteY
    }

    private fun bigIntToByteArray32(bigInteger: BigInteger):ByteArray{
        var ba = bigInteger.toByteArray()

        if (ba.size < 32) {
            // append zeros in front
            ba = ByteArray(32) + ba
        }
        // get the last 32 bytes as bigint conversion sometimes put extra zeros at front
        return ba.copyOfRange(ba.size - 32, ba.size)
    }

    @SuppressLint("RestrictedApi")
    private fun handleKeyGetRequest(getRequest: ProviderGetCredentialRequest, extras: Bundle) {
        val publicKeyRequests = getRequest.credentialOptions as List<GetPublicKeyCredentialOption>

        val credentialIdEncoded = extras.getString(OpenPasskeyAuthService.CREDENTIAL_ID)
        val requestJson = PublicKeyCredentialRequestOptions(publicKeyRequests[0].requestJson)

        val credentialId = OpenPasskeyAuthUtils.b64Decode(credentialIdEncoded)
        val rpId = OpenPasskeyAuthUtils.validateRpId(
            applicationContext,
            getRequest.callingAppInfo,
            requestJson.rpId
        )

        val passkey = CredentialDataManager.load(this, rpId, credentialId!!)
        val uid = passkey!!.userHandle
        val origin = OpenPasskeyAuthUtils.appInfoToOrigin(applicationContext, getRequest.callingAppInfo)
        val packageName = getRequest.callingAppInfo.packageName
        val clientDataHash = publicKeyRequests[0].clientDataHash

        validatePasskey(
            publicKeyRequests[0].requestJson,
            origin,
            packageName,
            uid,
            credentialId,
            clientDataHash
        )
    }

    // https://developer.android.com/training/sign-in/credential-provider#passkeys-implement
    @SuppressLint("RestrictedApi")
    fun validatePasskey(requestJson: String, origin: String, packageName: String, uid: ByteArray,
                        credId: ByteArray, clientDataHash: ByteArray?){
        val request = PublicKeyCredentialRequestOptions(requestJson)

        val authResponse = AuthenticatorAssertionResponse(
            requestOptions = request,
            credentialId = credId,
            origin = origin,
            up = true,
            uv = true,
            be = true,
            bs = true,
            userHandle = uid,
            packageName = packageName,
            clientDataHash = clientDataHash
        )

        val keyStore = KeyStore.getInstance("AndroidKeyStore").apply {
            load(null)
        }
        val entry: KeyStore.Entry = keyStore.getEntry("opa-" + OpenPasskeyAuthUtils.b64Encode(credId), null)

        if (entry !is KeyStore.PrivateKeyEntry) {
            setResult(RESULT_CANCELED)
            finish()
            return
        }

        val signature: Signature = Signature.getInstance("SHA256withECDSA").run {
            initSign(entry.privateKey)
            update(authResponse.dataToSign())
            this
        }

        val biometricPrompt = BiometricPrompt(
            this,
            this.mainExecutor,
            object : BiometricPrompt.AuthenticationCallback() {
                override fun onAuthenticationError(
                    errorCode: Int, errString: CharSequence
                ) {
                    super.onAuthenticationError(errorCode, errString)

                    Log.e("OPA", "onAuthenticationError $errorCode $errString")
                }

                override fun onAuthenticationFailed() {
                    super.onAuthenticationFailed()

                    Log.e("OPA", "onAuthenticationFailed")
                }

                override fun onAuthenticationSucceeded(
                    authResult: BiometricPrompt.AuthenticationResult
                ) {
                    super.onAuthenticationSucceeded(authResult)

                    if (authResult.cryptoObject?.signature == null) {
                        setResult(RESULT_CANCELED)
                        finish()
                    }

                    authResponse.signature = authResult.cryptoObject!!.signature!!.run {
                        sign()
                    }

                    val clientDataJSONb64 = getClientDataJSONb64(origin,
                        OpenPasskeyAuthUtils.b64Encode(request.challenge), packageName)
                    val response = DeviceKeyAuthenticationResponse(authResponse, clientDataJSONb64)

                    val credential = FidoPublicKeyCredential(
                        rawId = credId, response = response
                        , authenticatorAttachment = "platform")

                    val result = Intent()
                    val passkeyCredential = PublicKeyCredential(credential.json())

                    PendingIntentHandler.setGetCredentialResponse(
                        result, GetCredentialResponse(passkeyCredential)
                    )

                    setResult(RESULT_OK, result)
                    finish()
                }
            }
        )

        val promptInfo = BiometricPrompt.PromptInfo.Builder()
            .setTitle(this.application.getString(R.string.use_screen_lock))
            .setSubtitle("${this.application.getString(R.string.use_passkey_for)} ${request.rpId}")
            .setAllowedAuthenticators(
                BiometricManager.Authenticators.BIOMETRIC_STRONG
            )
            .setNegativeButtonText(this.application.getString(R.string.cancel))
            .build()

        biometricPrompt.authenticate(promptInfo, CryptoObject(signature))
    }

    private fun getClientDataJSONb64(origin: String, challenge: String, packageName: String): String {
        val jsonString =
            """
            {"type":"webauthn.get","challenge":"$challenge","origin":"$origin","crossOrigin":"false","hashAlgorithm":"SHA-256","androidPackageName":"$packageName"}
            """.trimIndent()

        val jsonByteArray = jsonString.toByteArray()

        return OpenPasskeyAuthUtils.b64Encode(jsonByteArray)
    }

    @OptIn(ExperimentalSerializationApi::class)
    val json = Json {
        ignoreUnknownKeys = true
        explicitNulls = false
        classDiscriminatorMode = ClassDiscriminatorMode.NONE
    }

    val base64 = Base64.UrlSafe.withPadding(Base64.PaddingOption.ABSENT_OPTIONAL)

    @OptIn(ExperimentalSerializationApi::class)
    val hardwareKeyCreationGetResult = registerForActivityResult(ActivityResultContracts.StartIntentSenderForResult()) {
        if (it.resultCode != RESULT_OK) {
            setResult(RESULT_CANCELED)
            finish()
            return@registerForActivityResult
        }

        val responseData =
            it.data?.getByteArrayExtra(Fido.FIDO2_KEY_CREDENTIAL_EXTRA) ?: run {
                setResult(RESULT_CANCELED)
                finish()
                return@registerForActivityResult
            }

        val publicKeyCreds =
            com.google.android.gms.fido.fido2.api.common.PublicKeyCredential.deserializeFromBytes(
                responseData
            )

        val cbor = kotlinx.serialization.cbor.Cbor {
            ignoreUnknownKeys = true
        }

        val publicKeyData =
            PublicKeyCredentialData.parseMicroGPublicKeyCredentialData(
                Base64.UrlSafe.withPadding(Base64.PaddingOption.ABSENT_OPTIONAL),
                cbor,
                publicKeyCreds
            ) ?: run {
                setResult(RESULT_CANCELED)
                finish()
                return@registerForActivityResult
            }

        val response = CreatePublicKeyCredentialResponse(
            json.encodeToString(publicKeyData)
        )

        val responseIntent = Intent()
        PendingIntentHandler.setCreateCredentialResponse(responseIntent, response)
        setResult(RESULT_OK, responseIntent)
        finish()
    }

    @SuppressLint("RestrictedApi")
    private fun createHardwareKeyPasskey(
        requestJson: String,
        callingAppInfo: CallingAppInfo,
        clientDataHash: ByteArray?
    ) {

        val createRequest = requestJson.parseCreateCredentialRequest(json) ?: run {
            Log.d("OPA", "Creation request init parsing failed! $requestJson")
            setResult(RESULT_CANCELED)
            finish()
            return
        }

        val user = createRequest.user.let {
            com.google.android.gms.fido.fido2.api.common.PublicKeyCredentialUserEntity(base64.decode(it.id), it.name, null, it.displayName)
        }

        val authenticatorSelection = createRequest.authenticatorSelection.let { selection ->
            AuthenticatorSelectionCriteria.Builder().apply {
                selection.residentKey?.let { setResidentKeyRequirement(ResidentKeyRequirement.fromString(it)) }
                selection.requireResidentKey?.let { setRequireResidentKey(it) }
            }.build()
        }

        val params = createRequest.pubKeyCredParam.map { PublicKeyCredentialParameters(it.type, it.alg) }

        val challenge = base64.decode(createRequest.challenge)
        val publicKey = createRequest.rp.let { PublicKeyCredentialRpEntity(it.id, it.name, null) }

        val origin = OpenPasskeyAuthUtils.appInfoToOrigin(applicationContext, callingAppInfo)

        val fidoClient = Fido.getFido2PrivilegedApiClient(this)
        val extensions = AuthenticationExtensions.Builder().apply {
            createRequest.extensions?.appId?.let {
                setFido2Extension(
                    FidoAppIdExtension(it)
                )
            }
        }.build()

        val requestOptions = com.google.android.gms.fido.fido2.api.common.PublicKeyCredentialCreationOptions.Builder()
            .setAttestationConveyancePreference(AttestationConveyancePreference.fromString(createRequest.attestation))
            .setTimeoutSeconds(20.0)
            .setAuthenticationExtensions(extensions)
            .setUser(user)
            .setAuthenticatorSelection(authenticatorSelection)
            .setParameters(params)
            .setChallenge(challenge)
            .setRp(publicKey)
            .build()

        val browserOptions = BrowserPublicKeyCredentialCreationOptions.Builder()
            .setPublicKeyCredentialCreationOptions(requestOptions)
            .setOrigin(origin.toUri())
            .build()

        val intentTask = fidoClient.getRegisterPendingIntent(browserOptions)

        intentTask.addOnSuccessListener { pendingIntent ->
            try {
                val intentSenderRequest = IntentSenderRequest.Builder(pendingIntent).build()
                hardwareKeyCreationGetResult.launch(intentSenderRequest)
            } catch (err: IntentSender.SendIntentException) {
                Log.d("OPA", "Error while trying to communicate to MicroG! $err")
                setResult(RESULT_CANCELED)
                finish()
                return@addOnSuccessListener
            }
        }
        intentTask.addOnFailureListener {
            setResult(RESULT_CANCELED)
            finish()
        }
    }

    @OptIn(ExperimentalSerializationApi::class)
    val hardwareKeyGetResult = registerForActivityResult(ActivityResultContracts.StartIntentSenderForResult()) {
        if (it.resultCode != RESULT_OK) {
            setResult(RESULT_CANCELED)
            finish()
            return@registerForActivityResult
        }

        val responseData =
            it.data?.getByteArrayExtra(Fido.FIDO2_KEY_CREDENTIAL_EXTRA) ?: run {
                setResult(RESULT_CANCELED)
                finish()
                return@registerForActivityResult
            }

        val publicKey =
            com.google.android.gms.fido.fido2.api.common.PublicKeyCredential.deserializeFromBytes(
                responseData
            )

        if (publicKey.response is AuthenticatorErrorResponse) {
            setResult(RESULT_CANCELED)
            finish()
            return@registerForActivityResult
        }

        val publicKeyData =
            PublicKeyCredentialData.parseMicroGPublicKeyCredentialData(
                base64,
                null,
                publicKey
            ) ?: run {
                setResult(RESULT_CANCELED)
                finish()
                return@registerForActivityResult
            }

        val response = GetCredentialResponse(
            PublicKeyCredential(
                json.encodeToString(publicKeyData)
            )
        )

        val responseIntent = Intent()
        PendingIntentHandler.setGetCredentialResponse(responseIntent, response)
        setResult(RESULT_OK, responseIntent)
        finish()
    }

    @OptIn(ExperimentalSerializationApi::class, ExperimentalMaterial3Api::class)
    @SuppressLint("RestrictedApi")
    @Composable
    private fun HandleHardwareKeyGetRequest(getRequest: ProviderGetCredentialRequest) {
        var showBottomSheet by remember { mutableStateOf(true) }
        if (showBottomSheet) {
            ModalBottomSheet(
                onDismissRequest = { },
                // sheetState = rememberModalBottomSheetState(),
            ) {
                val publicKeyRequests =
                    getRequest.credentialOptions as List<GetPublicKeyCredentialOption>

                val pubKeyRequest =
                    PublicKeyCredentialRequestOptions(publicKeyRequests[0].requestJson)

                val callingAppInfo = getRequest.callingAppInfo
                val origin =
                    OpenPasskeyAuthUtils.appInfoToOrigin(applicationContext, callingAppInfo)

                val clientDataHash = publicKeyRequests[0].clientDataHash
                val challenge = pubKeyRequest.challenge


                val request = publicKeyRequests[0].requestJson.parseGetCredentialRequest(json)

                if (request == null) {
                    setResult(RESULT_CANCELED)
                    finish()
                    return@ModalBottomSheet
                }

                val allowList = request.allowCredentials
                    .mapNotNull {
                        PublicKeyCredentialDescriptor(
                            PublicKeyCredentialType.PUBLIC_KEY.toString(),
                            OpenPasskeyAuthUtils.b64Decode(it.id) ?: return@mapNotNull null,
                            it.transports.map { transport ->
                                Transport.fromString(
                                    transport.value
                                )
                            }
                        )
                    }

                val fidoClient = Fido.getFido2PrivilegedApiClient(this@OpenPasskeyAuthActivity)
                val ext = AuthenticationExtensions.Builder().build()
                val requestOptions =
                    com.google.android.gms.fido.fido2.api.common.PublicKeyCredentialRequestOptions.Builder()
                        .setTimeoutSeconds(20.0)
                        .setAllowList(allowList)
                        .setAuthenticationExtensions(ext)
                        .setChallenge(challenge)
                        .setRpId(request.rpId)
                        .setRequireUserVerification(request.userVerification.asMicrogVerification())
                        .build()

                val browserOptions = BrowserPublicKeyCredentialRequestOptions.Builder()
                    .setPublicKeyCredentialRequestOptions(requestOptions)
                    .setOrigin(origin.toUri())
                    .apply { clientDataHash?.let { setClientDataHash(it) } }
                    .build()

                val intentTask = fidoClient.getSignPendingIntent(browserOptions)

                intentTask.addOnSuccessListener { pendingIntent ->
                    try {
                        val intentSenderRequest = IntentSenderRequest.Builder(pendingIntent).build()
                        hardwareKeyGetResult.launch(intentSenderRequest)
                    } catch (err: IntentSender.SendIntentException) {
                        Log.d("OPA", "Error while trying to communicate to MicroG! $err")
                        setResult(RESULT_CANCELED)
                        finish()
                        return@addOnSuccessListener
                    }
                }
            }
        }



            /*val response = it.response()

            if (response is de.cotech.hw.fido2.domain.get.AuthenticatorAssertionResponse) {
                val constructionResponse = HardwareKeyAuthenticationResponse(
                    clientData = response.clientDataJson(),
                    userHandle = response.userHandle(),
                    authenticatorData = response.authenticatorData(),
                    signature = response.signature(),
                )

                val passkeyCredentials = FidoPublicKeyCredential(
                    rawId = it.rawId(), response = constructionResponse,
                    authenticatorAttachment = "platform"
                )

                val result = Intent()
                val passkeyCredential = PublicKeyCredential(passkeyCredentials.json())

                PendingIntentHandler.setGetCredentialResponse(
                    result, GetCredentialResponse(passkeyCredential)
                )

                setResult(RESULT_OK, result)
                finish()
            }
            setResult(RESULT_CANCELED)
            finish()*/
    }

    /*private fun webauthnDialogOption(): WebauthnDialogOptions {
        var prefs = UserPreferences(true, false, false, false, false)

        runBlocking {
            try {
                prefs =
                    Locator.userPreferencesRepository.userPreferences.first() // Suspend until value is available

                Log.d("OPA", "Load prefs: $prefs")
            } catch (e: Exception) {
                // Handle errors (e.g., repository not initialized, network issue)
                Log.e("OpenPasskeyAuthActivity", "Error loading scramblePinLayout: ${e.message}", e)
            }
        }

        return WebauthnDialogOptions.builder()
            .setTitle("OPA: Hardware Passkey Communication")
            .setShowSdkLogo(false)
            .setPreventScreenshots(!prefs.allowScreenshots)
            .setScramblePinLayout(prefs.scramblePin)
            .setAllowKeyboard(prefs.allowKeyboard)
            .setTheme(R.style.HwSecurity_Dialog)
            .build()
    }*/


}