package de.noisruker.openPasskeyAuth.ui

import android.content.Intent
import android.os.Bundle
import android.security.keystore.KeyProperties
import android.view.WindowManager
import androidx.activity.compose.setContent
import androidx.annotation.StringRes
import androidx.compose.foundation.clickable
import androidx.compose.foundation.interaction.MutableInteractionSource
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.outlined.Delete
import androidx.compose.material.icons.outlined.DragHandle
import androidx.compose.material.icons.outlined.Edit
import androidx.compose.material.icons.outlined.Info
import androidx.compose.material.icons.outlined.Lock
import androidx.compose.material.icons.outlined.Settings
import androidx.compose.material3.Card
import androidx.compose.material3.CardDefaults
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.material3.TopAppBar
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.remember
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.core.net.toUri
import de.noisruker.openPasskeyAuth.Locator
import de.noisruker.openPasskeyAuth.R
import de.noisruker.openPasskeyAuth.ui.settings.SettingsActivity
import de.noisruker.openPasskeyAuth.ui.theme.OpenPasskeyAuthTheme
import de.noisruker.openPasskeyAuth.utils.CredentialDataManager
import de.noisruker.openPasskeyAuth.utils.OpenPasskeyAuthUtils
import de.noisruker.openPasskeyAuth.utils.preferences.UserPreferences
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.runBlocking
import sh.calvin.reorderable.DragGestureDetector
import sh.calvin.reorderable.ReorderableColumn

class MainActivity : BaseActivity() {

    @OptIn(ExperimentalMaterial3Api::class)
    override fun onResume() {
        super.onResume()

        var prefs: UserPreferences

        runBlocking {
            prefs = Locator.userPreferencesRepository.userPreferences.first()
        }

        if (!prefs.allowScreenshots)
            window.setFlags(WindowManager.LayoutParams.FLAG_SECURE, WindowManager.LayoutParams.FLAG_SECURE)
        else
            window.clearFlags(WindowManager.LayoutParams.FLAG_SECURE)


        setContent {
            OpenPasskeyAuthTheme {
                val isLockPrefSet by Locator.userPreferencesRepository.userPreferences.collectAsState(UserPreferences())

                if (!isLockPrefSet.appLock || Locator.appSecret != null) {
                    val credentialList = CredentialDataManager.loadAll(this@MainActivity)

                    Scaffold(
                        topBar = {
                            TopAppBar(
                                title = {
                                    Text(
                                        text = "${stringResource(R.string.title_activity_main)} - ${
                                            stringResource(
                                                R.string.passkeys
                                            )
                                        }"
                                    )
                                },
                                actions = {
                                    if (isLockPrefSet.appLock) {
                                        IconButton(onClick = {
                                            super.lockApp()
                                            recreate()
                                        }) {
                                            Icon(
                                                Icons.Outlined.Lock,
                                                contentDescription = stringResource(R.string.lock)
                                            )
                                        }
                                    }
                                    IconButton(onClick = {
                                        val intent =
                                            Intent(applicationContext, InfoActivity::class.java)
                                        startActivity(intent)
                                    }) {
                                        Icon(
                                            Icons.Outlined.Info,
                                            contentDescription = stringResource(R.string.info_activity_name)
                                        )
                                    }
                                    IconButton(onClick = {
                                        val intent =
                                            Intent(applicationContext, SettingsActivity::class.java)
                                        startActivity(intent)
                                    }) {
                                        Icon(
                                            Icons.Outlined.Settings,
                                            contentDescription = stringResource(R.string.title_activity_settings)
                                        )
                                    }
                                }
                            )
                        }
                    ) { paddingValues ->
                        Column(
                            modifier = Modifier.padding(paddingValues)
                        ) {
                            CredentialList(credentialList)
                        }
                    }
                }
            }
        }
    }

    @Composable
    fun CredentialList(credentials: MutableList <CredentialDataManager.Credential>) {
        ReorderableColumn(
            list = credentials,
            onSettle = { fromIndex, toIndex ->
                CredentialDataManager.moveCredPos(fromIndex, toIndex, this.applicationContext)
                recreate()
            },
            modifier = Modifier
                .verticalScroll(rememberScrollState())
                .padding(0.dp, 0.dp, 0.dp, 20.dp)
        ) { index, credential, isDragging ->
            val interactionSource = remember { MutableInteractionSource() }

            val preferences by Locator.userPreferencesRepository.userPreferences.collectAsState(
                UserPreferences())

            ReorderableItem {
                Card(
                    modifier = Modifier.padding(16.dp)
                        .clickable {
                            if (!credential.rpid.startsWith("android")) {
                                val browserIntent =
                                    Intent(Intent.ACTION_VIEW, "https://${credential.rpid}".toUri())
                                startActivity(browserIntent)
                            }
                        }.draggableHandle(
                            interactionSource = interactionSource,
                            dragGestureDetector = DragGestureDetector.LongPress
                        ),
                    elevation = CardDefaults.cardElevation(
                        defaultElevation = 6.dp
                    ),
                ) {
                    Row(
                    modifier = Modifier
                        .padding(horizontal = 16.dp, vertical = 12.dp)
                        .fillMaxWidth(),
                    verticalAlignment = Alignment.CenterVertically
                ) {
                        Column(
                            Modifier
                                .weight(1.0f),
                            verticalArrangement = Arrangement.spacedBy(8.dp)
                        ) {
                            Text(
                                text = credential.serviceName,
                                color = MaterialTheme.colorScheme.primary,
                                fontSize = 30.sp,
                                fontWeight = FontWeight.Bold,
                                maxLines = 1,
                                overflow = TextOverflow.Ellipsis
                            )
                            NamedTextElement(R.string.key_url, credential.rpid)
                            NamedTextElement(R.string.key_id, credential.displayName)
                            if (credential.description.isNotBlank()) {
                                NamedTextElement(R.string.key_description, credential.description)
                            }
                            when (credential.securityLevel) {
                                KeyProperties.SECURITY_LEVEL_STRONGBOX -> NamedTextElement(
                                    R.string.security_status,
                                    stringResource(R.string.key_strong_box)
                                )

                                KeyProperties.SECURITY_LEVEL_TRUSTED_ENVIRONMENT -> NamedTextElement(
                                    R.string.security_status,
                                    stringResource(R.string.key_trusted)
                                )

                                KeyProperties.SECURITY_LEVEL_SOFTWARE -> NamedTextElement(
                                    R.string.security_status,
                                    stringResource(R.string.key_software),
                                    MaterialTheme.colorScheme.error
                                )

                                else -> NamedTextElement(
                                    R.string.security_status,
                                    stringResource(R.string.key_unknown),
                                    MaterialTheme.colorScheme.error
                                )
                            }
                        }
                        Column(
                            verticalArrangement = Arrangement.spacedBy(12.dp)
                        ) {
                            IconButton(onClick = {
                                val intent =
                                    Intent(applicationContext, EditCredentialActivity::class.java)

                                val data = Bundle()
                                data.putString(
                                    EditCredentialActivity.CREDENTIAL_ID,
                                    OpenPasskeyAuthUtils.b64Encode(credential.credentialId)
                                )
                                data.putString(
                                    EditCredentialActivity.RP_ID,
                                    credential.rpid
                                )

                                intent.putExtra(EditCredentialActivity.EDIT_CREDENTIAL_DATA, data)
                                startActivity(intent)
                            }) {
                                Icon(
                                    Icons.Outlined.Edit,
                                    tint = MaterialTheme.colorScheme.primary,
                                    contentDescription = stringResource(R.string.edit_key_description)
                                )
                            }
                            IconButton(
                                modifier = Modifier.draggableHandle(interactionSource = interactionSource),
                                onClick = {}) {
                                Icon(
                                    Icons.Outlined.DragHandle,
                                    tint = MaterialTheme.colorScheme.primary,
                                    contentDescription = stringResource(R.string.reorder)
                                )
                            }
                            IconButton(onClick = {
                                deleteValidation(
                                    applicationContext = this@MainActivity.applicationContext,
                                    prefs = preferences,
                                    deleteCredential = credential
                                )
                            }) {
                                Icon(
                                    Icons.Outlined.Delete,
                                    tint = MaterialTheme.colorScheme.error,
                                    contentDescription = stringResource(R.string.delete)
                                )
                            }
                        }
                    }
                }
            }
        }
    }

    @Composable
    fun NamedTextElement(@StringRes name: Int, text: String, color: Color = MaterialTheme.colorScheme.secondary) {
        Row {
            Text(text = "${stringResource(name)}: ", color = MaterialTheme.colorScheme.secondary, fontWeight = FontWeight.Bold, maxLines = 1, overflow = TextOverflow.Ellipsis)
            Text(text = text, color = color, maxLines = 1, overflow = TextOverflow.Ellipsis)
        }
    }

    @Preview
    @Composable
    fun PreviewCredentialEntry() {
        val credentials = mutableListOf(
            CredentialDataManager.Credential(serviceName = "Sample1", rpid = "www.example.com", displayName="Test 1", credentialId = byteArrayOf(0x01)),
            CredentialDataManager.Credential(serviceName = "Sample1", rpid = "www.example.com", displayName="Test 2", credentialId = byteArrayOf(0x02)),
            CredentialDataManager.Credential(serviceName = "Sample1", rpid = "www.example.com", displayName="Test 3", credentialId = byteArrayOf(0x03), description = "Test"),
            )
        CredentialList(credentials)
    }

}