package de.noisruker.openPasskeyAuth.ui

import android.security.keystore.KeyProperties
import androidx.activity.compose.setContent
import androidx.annotation.StringRes
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material.icons.filled.Save
import androidx.compose.material.icons.outlined.Delete
import androidx.compose.material.icons.outlined.Lock
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.FabPosition
import androidx.compose.material3.FloatingActionButton
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.material3.TopAppBar
import androidx.compose.runtime.Composable
import androidx.compose.runtime.MutableState
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.unit.dp
import de.noisruker.openPasskeyAuth.Locator
import de.noisruker.openPasskeyAuth.R
import de.noisruker.openPasskeyAuth.ui.settings.SettingsActivity
import de.noisruker.openPasskeyAuth.ui.theme.OpenPasskeyAuthTheme
import de.noisruker.openPasskeyAuth.utils.CredentialDataManager
import de.noisruker.openPasskeyAuth.utils.OpenPasskeyAuthUtils
import de.noisruker.openPasskeyAuth.utils.preferences.UserPreferences

class EditCredentialActivity: BaseActivity() {

    companion object {
        const val EDIT_CREDENTIAL_DATA = "de.noisruker.openPasskeyAuth.CREDENTIAL_DATA"
        const val CREDENTIAL_ID = "credentialId"
        const val RP_ID = "rpId"
    }

    @OptIn(ExperimentalMaterial3Api::class)
    override fun onResume() {
        super.onResume()

        if (intent == null) {
            finish()
            return
        }

        val extras = intent.getBundleExtra(EDIT_CREDENTIAL_DATA)

        if (extras == null) {
            finish()
            return
        }

        val credentialId = OpenPasskeyAuthUtils.b64Decode(extras.getString(CREDENTIAL_ID))
        val rpId = extras.getString(RP_ID)

        if (credentialId == null || rpId == null) {
            finish()
            return
        }

        val credential = CredentialDataManager.load(this.applicationContext, rpId, credentialId)

        if (credential == null) {
            finish()
            return
        }

        setContent {
            OpenPasskeyAuthTheme {
                val preferences by Locator.userPreferencesRepository.userPreferences.collectAsState(
                    UserPreferences())
                val description = remember { mutableStateOf(credential.description) }
                val serviceName = remember { mutableStateOf(credential.serviceName) }
                val displayName = remember { mutableStateOf(credential.displayName) }

                Scaffold(
                    topBar = @Composable {
                        TopAppBar(
                            title = { Text(text = stringResource(R.string.title_activity_edit_credential)) },
                            navigationIcon = {
                                IconButton(onClick = {
                                    finish()
                                }) {
                                    Icon(Icons.AutoMirrored.Filled.ArrowBack, contentDescription = stringResource(R.string.back))
                                }
                            },
                            actions = {
                                if (preferences.appLock) {
                                    IconButton(onClick = {
                                        super.lockApp()
                                        recreate()
                                    }) {
                                        Icon(
                                            Icons.Outlined.Lock,
                                            contentDescription = stringResource(R.string.lock)
                                        )
                                    }
                                }
                                IconButton(onClick = {
                                    deleteValidation(applicationContext = this@EditCredentialActivity.applicationContext, prefs = preferences, deleteCredential = credential)
                                }) {
                                    Icon(
                                        Icons.Outlined.Delete,
                                        tint = MaterialTheme.colorScheme.error,
                                        contentDescription = stringResource(R.string.delete)
                                    )
                                }
                            }
                        )
                    },
                    floatingActionButton = @Composable {
                        FloatingActionButton(onClick = {
                            CredentialDataManager.changeCredentialData(this.applicationContext, rpId, credentialId, serviceName.value, displayName.value, description.value)
                            finish()
                        },
                        ) {
                            Icon(Icons.Filled.Save, stringResource(R.string.save))
                        }
                    },
                    floatingActionButtonPosition = FabPosition.End,
                    content = @Composable { padding ->
                        Column(modifier = Modifier.padding(padding).verticalScroll(
                            rememberScrollState()
                        )) {
                            CredentialEditView(credential, description, serviceName, displayName)
                        }
                    }
                )
            }
        }
    }

    @Composable
    fun CredentialEditView(credential: CredentialDataManager.Credential, description: MutableState<String>, serviceName: MutableState<String>, displayName: MutableState<String>) {
        val editAllEntries = remember { mutableStateOf(false) }

        Column (
            modifier = Modifier
                .fillMaxWidth()
                .padding(horizontal = 16.dp, vertical = 8.dp),
            verticalArrangement = Arrangement.spacedBy(16.dp)
        ) {
            Text (
                stringResource(R.string.credential_data),
                color = MaterialTheme.colorScheme.primary,
                fontWeight = FontWeight.Bold,
                maxLines = 1,
                overflow = TextOverflow.Ellipsis
            )

            NamedTextElement(R.string.service_name, serviceName.value, { serviceName.value = it }, editAllEntries.value)
            NamedTextElement(R.string.key_url, credential.rpid)
            NamedTextElement(R.string.key_id, displayName.value, { displayName.value = it }, editAllEntries.value)
            NamedTextElement(R.string.key_description, description.value, { description.value = it }, true)

            when (credential.securityLevel) {
                KeyProperties.SECURITY_LEVEL_STRONGBOX -> NamedTextElement(R.string.security_status, stringResource(R.string.key_strong_box))
                KeyProperties.SECURITY_LEVEL_TRUSTED_ENVIRONMENT -> NamedTextElement(R.string.security_status, stringResource(R.string.key_trusted))
                KeyProperties.SECURITY_LEVEL_SOFTWARE -> NamedTextElement(R.string.security_status, stringResource(R.string.key_software))
                else -> NamedTextElement(R.string.security_status, stringResource(R.string.key_unknown))
            }

            Column (
                modifier = Modifier.fillMaxWidth().padding(vertical = 32.dp),
                verticalArrangement = Arrangement.spacedBy(8.dp)
            ) {
                Text (
                    stringResource(R.string.options),
                    color = MaterialTheme.colorScheme.primary,
                    fontWeight = FontWeight.Bold,
                    maxLines = 1,
                    overflow = TextOverflow.Ellipsis
                )

                SettingsActivity.SettingSwitchItem (
                    checked = editAllEntries.value,
                    onCheckedChange = { editAllEntries.value = it },
                    title = R.string.allow_editing_all_entries,
                    description = null,
                    enabled = true,
                )
            }
        }
    }

    @Composable
    fun NamedTextElement(@StringRes name: Int, value: String, onValueChange: (String) -> Unit = {}, enabled: Boolean = false) {
        Column (
            modifier = Modifier.fillMaxWidth(),
        ) {
            OutlinedTextField (
                value,
                onValueChange = onValueChange,
                enabled = enabled,
                label = {
                    Text (
                        text = "${stringResource(name)}:",
                        color = MaterialTheme.colorScheme.primary,
                        fontWeight = FontWeight.Bold,
                        maxLines = 1,
                        overflow = TextOverflow.Ellipsis
                    )
                },
                modifier = Modifier.fillMaxWidth()
            )
        }
    }

}