package de.noisruker.openPasskeyAuth.ui

import android.app.Application
import android.os.Bundle
import android.os.PersistableBundle
import android.util.Log
import androidx.activity.compose.setContent
import androidx.appcompat.app.AppCompatActivity
import androidx.biometric.BiometricManager
import androidx.biometric.BiometricPrompt
import androidx.biometric.BiometricPrompt.CryptoObject
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.size
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.outlined.Lock
import androidx.compose.material3.Icon
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.credentials.provider.PendingIntentHandler
import de.noisruker.openPasskeyAuth.Locator
import de.noisruker.openPasskeyAuth.OpenPasskeyAuthService
import de.noisruker.openPasskeyAuth.OpenPasskeyAuthServiceUtils
import de.noisruker.openPasskeyAuth.R
import de.noisruker.openPasskeyAuth.ui.theme.OpenPasskeyAuthTheme
import de.noisruker.openPasskeyAuth.utils.OpenPasskeyAuthUtils
import de.noisruker.openPasskeyAuth.utils.preferences.UserPreferences
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.runBlocking
import java.security.Key
import java.security.KeyStore
import java.util.concurrent.Executor
import javax.crypto.Cipher
import javax.crypto.SecretKey
import javax.crypto.spec.GCMParameterSpec

class BiometricUnlock: AppCompatActivity() {

    fun interface IAppUnlockSuccessHandler {
        fun success(secret: String?, iv: ByteArray?)
    }

    override fun onCreate(savedInstanceState: Bundle?, persistentState: PersistableBundle?) {
        super.onCreate(savedInstanceState, persistentState)
    }

    override fun onResume() {
        super.onResume()

        val successHandler = IAppUnlockSuccessHandler { success, _ ->
            Locator.appSecret = success

            setResult(RESULT_OK)

            Log.d("OPA", "Successfully unlocked!")
            if (intent.action == OpenPasskeyAuthService.UNLOCK_APP_INTENT_ACTION) {
                Log.d("OPA", "Preparing Request!")
                val request = PendingIntentHandler.retrieveBeginGetCredentialRequest(intent)
                if (request != null) {
                    Log.d("OPA", "Answer request!")
                    val response = OpenPasskeyAuthServiceUtils.processGetCredentialRequest(
                        this.application,
                        this,
                        request
                    )

                    PendingIntentHandler.setBeginGetCredentialResponse(intent, response)
                    setResult(RESULT_OK, intent)
                }
            }

            finish()
        }

        if (Locator.appSecret != null) {
            successHandler.success(Locator.appSecret, null)
            return
        }

        val cipher = unlockAppSecretCypher()
        val promptInfo = biometricPromptInfo(this.application)
        val biometricPrompt = unlockAppSecretPromt(successHandler, mainExecutor, this)

        if (cipher == null) {
            setContent {
                Text(stringResource(R.string.error_unlocking))
            }
            return
        }

        biometricPrompt.authenticate(promptInfo, CryptoObject(cipher))

        setContent {
            OpenPasskeyAuthTheme {
                TextButton(
                    onClick = {
                        biometricPrompt.authenticate(promptInfo, CryptoObject(cipher))
                    },
                    modifier = Modifier
                        .fillMaxWidth()
                        .fillMaxHeight()
                ) {
                    Column (
                        verticalArrangement = Arrangement.spacedBy(30.dp)
                    ) {
                        Icon(
                            Icons.Outlined.Lock,
                            contentDescription = null,
                            modifier = Modifier
                                .align(Alignment.CenterHorizontally)
                                .size(80.dp)
                        )
                        Text(text = stringResource(R.string.unlock), fontSize = 60.sp, fontWeight = FontWeight.Bold)
                    }

                }
            }
        }
    }

    companion object {
        fun unlockAppSecretCypher(): Cipher? {
            var prefs: UserPreferences? = null

            runBlocking {
                prefs = Locator.userPreferencesRepository.userPreferences.first()
            }

            if (prefs == null) {
                return null
            }

            val keyStore = KeyStore.getInstance("AndroidKeyStore").apply {
                load(null)
            }
            val entry: Key = keyStore.getKey("opa_lock_app_key", null)

            if (entry !is SecretKey || prefs.appUnlockSecret == null || prefs.appUnlockIV == null) {
                return null
            }

            return Cipher.getInstance(Locator.UNLOCK_ENCRYPTION_TRANSFORMATION).run {
                init(Cipher.DECRYPT_MODE, entry, GCMParameterSpec(128, OpenPasskeyAuthUtils.getIV(prefs.appUnlockIV).iv))
                this
            }
        }

        fun encryptAppSecretCypher(): Cipher? {
            val keyStore = KeyStore.getInstance("AndroidKeyStore").apply {
                load(null)
            }
            val entry = keyStore.getKey("opa_lock_app_key", null)



            return Cipher.getInstance(Locator.UNLOCK_ENCRYPTION_TRANSFORMATION).run {
                init(Cipher.ENCRYPT_MODE, entry)
                this
            }
        }

        fun biometricPromptInfo(application: Application): BiometricPrompt.PromptInfo {
            return BiometricPrompt.PromptInfo.Builder()
                .setTitle(application.getString(R.string.use_screen_lock))
                .setSubtitle(
                    "${application.getString(R.string.unlock)} ${
                        application.getString(
                            R.string.app_name
                        )
                    }"
                )
                .setAllowedAuthenticators(BiometricManager.Authenticators.BIOMETRIC_STRONG)
                .setNegativeButtonText(application.getString(R.string.cancel))
                .build()
        }

        fun unlockAppSecretPromt(successHandler: IAppUnlockSuccessHandler, mainExecutor: Executor, activity: AppCompatActivity): BiometricPrompt {
            val biometricPrompt = BiometricPrompt(
                activity,
                mainExecutor,
                object : BiometricPrompt.AuthenticationCallback() {
                    override fun onAuthenticationError(errorCode: Int, errString: CharSequence) {
                        super.onAuthenticationError(errorCode, errString)
                    }

                    override fun onAuthenticationFailed() {
                        super.onAuthenticationFailed()
                    }

                    override fun onAuthenticationSucceeded(result: BiometricPrompt.AuthenticationResult) {
                        super.onAuthenticationSucceeded(result)

                        val cipher = result.cryptoObject?.cipher ?: return

                        var prefs: UserPreferences

                        runBlocking {
                            prefs = Locator.userPreferencesRepository.userPreferences.first()
                        }

                        val data = cipher.doFinal(OpenPasskeyAuthUtils.b64Decode(prefs.appUnlockSecret))

                        val result = OpenPasskeyAuthUtils.b64Encode(data)

                        successHandler.success(result, cipher.iv)
                    }
                })

            return biometricPrompt
        }
    }
}