package de.noisruker.openPasskeyAuth.ui

import android.content.Context
import android.content.Intent
import android.content.pm.PackageManager
import android.os.Bundle
import android.os.PersistableBundle
import android.security.keystore.KeyGenParameterSpec
import android.security.keystore.KeyProperties
import android.security.keystore.StrongBoxUnavailableException
import android.util.Log
import androidx.appcompat.app.AppCompatActivity
import androidx.biometric.BiometricManager
import androidx.biometric.BiometricPrompt
import androidx.biometric.BiometricPrompt.CryptoObject
import de.noisruker.openPasskeyAuth.Locator
import de.noisruker.openPasskeyAuth.R
import de.noisruker.openPasskeyAuth.utils.CredentialDataManager
import de.noisruker.openPasskeyAuth.utils.OpenPasskeyAuthUtils
import de.noisruker.openPasskeyAuth.utils.preferences.UserPreferences
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.runBlocking
import java.security.KeyStore
import javax.crypto.KeyGenerator

abstract class BaseActivity: AppCompatActivity() {

    override fun onCreate(savedInstanceState: Bundle?, persistentState: PersistableBundle?) {
        super.onCreate(savedInstanceState, persistentState)
    }

    override fun onResume() {
        super.onResume()

        var prefs: UserPreferences?

        if (!Locator.isLoaded()) {
            Locator.initWith(this.application)
        }

        runBlocking {
            prefs = Locator.userPreferencesRepository.userPreferences.first()
        }

        Log.d("OPA", "Unlocked in resume: ${prefs?.appUnlockSecret}")

        if (prefs?.appLock == true && Locator.appSecret == null) {
            val unlockIntent = Intent(this, BiometricUnlock::class.java)
            startActivity(unlockIntent)
        } else if (prefs?.appLock != true) {
            Locator.appSecret = prefs?.appUnlockSecret
        }

        val ks: KeyStore = KeyStore.getInstance("AndroidKeyStore").apply {
            load(null)
        }

        val aliases = ks.aliases().toList()

        if ("opa_lock_app_key" !in aliases) {
            generateAppLockKey(ks)
        }

        runBlocking {
            val preferences = Locator.userPreferencesRepository.userPreferences.first()

            if (Locator.appSecret == null && (preferences.appUnlockSecret == null || preferences.appUnlockSecret.isBlank())) {
                Log.d("OPA", "Generate App Lock Secret")

                val keyGenerator = KeyGenerator.getInstance("AES")
                keyGenerator.init(256) // You can also use 128 or 192 bits
                val key = keyGenerator.generateKey()

                val encodedKey = OpenPasskeyAuthUtils.b64Encode(key.encoded)

                Locator.userPreferencesRepository.setAppUnlockSecret(encodedKey)
                Locator.userPreferencesRepository.enableScreenLock(false)

                Locator.appSecret = encodedKey

                CredentialDataManager.updateEncrypt(this@BaseActivity)
            }
        }
    }

    protected fun generateAppLockKey(ks: KeyStore) {
        ks.deleteEntry("opa_lock_app_key")

        Log.d("OPA", "Generate App Lock Keys")
        val spec = KeyGenParameterSpec.Builder(
            "opa_lock_app_key",
            KeyProperties.PURPOSE_ENCRYPT or KeyProperties.PURPOSE_DECRYPT
        ).run {
            setBlockModes(KeyProperties.BLOCK_MODE_GCM)
            setEncryptionPaddings(KeyProperties.ENCRYPTION_PADDING_NONE)
            setRandomizedEncryptionRequired(false)

            if (applicationContext.packageManager.hasSystemFeature(PackageManager.FEATURE_STRONGBOX_KEYSTORE)) {
                setIsStrongBoxBacked(true)
            }
            setUserAuthenticationRequired(true)
            setUserAuthenticationParameters(0, KeyProperties.AUTH_BIOMETRIC_STRONG)
            setUnlockedDeviceRequired(true)
            setInvalidatedByBiometricEnrollment(false)

            build()
        }
        val keyPairGen = KeyGenerator.getInstance(KeyProperties.KEY_ALGORITHM_AES, "AndroidKeyStore")

        try {
            keyPairGen.init(spec)
        } catch (_: StrongBoxUnavailableException) {
            val spec = KeyGenParameterSpec.Builder(
                "opa_lock_app_key",
                KeyProperties.PURPOSE_ENCRYPT or KeyProperties.PURPOSE_DECRYPT
            ).run {
                setBlockModes(KeyProperties.BLOCK_MODE_GCM)
                setEncryptionPaddings(KeyProperties.ENCRYPTION_PADDING_NONE)
                setRandomizedEncryptionRequired(false)

                setUserAuthenticationRequired(true)
                setUserAuthenticationParameters(0, KeyProperties.AUTH_BIOMETRIC_STRONG)
                setUnlockedDeviceRequired(true)
                setInvalidatedByBiometricEnrollment(false)

                build()
            }
            keyPairGen.init(spec)
        } catch(_: Exception) {
            Log.e("OPA", "App locking not available")
            return
        }
        keyPairGen.generateKey()
    }

    protected fun lockApp() {
        Locator.appSecret = null
        recreate()
    }

    protected fun deleteValidation(applicationContext: Context, prefs: UserPreferences, deleteCredential: CredentialDataManager.Credential) {
        Locator.appSecret = null

        val biometricPrompt = BiometricPrompt(
            this,
            mainExecutor,
            object : BiometricPrompt.AuthenticationCallback() {
                override fun onAuthenticationError(errorCode: Int, errString: CharSequence) {
                    super.onAuthenticationError(errorCode, errString)
                }

                override fun onAuthenticationFailed() {
                    super.onAuthenticationFailed()
                }

                override fun onAuthenticationSucceeded(result: BiometricPrompt.AuthenticationResult) {
                    super.onAuthenticationSucceeded(result)

                    if (prefs.appLock) {
                        val cipher = result.cryptoObject?.cipher

                        if (cipher == null) return

                        val data = cipher.doFinal(OpenPasskeyAuthUtils.b64Decode(prefs.appUnlockSecret))

                        val result = OpenPasskeyAuthUtils.b64Encode(data)

                        Locator.appSecret = result
                    } else {
                        Locator.appSecret = prefs.appUnlockSecret
                    }

                    CredentialDataManager.delete(applicationContext, deleteCredential.rpid, deleteCredential.credentialId)
                    recreate()
                }
            })

        val promptInfo = BiometricPrompt.PromptInfo.Builder()
            .setTitle(application.getString(R.string.use_screen_lock))
            .setSubtitle(
                "${application.getString(R.string.delete_key)} ${
                    deleteCredential.displayName
                }"
            )
            .setAllowedAuthenticators(BiometricManager.Authenticators.BIOMETRIC_STRONG)
            .setNegativeButtonText(application.getString(R.string.cancel))
            .build()

        if (prefs.appLock) {
            val cipher = BiometricUnlock.unlockAppSecretCypher()

            if (cipher == null) return

            biometricPrompt.authenticate(promptInfo, CryptoObject(cipher))
        } else {
            biometricPrompt.authenticate(promptInfo)
        }
    }
}