package de.noisruker.openPasskeyAuth

import android.annotation.SuppressLint
import android.app.Application
import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import android.os.Bundle
import android.util.Log
import androidx.credentials.exceptions.GetCredentialUnsupportedException
import androidx.credentials.provider.AuthenticationAction
import androidx.credentials.provider.BeginCreateCredentialRequest
import androidx.credentials.provider.BeginCreateCredentialResponse
import androidx.credentials.provider.BeginCreatePublicKeyCredentialRequest
import androidx.credentials.provider.BeginGetCredentialRequest
import androidx.credentials.provider.BeginGetCredentialResponse
import androidx.credentials.provider.BeginGetPasswordOption
import androidx.credentials.provider.BeginGetPublicKeyCredentialOption
import androidx.credentials.provider.CallingAppInfo
import androidx.credentials.provider.CreateEntry
import androidx.credentials.provider.CredentialEntry
import androidx.credentials.provider.PublicKeyCredentialEntry
import androidx.credentials.webauthn.PublicKeyCredentialRequestOptions
import de.noisruker.openPasskeyAuth.OpenPasskeyAuthService.Companion.ACCOUNT_ID
import de.noisruker.openPasskeyAuth.OpenPasskeyAuthService.Companion.CREATE_PASSKEY_INTENT_ACTION
import de.noisruker.openPasskeyAuth.OpenPasskeyAuthService.Companion.CREDENTIAL_DATA_EXTRA
import de.noisruker.openPasskeyAuth.OpenPasskeyAuthService.Companion.CREDENTIAL_ID
import de.noisruker.openPasskeyAuth.OpenPasskeyAuthService.Companion.DEVICE_ACCOUNT
import de.noisruker.openPasskeyAuth.OpenPasskeyAuthService.Companion.GET_PASSKEY_INTENT_ACTION
import de.noisruker.openPasskeyAuth.OpenPasskeyAuthService.Companion.HARDWARE_KEY_ACCOUNT
import de.noisruker.openPasskeyAuth.OpenPasskeyAuthService.Companion.UNLOCK_APP_INTENT_ACTION
import de.noisruker.openPasskeyAuth.utils.CredentialDataManager
import de.noisruker.openPasskeyAuth.utils.OpenPasskeyAuthUtils
import de.noisruker.openPasskeyAuth.utils.preferences.UserPreferences
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.runBlocking

object OpenPasskeyAuthServiceUtils {

    /**
     * Gathers all available passkeys and creates the response for the user to choose which credential to use for answering the [request].
     *
     * Note: Only Passkeys are supported!
     */
    fun processGetCredentialRequest(application: Application, context: Context, request: BeginGetCredentialRequest): BeginGetCredentialResponse {
        val credentialEntries = mutableListOf<CredentialEntry>()
        val authenticationActions = mutableListOf<AuthenticationAction>()
        for (option in request.beginGetCredentialOptions) {
            when (option) {
                is BeginGetPasswordOption -> {
                    Log.i("JuhuCredMan", "Password not supported")
                    throw GetCredentialUnsupportedException("Password not supported")
                }

                is BeginGetPublicKeyCredentialOption -> {
                    if (Locator.appSecret == null) {
                        var prefs: UserPreferences?

                        if (!Locator.isLoaded()) {
                            Locator.initWith(application)
                        }

                        runBlocking {
                            prefs = Locator.userPreferencesRepository.userPreferences.first()
                        }

                        Log.d("OPA", "Unlocked in passkey request: ${prefs?.appUnlockSecret}")

                        if (prefs?.appLock == true && Locator.appSecret == null) {
                            val data = Bundle()

                            authenticationActions.add(
                                AuthenticationAction(
                                    title = application.getString(R.string.opa_locked),
                                    pendingIntent = createPendingIntent(
                                        application,
                                        UNLOCK_APP_INTENT_ACTION,
                                        data
                                    ),
                                )
                            )
                        } else if (prefs?.appLock != true) {
                            Locator.appSecret = prefs?.appUnlockSecret
                        }
                    }

                    credentialEntries.add(getHardwareKeyCredentialEntry(application, context, option))

                    credentialEntries.addAll(
                        getKeyCredentialEntries(
                            application,
                            context,
                            option,
                            request.callingAppInfo!!
                        )
                    )
                }

                else -> {
                    Log.i("JuhuCredMan", "Request not supported")
                    throw GetCredentialUnsupportedException("Request not supported")
                }
            }
        }



        return BeginGetCredentialResponse(
            credentialEntries,
            authenticationActions = authenticationActions
        )
    }

    /**
     * Create a key get credential entry for every saved passkey, that matches the request specified by [option] and the [callingAppInfo]
     */
    @SuppressLint("RestrictedApi")
    private fun getKeyCredentialEntries(
        application: Application,
        context: Context,
        option: BeginGetPublicKeyCredentialOption,
        callingAppInfo: CallingAppInfo
    ): List<CredentialEntry> {
        val request = PublicKeyCredentialRequestOptions(option.requestJson)

        val rpId = OpenPasskeyAuthUtils.validateRpId(context, callingAppInfo, request.rpId)


        // Retrieve Passkey from database
        val credentials = CredentialDataManager.load(context, rpId)

        return credentials.map {
            val data = Bundle()
            data.putString(
                CREDENTIAL_ID,
                OpenPasskeyAuthUtils.b64Encode(it.credentialId)
            )

            var builder = PublicKeyCredentialEntry.Builder(
                context = context,
                username = it.description.ifBlank { it.displayName },
                pendingIntent = createPendingIntent(application, GET_PASSKEY_INTENT_ACTION, data),
                beginGetPublicKeyCredentialOption = option
            )

            if (it.description.isNotBlank())
                builder = builder.setDisplayName(it.displayName)
            builder.build()
        }
    }

    /**
     * Creates a hardware key get credential entry for the PublicKeyRequest defined by [option]
     */
    private fun getHardwareKeyCredentialEntry(application: Application, context: Context, option: BeginGetPublicKeyCredentialOption): CredentialEntry {
        val intent = createPendingIntent(application, GET_PASSKEY_INTENT_ACTION)

        return PublicKeyCredentialEntry.Builder(
            username = "FiDO Hardware Token",
            context = context,
            pendingIntent = intent,
            beginGetPublicKeyCredentialOption = option,
        ).build()
    }

    /**
     * Creates a new pending intent of type [action] for this app.
     *
     * @param extra: Additional input parameters put as extra with name [CREDENTIAL_DATA_EXTRA]
     */
    private fun createPendingIntent(
        application: Application,
        action: String,
        extra: Bundle? = null
    ): PendingIntent {
        val intent = Intent(action).setPackage(application.packageName)

        if (extra != null) {
            intent.putExtra(CREDENTIAL_DATA_EXTRA, extra)
        }

        val requestCode = (1..9999).random()

        return PendingIntent.getActivity(
            application.applicationContext, requestCode, intent,
            (PendingIntent.FLAG_MUTABLE or PendingIntent.FLAG_UPDATE_CURRENT)
        )
    }

    fun processCreateCredentialRequest(application: Application, request: BeginCreateCredentialRequest): BeginCreateCredentialResponse? {
        return when (request) {
            is BeginCreatePublicKeyCredentialRequest -> {
                handleCreatePasskeyQuery(application)
            }

            else -> {
                null
            }
        }
    }

    private fun handleCreatePasskeyQuery(application: Application): BeginCreateCredentialResponse {
        val createEntries: MutableList<CreateEntry> = mutableListOf()

        val device = Bundle()
        device.putString(ACCOUNT_ID, DEVICE_ACCOUNT)

        createEntries.add(
            CreateEntry(
                DEVICE_ACCOUNT, createPendingIntent(
                    application, CREATE_PASSKEY_INTENT_ACTION, device
                )
            )
        )

        val key = Bundle()
        key.putString(ACCOUNT_ID, HARDWARE_KEY_ACCOUNT)

        createEntries.add(
            CreateEntry(
                HARDWARE_KEY_ACCOUNT, createPendingIntent(
                    application,
                    CREATE_PASSKEY_INTENT_ACTION, key
                )
            )
        )


        return BeginCreateCredentialResponse(createEntries)
    }

}