package de.noisruker.openPasskeyAuth

import android.app.Application
import android.content.Context
import android.util.Log
import androidx.datastore.preferences.preferencesDataStore
import de.noisruker.openPasskeyAuth.ui.settings.SettingsViewModel
import de.noisruker.openPasskeyAuth.utils.OpenPasskeyAuthUtils
import de.noisruker.openPasskeyAuth.utils.preferences.UserManager
import de.noisruker.openPasskeyAuth.utils.preferences.usecase.AllowKeyboard
import de.noisruker.openPasskeyAuth.utils.preferences.usecase.AppUnlockSecret
import de.noisruker.openPasskeyAuth.utils.preferences.usecase.AskForStrongBoxKeyCreation
import de.noisruker.openPasskeyAuth.utils.preferences.usecase.EnableAppLock
import de.noisruker.openPasskeyAuth.utils.preferences.usecase.EnableScramblePinLayout
import de.noisruker.openPasskeyAuth.utils.preferences.usecase.EnableScreenshots
import de.noisruker.openPasskeyAuth.utils.preferences.usecase.StrongBoxKey
import de.noisruker.openPasskeyAuth.utils.preferences.usecase.GetSettings
import javax.crypto.SecretKey
import javax.crypto.spec.SecretKeySpec

object Locator {

    const val UNLOCK_ENCRYPTION_TRANSFORMATION = "AES/GCM/NoPadding"
    const val FIXED_IV = ""

    private var application: Application? = null

    private inline val requireApplication
        get() = application ?: error("Missing call: initWith(application)")

    fun initWith(application: Application?) {
        this.application = application
    }

    fun isLoaded(): Boolean {
        return application != null
    }

    /**
     * App secret. This is cleared if the App gets locked.
     */
    var appSecret: String? = null

    fun getSecretKey(): SecretKeySpec? {
        return appSecret?.let { SecretKeySpec(OpenPasskeyAuthUtils.b64Decode(it), "AES") }
    }

    val settingsViewModelFactory
        get() = SettingsViewModel.Factory(
            strongBoxKey = strongBoxKey,
            askForStrongBoxKeyCreation = askForStrongBoxKeyCreation,
            allowKeyboard = allowKeyboard,
            enableScreenshots = enableScreenshots,
            enableScramblePinLayout = enableScramblePinLayout,
            appLock = appLock,
            getSettings = getSettings,
        )

    private val strongBoxKey get() = StrongBoxKey(userPreferencesRepository)

    private val allowKeyboard get() = AllowKeyboard(userPreferencesRepository)

    private val enableScreenshots get() = EnableScreenshots(userPreferencesRepository)

    private val enableScramblePinLayout get() = EnableScramblePinLayout(userPreferencesRepository)

    private val getSettings get() = GetSettings(userPreferencesRepository)

    private val appLock get() = EnableAppLock(userPreferencesRepository)

    private val askForStrongBoxKeyCreation get() = AskForStrongBoxKeyCreation(userPreferencesRepository)

    /**
     * This is the encrypted Secret if appLock is activated.
     * Use appSecret to get the unencrypted one. It is only set while the app is unlocked!
     */
    private val appUnlockSecret get() = AppUnlockSecret(userPreferencesRepository)

    private val Context.dataStore by preferencesDataStore(name = "settings")

    val userPreferencesRepository by lazy {
        Log.d("OPA", "try init: ${application.toString()}")
        UserManager(requireApplication.dataStore)
    }

}