package de.lxtools.noteshop.ui.webapp

import android.app.Application
import android.content.Context
import android.widget.Toast
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.setValue
import androidx.lifecycle.AndroidViewModel
import androidx.lifecycle.viewModelScope
import de.lxtools.noteshop.R
import de.lxtools.noteshop.data.NoteshopRepository
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.launch

class WebAppIntegrationViewModel(private val repository: NoteshopRepository, application: Application) : AndroidViewModel(application) {

    private val sharedPrefs = application.getSharedPreferences("webapp_prefs", Context.MODE_PRIVATE)

    var webAppUrl by mutableStateOf("")
        private set
    var username by mutableStateOf("")
        private set
    var password by mutableStateOf("")
        private set
    var deletePassword by mutableStateOf("")
        private set

    private val _isWebAppEnabled = MutableStateFlow(false)
    val isWebAppEnabled: StateFlow<Boolean> = _isWebAppEnabled

    private val _showPassword = mutableStateOf(false)
    val showPassword: Boolean by _showPassword

    private val _showDeletePassword = mutableStateOf(false)
    val showDeletePassword: Boolean by _showDeletePassword

    init {
        loadCredentials()
    }

    fun togglePasswordVisibility() {
        _showPassword.value = !_showPassword.value
        if (_showPassword.value) {
            viewModelScope.launch {
                kotlinx.coroutines.delay(10000) // Hide after 10 seconds
                _showPassword.value = false
            }
        }
    }

    fun toggleDeletePasswordVisibility() {
        _showDeletePassword.value = !_showDeletePassword.value
        if (_showDeletePassword.value) {
            viewModelScope.launch {
                kotlinx.coroutines.delay(10000) // Hide after 10 seconds
                _showDeletePassword.value = false
            }
        }
    }

    fun onWebAppUrlChange(newUrl: String) {
        webAppUrl = newUrl
    }

    fun onUsernameChange(newUser: String) {
        username = newUser
    }

    fun onPasswordChange(newPass: String) {
        password = newPass
    }

    fun onDeletePasswordChange(newDeletePass: String) {
        deletePassword = newDeletePass
    }

    private fun loadCredentials() {
        viewModelScope.launch {
            webAppUrl = sharedPrefs.getString("webapp_url", "") ?: ""
            username = sharedPrefs.getString("username", "") ?: ""
            password = sharedPrefs.getString("password", "") ?: ""
            deletePassword = sharedPrefs.getString("delete_password", "") ?: ""
            _isWebAppEnabled.value = sharedPrefs.getBoolean("webapp_enabled", false)
        }
    }

    fun setEnabled(enabled: Boolean) {
        _isWebAppEnabled.value = enabled
        sharedPrefs.edit().putBoolean("webapp_enabled", enabled).apply()
    }

    fun saveSettings() {
        viewModelScope.launch {
            sharedPrefs.edit().apply {
                putString("webapp_url", webAppUrl)
                putString("username", username)
                putString("password", password)
                putString("delete_password", deletePassword)
                apply()
            }
            Toast.makeText(getApplication(), R.string.settings_saved, Toast.LENGTH_SHORT).show()
        }
    }

    fun testConnection() {
        viewModelScope.launch {
            val (isSuccess, message) = repository.testWebAppConnection(webAppUrl, username, password)
            if (isSuccess) {
                Toast.makeText(getApplication(), R.string.connection_successful, Toast.LENGTH_SHORT).show()
            } else {
                Toast.makeText(getApplication(), getApplication<Application>().getString(R.string.connection_failed, message), Toast.LENGTH_LONG).show()
            }
        }
    }

    fun resetWebAppIntegration() {
        viewModelScope.launch {
            webAppUrl = ""
            username = ""
            password = ""
            deletePassword = ""
            _isWebAppEnabled.value = false

            sharedPrefs.edit().apply {
                remove("webapp_url")
                remove("username")
                remove("password")
                remove("delete_password")
                remove("webapp_enabled")
                apply()
            }
            Toast.makeText(getApplication(), getApplication<Application>().getString(R.string.web_app_integration_reset_successful), Toast.LENGTH_SHORT).show()
        }
    }
}
