package de.lxtools.noteshop.ui.webapp

import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Visibility
import androidx.compose.material.icons.filled.VisibilityOff
import androidx.compose.material3.AlertDialog
import androidx.compose.material3.Button
import androidx.compose.material3.ButtonDefaults
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.ExperimentalComposeUiApi
import androidx.compose.ui.Modifier
import androidx.compose.ui.autofill.AutofillNode
import androidx.compose.ui.autofill.AutofillType
import androidx.compose.ui.composed
import androidx.compose.ui.focus.onFocusChanged
import androidx.compose.ui.layout.boundsInWindow
import androidx.compose.ui.layout.onGloballyPositioned
import androidx.compose.ui.platform.LocalAutofill
import androidx.compose.ui.platform.LocalAutofillTree
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.input.PasswordVisualTransformation
import androidx.compose.ui.text.input.VisualTransformation
import androidx.compose.ui.unit.dp
import androidx.compose.ui.platform.LocalUriHandler
import androidx.compose.foundation.text.ClickableText
import androidx.compose.ui.text.SpanStyle
import androidx.compose.ui.text.buildAnnotatedString
import androidx.compose.ui.text.withStyle
import de.lxtools.noteshop.R

@OptIn(ExperimentalComposeUiApi::class)
fun Modifier.autofill(
    autofillTypes: List<AutofillType>,
    onFill: (String) -> Unit,
) = composed {
    val autofillNode = remember {
        AutofillNode(
            autofillTypes = autofillTypes,
            onFill = onFill
        )
    }
    LocalAutofillTree.current += autofillNode
    val autofill = LocalAutofill.current

    this
        .onGloballyPositioned {
            autofillNode.boundingBox = it.boundsInWindow()
        }
        .onFocusChanged { focusState ->
            if (focusState.isFocused) {
                autofill?.requestAutofillForNode(autofillNode)
            } else {
                autofill?.cancelAutofillForNode(autofillNode)
            }
        }
}

@OptIn(ExperimentalMaterial3Api::class, ExperimentalComposeUiApi::class)
@Composable
fun WebAppIntegrationScreen(
    webAppIntegrationViewModel: WebAppIntegrationViewModel,
    padding: PaddingValues,
    onAuthenticateAndTogglePasswordVisibility: (Boolean) -> Unit // New parameter
) {
    val isWebAppEnabled by webAppIntegrationViewModel.isWebAppEnabled.collectAsState()
    var showResetConfirmationDialog by remember { mutableStateOf(false) }

    Column(
        modifier = Modifier
            .padding(padding)
            .padding(horizontal = 16.dp)
            .fillMaxSize()
            .verticalScroll(rememberScrollState()),
        horizontalAlignment = Alignment.CenterHorizontally,
        verticalArrangement = Arrangement.spacedBy(8.dp)
    ) {
        OutlinedTextField(
            value = webAppIntegrationViewModel.webAppUrl,
            onValueChange = webAppIntegrationViewModel::onWebAppUrlChange,
            label = { Text(stringResource(R.string.webapp_url_label)) },
            modifier = Modifier
                .fillMaxWidth()
        )
        OutlinedTextField(
            value = webAppIntegrationViewModel.username,
            onValueChange = webAppIntegrationViewModel::onUsernameChange,
            label = { Text(stringResource(R.string.username_label)) },
            modifier = Modifier
                .fillMaxWidth()
                .autofill(
                    autofillTypes = listOf(AutofillType.Username),
                    onFill = webAppIntegrationViewModel::onUsernameChange
                )
        )
        OutlinedTextField(
            value = webAppIntegrationViewModel.password,
            onValueChange = webAppIntegrationViewModel::onPasswordChange,
            label = { Text(stringResource(R.string.password_label)) },
            visualTransformation = if (webAppIntegrationViewModel.showPassword) VisualTransformation.None else PasswordVisualTransformation(),
            trailingIcon = {
                val image = if (webAppIntegrationViewModel.showPassword)
                    Icons.Filled.Visibility
                else Icons.Filled.VisibilityOff
                val description = if (webAppIntegrationViewModel.showPassword) "Hide password" else "Show password"

                IconButton(onClick = { onAuthenticateAndTogglePasswordVisibility(true) }) {
                    Icon(imageVector = image, contentDescription = description)
                }
            },
            modifier = Modifier
                .fillMaxWidth()
                .autofill(
                    autofillTypes = listOf(AutofillType.Password),
                    onFill = webAppIntegrationViewModel::onPasswordChange
                )
        )
        OutlinedTextField(
            value = webAppIntegrationViewModel.deletePassword,
            onValueChange = webAppIntegrationViewModel::onDeletePasswordChange,
            label = { Text(stringResource(R.string.delete_password_label) + " (optional)") },
            visualTransformation = if (webAppIntegrationViewModel.showDeletePassword) VisualTransformation.None else PasswordVisualTransformation(),
            trailingIcon = {
                val image = if (webAppIntegrationViewModel.showDeletePassword)
                    Icons.Filled.Visibility
                else Icons.Filled.VisibilityOff
                val description = if (webAppIntegrationViewModel.showDeletePassword) "Hide password" else "Show password"

                IconButton(onClick = { onAuthenticateAndTogglePasswordVisibility(false) }) {
                    Icon(imageVector = image, contentDescription = description)
                }
            },
            modifier = Modifier.fillMaxWidth()
        )

        Spacer(modifier = Modifier.height(16.dp))

        Button(
            onClick = webAppIntegrationViewModel::saveSettings,
            modifier = Modifier.fillMaxWidth()
        ) {
            Text(stringResource(R.string.save_settings_button))
        }

        Spacer(modifier = Modifier.height(8.dp))

        Button(
            onClick = webAppIntegrationViewModel::testConnection,
            modifier = Modifier.fillMaxWidth()
        ) {
            Text(stringResource(R.string.test_connection_button))
        }

        Spacer(modifier = Modifier.height(8.dp)) // Add some space between buttons

        Button(
            onClick = { showResetConfirmationDialog = true },
            modifier = Modifier.fillMaxWidth(),
            colors = ButtonDefaults.buttonColors(containerColor = MaterialTheme.colorScheme.error)
        ) {
            Text(stringResource(R.string.reset_web_app_integration))
        }

        Spacer(modifier = Modifier.height(16.dp))

        val uriHandler = LocalUriHandler.current
        val fullText = stringResource(R.string.webapp_docker_info)
        val url = "https://hub.docker.com/r/lxtools/noteshop-webapp"

        val annotatedString = buildAnnotatedString {
            val startIndex = fullText.indexOf(url)
            val endIndex = startIndex + url.length

            if (startIndex > 0) {
                append(fullText.substring(0, startIndex))
            }
            pushStringAnnotation(tag = "URL", annotation = url)
            withStyle(style = SpanStyle(color = MaterialTheme.colorScheme.primary)) {
                append(fullText.substring(startIndex, endIndex))
            }
            pop()
            if (endIndex < fullText.length) {
                append(fullText.substring(endIndex, fullText.length))
            }
        }

        ClickableText(
            text = annotatedString,
            onClick = { offset ->
                annotatedString.getStringAnnotations(tag = "URL", start = offset, end = offset)
                    .firstOrNull()?.let { stringAnnotation ->
                        uriHandler.openUri(stringAnnotation.item)
                    }
            },
            style = MaterialTheme.typography.bodySmall.copy(color = MaterialTheme.colorScheme.onSurface),
            modifier = Modifier.padding(horizontal = 8.dp)
        )

        if (showResetConfirmationDialog) {
            AlertDialog(
                onDismissRequest = { showResetConfirmationDialog = false },
                title = { Text(stringResource(R.string.reset_web_app_integration)) },
                text = { Text(stringResource(R.string.reset_web_app_integration_confirmation)) },
                confirmButton = {
                    Button(
                        onClick = {
                            webAppIntegrationViewModel.resetWebAppIntegration()
                            showResetConfirmationDialog = false
                        }
                    ) {
                        Text(stringResource(R.string.delete)) // Using existing "Delete" string for confirmation
                    }
                },
                dismissButton = {
                    TextButton(onClick = { showResetConfirmationDialog = false }) {
                        Text(stringResource(R.string.cancel))
                    }
                }
            )
        }
    }
}