package de.lxtools.noteshop.ui.theme

import android.os.Build
import androidx.compose.foundation.isSystemInDarkTheme
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.darkColorScheme
import androidx.compose.material3.dynamicDarkColorScheme
import androidx.compose.material3.dynamicLightColorScheme
import androidx.compose.material3.lightColorScheme
import androidx.compose.runtime.Composable
import androidx.compose.ui.platform.LocalContext
import de.lxtools.noteshop.R
import androidx.compose.ui.graphics.Color
import androidx.compose.material3.ColorScheme

enum class ColorTheme(val titleRes: Int) {
    STANDARD(R.string.color_theme_standard),
    GREEN(R.string.color_theme_green),
    VIOLET(R.string.color_theme_violet),
    YELLOW(R.string.color_theme_yellow),
    ORANGE(R.string.color_theme_orange),
    RED(R.string.color_theme_red)
}

// Color definitions
val GreenPrimary = Color(0xFF4CAF50)
val VioletPrimary = Color(0xFF9C27B0)
val YellowPrimary = Color(0xFFFBC02D)
val OrangePrimary = Color(0xFFF57C00)
val RedPrimary = Color(0xFFD32F2F)

// The original, default colors
private val DarkColorScheme = darkColorScheme(
    primary = BluePrimary,
    onPrimary = Color.White,
    secondary = PurpleGrey80,
    tertiary = Pink80
)

private val LightColorScheme = lightColorScheme(
    primary = BluePrimary,
    onPrimary = Color.White,
    secondary = PurpleGrey40,
    tertiary = Pink40
)

fun getAppColorScheme(colorTheme: ColorTheme, isDark: Boolean) : ColorScheme {
    if (colorTheme == ColorTheme.STANDARD) {
        return if (isDark) DarkColorScheme else LightColorScheme
    }

    val baseScheme = if (isDark) DarkColorScheme else LightColorScheme

    val (primaryColor, onPrimaryColor) = when (colorTheme) {
        ColorTheme.GREEN -> GreenPrimary to Color.White
        ColorTheme.VIOLET -> VioletPrimary to Color.White
        ColorTheme.YELLOW -> YellowPrimary to Color.Black
        ColorTheme.ORANGE -> OrangePrimary to Color.White
        ColorTheme.RED -> RedPrimary to Color.White
        else -> BluePrimary to Color.White // Default for STANDARD or any future additions
    }

    return baseScheme.copy(
        primary = primaryColor,
        onPrimary = onPrimaryColor
    )
}

@Composable
fun NoteshopTheme(
    darkTheme: Boolean = isSystemInDarkTheme(),
    // Dynamic color is available on Android 12+
    dynamicColor: Boolean = false,
    colorTheme: ColorTheme = ColorTheme.STANDARD,
    content: @Composable () -> Unit
) {
    val colorScheme = when {
        dynamicColor && Build.VERSION.SDK_INT >= Build.VERSION_CODES.S -> {
            val context = LocalContext.current
            if (darkTheme) dynamicDarkColorScheme(context) else dynamicLightColorScheme(context)
        }
        else -> getAppColorScheme(colorTheme, darkTheme)
    }

    MaterialTheme(
        colorScheme = colorScheme,
        typography = Typography,
        content = content
    )
}