package de.lxtools.noteshop.ui.shopping

import android.app.Application
import android.content.Context
import android.util.Log
import android.widget.Toast
import androidx.lifecycle.AndroidViewModel
import androidx.lifecycle.viewModelScope
import de.lxtools.noteshop.data.NoteshopRepository
import de.lxtools.noteshop.data.ShoppingList
import de.lxtools.noteshop.data.ShoppingListItem
import de.lxtools.noteshop.data.ShoppingListWithItems
import de.lxtools.noteshop.security.hashString
import kotlinx.coroutines.flow.Flow
import de.lxtools.noteshop.api.WebAppShoppingList
import de.lxtools.noteshop.api.WebAppClient
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.flow.firstOrNull
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.launch
import kotlinx.serialization.json.Json

class ShoppingListsViewModel(private val noteshopRepository: NoteshopRepository, application: Application) : AndroidViewModel(application) {

    private val webAppClient = WebAppClient() // Instantiate WebAppClient

    private val _searchQuery = MutableStateFlow("") // New search query state
    val searchQuery: StateFlow<String> = _searchQuery.asStateFlow()

    private val _showCompletedItems = MutableStateFlow(false) // New state for showing completed items
    val showCompletedItems: StateFlow<Boolean> = _showCompletedItems.asStateFlow()

    // State for the detail screen
    private val _newItemName = MutableStateFlow("")
    val newItemName: StateFlow<String> = _newItemName.asStateFlow()

    private val _selectedItem = MutableStateFlow<ShoppingListItem?>(null)
    val selectedItem: StateFlow<ShoppingListItem?> = _selectedItem.asStateFlow()

    private val _showRenameDialog = MutableStateFlow(false)
    val showRenameDialog: StateFlow<Boolean> = _showRenameDialog.asStateFlow()

    private val _showQuantityDialog = MutableStateFlow(false)
    val showQuantityDialog: StateFlow<Boolean> = _showQuantityDialog.asStateFlow()

    private val _isReorderMode = MutableStateFlow(false)
    val isReorderMode: StateFlow<Boolean> = _isReorderMode.asStateFlow()

    private val _isDetailSearchActive = MutableStateFlow(false)
    val isDetailSearchActive: StateFlow<Boolean> = _isDetailSearchActive.asStateFlow()

    private val _isWebAppEnabled = MutableStateFlow(false)
    val isWebAppEnabled: StateFlow<Boolean> = _isWebAppEnabled

    // New states for WebApp list selection dialog
    private val _showWebAppListSelectionDialog = MutableStateFlow(false)
    val showWebAppListSelectionDialog: StateFlow<Boolean> = _showWebAppListSelectionDialog.asStateFlow()

    private val _webAppShoppingLists = MutableStateFlow<List<WebAppShoppingList>>(emptyList())
    val webAppShoppingLists: StateFlow<List<WebAppShoppingList>> = _webAppShoppingLists.asStateFlow()

    private val _deleteCompletedOnImport = MutableStateFlow(false)
    val deleteCompletedOnImport: StateFlow<Boolean> = _deleteCompletedOnImport.asStateFlow()

    private val _currentLocalListIdForWebAppImport = MutableStateFlow(0)

    init {
        viewModelScope.launch {
            val webAppPrefs = getApplication<Application>().getSharedPreferences("webapp_prefs", Context.MODE_PRIVATE)
            _isWebAppEnabled.value = webAppPrefs.getBoolean("webapp_enabled", false)
        }
    }

    fun onDeleteCompletedOnImportChange(delete: Boolean) {
        _deleteCompletedOnImport.value = delete
    }


    fun toggleWebAppEnabled() {
        viewModelScope.launch {
            val webAppPrefs = getApplication<Application>().getSharedPreferences("webapp_prefs", Context.MODE_PRIVATE)
            val newEnabledState = !_isWebAppEnabled.value
            webAppPrefs.edit().putBoolean("webapp_enabled", newEnabledState).apply()
            _isWebAppEnabled.value = newEnabledState
        }
    }

    fun onNewItemNameChange(name: String) {
        _newItemName.value = name
    }

    fun onSelectItem(item: ShoppingListItem?) {
        _selectedItem.value = item
    }

    fun onShowRenameDialog(show: Boolean) {
        _showRenameDialog.value = show
    }

    fun onShowQuantityDialog(show: Boolean) {
        _showQuantityDialog.value = show
    }

    fun enableReorderMode() {
        _isReorderMode.value = true
    }

    fun disableReorderMode() {
        _isReorderMode.value = false
    }

    fun toggleDetailSearch() {
        _isDetailSearchActive.value = !_isDetailSearchActive.value
    }

    val uiState: StateFlow<ShoppingListsUiState> = combine(
        noteshopRepository.getAllShoppingListsWithItemsStream(),
        searchQuery
    ) { allLists, query ->
        val filteredLists = if (query.isBlank()) {
            allLists
        } else {
            allLists.filter { it.shoppingList.name.contains(query, ignoreCase = true) }
        }
        ShoppingListsUiState(filteredLists)
    }
        .stateIn(
            scope = viewModelScope,
            started = SharingStarted.WhileSubscribed(TIMEOUT_MILLIS),
            initialValue = ShoppingListsUiState()
        )

    private val _listDetails = MutableStateFlow(ShoppingListDetails())
    val listDetails: StateFlow<ShoppingListDetails> = _listDetails.asStateFlow()

    fun updateListDetails(list: ShoppingList) {
        _listDetails.value = ShoppingListDetails(
            id = list.id,
            name = list.name,
            displayOrder = list.displayOrder,
            protectionHash = list.protectionHash,
            protectionType = list.protectionType,
            lockMethod = list.lockMethod
        )
    }

    fun updateListDetails(listDetails: ShoppingListDetails) {
        _listDetails.value = listDetails
    }

    suspend fun saveList() {
        if (listDetails.value.isValid()) {
            var currentList = listDetails.value.toShoppingList()

            if (currentList.id == 0) {
                val listsCount = noteshopRepository.getListsCount()
                currentList = currentList.copy(displayOrder = listsCount)
                noteshopRepository.insertShoppingList(currentList)
            } else {
                val existingList = noteshopRepository.getShoppingListWithItemsStream(currentList.id).firstOrNull()?.shoppingList
                val listToUpdate = existingList?.copy(
                    name = currentList.name,
                    displayOrder = currentList.displayOrder,
                    protectionHash = currentList.protectionHash,
                    protectionType = currentList.protectionType,
                    lockMethod = currentList.lockMethod
                ) ?: currentList
                noteshopRepository.updateShoppingList(listToUpdate)
            }
        }
    }

    fun setProtection(listId: Int, password: String) {
        viewModelScope.launch {
            val listToUpdate = noteshopRepository.getShoppingListWithItemsStream(listId).firstOrNull()?.shoppingList

            listToUpdate?.let { existingList ->
                if (password.isNotBlank()) {
                    val hashedPassword = hashString(password)
                    val updatedList = existingList.copy(
                        protectionHash = hashedPassword,
                        protectionType = 1, // 1 for password protection
                        lockMethod = 1
                    )
                    noteshopRepository.updateShoppingList(updatedList)
                    updateListDetails(updatedList) // Update the UI state
                } else {
                    // Password is blank, so we remove protection
                    val updatedList = existingList.copy(
                        protectionHash = "",
                        protectionType = 0,
                        lockMethod = 0
                    )
                    noteshopRepository.updateShoppingList(updatedList)
                    updateListDetails(updatedList) // Update the UI state
                }
            } ?: run {
                Log.e("ShoppingListsViewModel", "Shopping list with ID $listId not found for protection update or is unsaved.")
            }
        }
    }

    fun setProtectionBiometric(listId: Int) {
        viewModelScope.launch {
            val list = noteshopRepository.getShoppingListWithItemsStream(listId).firstOrNull()?.shoppingList
            list?.let {
                val updatedList = it.copy(
                    protectionHash = "", // Clear password hash
                    protectionType = 2, // 2 for biometric protection
                    lockMethod = 2
                )
                noteshopRepository.updateShoppingList(updatedList)
            }
        }
    }

    suspend fun deleteList(list: ShoppingList) {
        noteshopRepository.deleteShoppingList(list)
    }

    fun resetListDetails() {
        _listDetails.value = ShoppingListDetails()
    }

    suspend fun saveShoppingListItem(item: ShoppingListItem) {
        if (item.name.isNotBlank()) {
            val parentList = noteshopRepository.getShoppingListWithItemsStream(item.listId).firstOrNull()?.shoppingList
            if (parentList?.protectionHash?.isNotEmpty() == true) {
                Log.w("ShoppingListsViewModel", "Attempted to save item to a locked list.")
                return
            }

            if (item.id == 0) {
                val itemsCount = noteshopRepository.getItemsCount(item.listId)
                val newItem = item.copy(displayOrder = itemsCount)
                noteshopRepository.insertShoppingListItem(newItem)
            } else {
                noteshopRepository.updateShoppingListItem(item)
            }
        }
    }

    suspend fun saveShoppingListItem(listId: Int, itemName: String, shouldSplit: Boolean = true) {
        if (itemName.isNotBlank()) {
            val listWithItems = noteshopRepository.getShoppingListWithItemsStream(listId).firstOrNull()
            listWithItems?.let {
                if (it.shoppingList.protectionHash.isNotEmpty()) {
                    Log.w("ShoppingListsViewModel", "Attempted to save items to a locked list.")
                    return
                }

                val existingItemNames = it.items.map { item ->
                    item.name.trim().lowercase()
                }

                val itemsToAdd = if (shouldSplit && itemName.contains(',')) {
                    itemName.split(',').map { name -> name.trim() }.filter { name -> name.isNotBlank() && !existingItemNames.contains(name.lowercase()) }
                } else {
                    if (!existingItemNames.contains(itemName.trim().lowercase())) {
                        listOf(itemName.trim())
                    } else {
                        emptyList()
                    }
                }

                if (itemsToAdd.isNotEmpty()) {
                    val initialItemsCount = noteshopRepository.getItemsCount(listId)
                    val newItems = itemsToAdd.mapIndexed { index, name ->
                        ShoppingListItem(
                            name = name,
                            listId = listId,
                            displayOrder = initialItemsCount + index
                        )
                    }
                    noteshopRepository.insertShoppingListItems(newItems)
                }
            }
        }
    }

    fun getShoppingListWithItemsStream(listId: Int): Flow<ShoppingListWithItems?> {
        return noteshopRepository.getShoppingListWithItemsStream(listId).map { listWithItems ->
            listWithItems?.copy(items = listWithItems.items.sortedBy { it.displayOrder })
        }
    }

    fun updateSearchQuery(query: String) {
        _searchQuery.value = query
    }

    fun toggleShowCompletedItems() {
        _showCompletedItems.value = !_showCompletedItems.value
    }

    suspend fun deleteCompletedItems(listId: Int) {
        val listWithItems = noteshopRepository.getShoppingListWithItemsStream(listId).firstOrNull()
        listWithItems?.items?.filter { it.isChecked }?.let { itemsToDelete ->
            if (itemsToDelete.isNotEmpty()) {
                noteshopRepository.deleteShoppingListItems(itemsToDelete)
                deleteMarkedItemsInWebApp()
            }
        }
    }

    fun moveList(from: Int, to: Int) {
        viewModelScope.launch {
            val list = uiState.value.shoppingLists.toMutableList()
            list.add(to, list.removeAt(from))
            val updatedLists = list.mapIndexed { index, item ->
                item.shoppingList.copy(displayOrder = index)
            }
            noteshopRepository.updateShoppingLists(updatedLists)
        }
    }

    fun moveItem(items: List<ShoppingListItem>) {
        viewModelScope.launch {
            val updatedItems = items.mapIndexed { index, item ->
                item.copy(displayOrder = index)
            }
            noteshopRepository.updateShoppingListItems(updatedItems)
        }
    }

    fun createStandardList(standardListName: String) {
        viewModelScope.launch {
            var newListName = standardListName
            var counter = 1
            while (noteshopRepository.getShoppingListByName(newListName).firstOrNull() != null) {
                newListName = "$standardListName ($counter)"
                counter++
            }

            val listsCount = noteshopRepository.getListsCount()
            val newList = ShoppingList(name = newListName, displayOrder = listsCount)
            val newListId = noteshopRepository.insertShoppingList(newList)

            val standardItems = noteshopRepository.getStandardListItems()
            val newItems = standardItems.mapIndexed { index, itemName ->
                ShoppingListItem(
                    name = itemName,
                    listId = newListId.toInt(),
                    displayOrder = index
                )
            }
            if (newItems.isNotEmpty()) {
                noteshopRepository.insertShoppingListItems(newItems)
            }
        }
    }

    fun formatShoppingListForExport(listWithItems: ShoppingListWithItems): String {
        val builder = StringBuilder()
        builder.appendLine(listWithItems.shoppingList.name)
        listWithItems.items.sortedBy { it.displayOrder }.forEach { item ->
            val checkedMarker = if (item.isChecked) "[x]" else "[ ]"
            val quantity = if (item.quantity != null) " (${item.quantity})" else ""
            builder.appendLine("- $checkedMarker ${item.name}$quantity")
        }
        return builder.toString()
    }

    fun exportShoppingListsToJson(shoppingLists: List<ShoppingListWithItems>): String {
        return Json.encodeToString(shoppingLists)
    }

    suspend fun importListFromTxt(listName: String, content: String) {
        val listsCount = noteshopRepository.getListsCount()
        val newList = ShoppingList(name = listName, displayOrder = listsCount)
        val newListId = noteshopRepository.insertShoppingList(newList)

        val itemsToInsert = content.lines()
            .map { it.trim() }
            .filter { it.isNotBlank() }
            .mapIndexed { index, itemName ->
                ShoppingListItem(
                    name = itemName,
                    listId = newListId.toInt(),
                    displayOrder = index
                )
            }

        if (itemsToInsert.isNotEmpty()) {
            noteshopRepository.insertShoppingListItems(itemsToInsert)
        }
    }

    suspend fun toggleAllItemsCompletion(listId: Int, markAsComplete: Boolean) {
        val listWithItems = noteshopRepository.getShoppingListWithItemsStream(listId).firstOrNull()
        listWithItems?.let {
            val itemsToUpdate = it.items.map { item ->
                item.copy(isChecked = markAsComplete)
            }
            if (itemsToUpdate.isNotEmpty()) {
                noteshopRepository.updateShoppingListItems(itemsToUpdate)
            }
        }
    }

    suspend fun importShoppingListsFromJson(jsonString: String) {
        try {
            val shoppingLists = Json.decodeFromString<List<ShoppingListWithItems>>(jsonString)
            shoppingLists.forEach { listWithItems ->
                val newList = listWithItems.shoppingList.copy(id = 0)
                val newId = noteshopRepository.insertShoppingList(newList)
                listWithItems.items.forEach { item ->
                    val newItem = item.copy(id = 0, listId = newId.toInt())
                    noteshopRepository.insertShoppingListItem(newItem)
                }
            }
        } catch (e: Exception) {
            Log.e("ShoppingListsViewModel", "Error parsing shopping lists JSON", e)
        }
    }

    fun dismissWebAppListSelectionDialog() {
        _showWebAppListSelectionDialog.value = false
        _webAppShoppingLists.value = emptyList() // Clear lists when dialog is dismissed
        _currentLocalListIdForWebAppImport.value = 0
    }

    fun refreshWebAppShoppingLists(localListId: Int) {
        viewModelScope.launch {
            val webAppPrefs = getApplication<Application>().getSharedPreferences("webapp_prefs", Context.MODE_PRIVATE)
            val url = webAppPrefs.getString("webapp_url", null)
            val username = webAppPrefs.getString("username", null)
            val password = webAppPrefs.getString("password", null)

            if (url.isNullOrBlank() || username.isNullOrBlank() || password.isNullOrBlank()) {
                Toast.makeText(getApplication(), de.lxtools.noteshop.R.string.webapp_credentials_not_set_up, Toast.LENGTH_SHORT).show()
                return@launch
            }

            try {
                val fetchedLists = webAppClient.fetchWebAppShoppingLists(url, username, password)
                _webAppShoppingLists.value = fetchedLists
                _currentLocalListIdForWebAppImport.value = localListId
                _showWebAppListSelectionDialog.value = true
            } catch (e: Exception) {
                Toast.makeText(getApplication(), getApplication<Application>().getString(de.lxtools.noteshop.R.string.import_failed, e.message), Toast.LENGTH_LONG).show()
            }
        }
    }

    fun importItemsFromWebApp(localListId: Int, webAppListId: Int, deleteCompleted: Boolean) {
        viewModelScope.launch {
            val webAppPrefs = getApplication<Application>().getSharedPreferences("webapp_prefs", Context.MODE_PRIVATE)
            val url = webAppPrefs.getString("webapp_url", null)
            val username = webAppPrefs.getString("username", null)
            val password = webAppPrefs.getString("password", null)

            if (url.isNullOrBlank() || username.isNullOrBlank() || password.isNullOrBlank()) {
                Toast.makeText(getApplication(), de.lxtools.noteshop.R.string.webapp_credentials_not_set_up, Toast.LENGTH_SHORT).show()
                return@launch
            }

            try {
                // Save the selected webAppListId to shared preferences for markItemInWebApp and deleteMarkedItemsInWebApp
                webAppPrefs.edit().putInt("selected_webapp_list_id", webAppListId).apply()

                if (deleteCompleted) {
                    deleteMarkedItemsInWebApp()
                }

                val importedCount = noteshopRepository.importItemsFromWebApp(localListId, url, username, password, webAppListId)
                if (importedCount > 0) {
                    val message = getApplication<Application>().resources.getQuantityString(de.lxtools.noteshop.R.plurals.import_successful, importedCount, importedCount)
                    Toast.makeText(getApplication(), message, Toast.LENGTH_SHORT).show()
                } else {
                    Toast.makeText(getApplication(), de.lxtools.noteshop.R.string.no_new_items_to_import, Toast.LENGTH_SHORT).show()
                }
            } catch (e: Exception) {
                Toast.makeText(getApplication(), getApplication<Application>().getString(de.lxtools.noteshop.R.string.import_failed, e.message), Toast.LENGTH_LONG).show()
            } finally {
                dismissWebAppListSelectionDialog() // Dismiss dialog after import attempt
            }
        }
    }

    fun markItemInWebApp(itemName: String) {
        if (!isWebAppEnabled.value) return
        viewModelScope.launch {
            val webAppPrefs = getApplication<Application>().getSharedPreferences("webapp_prefs", Context.MODE_PRIVATE)
            val url = webAppPrefs.getString("webapp_url", null)
            val username = webAppPrefs.getString("username", null)
            val password = webAppPrefs.getString("password", null)
            val webAppListId = webAppPrefs.getInt("selected_webapp_list_id", -1)

            if (url.isNullOrBlank() || username.isNullOrBlank() || password.isNullOrBlank() || webAppListId == -1) {
                return@launch // Silently fail if not configured or no list selected
            }

            noteshopRepository.markItemInWebApp(url, username, password, webAppListId, itemName)
        }
    }

    fun deleteMarkedItemsInWebApp() {
        if (!isWebAppEnabled.value) return
        viewModelScope.launch {
            val webAppPrefs = getApplication<Application>().getSharedPreferences("webapp_prefs", Context.MODE_PRIVATE)
            val url = webAppPrefs.getString("webapp_url", null)
            val deletePass = webAppPrefs.getString("delete_password", null)
            val username = webAppPrefs.getString("username", null)
            val password = webAppPrefs.getString("password", null)
            val webAppListId = webAppPrefs.getInt("selected_webapp_list_id", -1)

            if (url.isNullOrBlank() || username.isNullOrBlank() || password.isNullOrBlank() || webAppListId == -1) {
                return@launch // Silently fail if not configured or no list selected
            }

            noteshopRepository.deleteMarkedItemsInWebApp(url, username, password, webAppListId, deletePass)
        }
    }

    companion object {
        private const val TIMEOUT_MILLIS = 5_000L
    }
}

data class ShoppingListsUiState(
    val shoppingLists: List<ShoppingListWithItems> = listOf()
)

data class ShoppingListDetails(
    val id: Int = 0,
    val name: String = "",
    val displayOrder: Int = 0,
    val protectionHash: String = "",
    val protectionType: Int = 0,
    val lockMethod: Int = 0
) {
    fun toShoppingList(): ShoppingList = ShoppingList(
        id = id,
        name = name,
        displayOrder = displayOrder,
        protectionHash = protectionHash,
        protectionType = protectionType,
        lockMethod = lockMethod
    )

    fun isValid(): Boolean {
        return name.isNotBlank()
    }
}
