package de.lxtools.noteshop.ui.shopping

import android.annotation.SuppressLint
import androidx.compose.foundation.gestures.detectTapGestures
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.foundation.lazy.rememberLazyListState
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Delete
import androidx.compose.material.icons.filled.Edit
import androidx.compose.material.icons.filled.Lock
import androidx.compose.material.icons.rounded.DragHandle
import androidx.compose.material3.AlertDialog
import androidx.compose.material3.Button
import androidx.compose.material3.Card
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.input.pointer.pointerInput
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import de.lxtools.noteshop.R
import de.lxtools.noteshop.data.ShoppingListWithItems
import kotlinx.coroutines.launch
import sh.calvin.reorderable.ReorderableCollectionItemScope
import sh.calvin.reorderable.ReorderableItem
import sh.calvin.reorderable.rememberReorderableLazyListState

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun ShoppingListsScreen(
    shoppingListsViewModel: ShoppingListsViewModel,
    @SuppressLint("ModifierParameter") modifier: Modifier = Modifier,
    onListClick: (Int, Int) -> Unit,
    onEditList: (ShoppingListWithItems) -> Unit
) {
    val uiState by shoppingListsViewModel.uiState.collectAsState()
    val isReorderMode by shoppingListsViewModel.isReorderMode.collectAsState()
    val coroutineScope = rememberCoroutineScope()

    Column(
        modifier = modifier.fillMaxSize()
    ) {
        if (uiState.shoppingLists.isEmpty()) {
            Text(text = stringResource(R.string.no_shopping_lists_yet))
        } else {
            val lazyListState = rememberLazyListState()
            val reorderableLazyListState = rememberReorderableLazyListState(lazyListState) { from, to ->
                shoppingListsViewModel.moveList(from.index, to.index)
            }
            LazyColumn(
                state = lazyListState,
            ) {
                items(uiState.shoppingLists, { it.shoppingList.id }) { listWithItems ->
                    ReorderableItem(reorderableLazyListState, key = listWithItems.shoppingList.id) { _ ->

                        ShoppingListCard(
                            listWithItems = listWithItems,
                            onEditList = onEditList,
                            onDeleteList = {
                                coroutineScope.launch {
                                    shoppingListsViewModel.deleteList(it.shoppingList)
                                }
                            },
                            modifier = Modifier.pointerInput(isReorderMode) {
                                if (!isReorderMode) {
                                    detectTapGestures(
                                        onLongPress = { shoppingListsViewModel.enableReorderMode() },
                                        onTap = { onListClick(listWithItems.shoppingList.id, listWithItems.shoppingList.protectionType) }
                                    )
                                }
                            },
                            scope = this,
                            isReorderMode = isReorderMode
                        )
                    }
                }
            }
        }
    }
}



@Composable
fun ShoppingListCard(
    listWithItems: ShoppingListWithItems,
    onEditList: (ShoppingListWithItems) -> Unit,
    onDeleteList: (ShoppingListWithItems) -> Unit,
    modifier: Modifier = Modifier,
    scope: ReorderableCollectionItemScope,
    isReorderMode: Boolean
) {
    Card(
        modifier = modifier
            .fillMaxWidth()
            .padding(vertical = 4.dp, horizontal = 16.dp)
    ) {
        Column(modifier = Modifier.padding(16.dp)) {
            Row(
                modifier = Modifier.fillMaxWidth(),
                horizontalArrangement = Arrangement.SpaceBetween,
                verticalAlignment = Alignment.CenterVertically
            ) {
                if (isReorderMode) {
                    IconButton(modifier = with(scope) { Modifier.draggableHandle() }, onClick = {}) {
                        Icon(Icons.Rounded.DragHandle, contentDescription = stringResource(R.string.reorder))
                    }
                }
                if (listWithItems.shoppingList.lockMethod != 0) {
                    Icon(Icons.Default.Lock, contentDescription = null, modifier = Modifier.padding(end = 8.dp))
                }
                Text(
                    text = listWithItems.shoppingList.name,
                    style = MaterialTheme.typography.titleMedium,
                    modifier = Modifier.weight(1f)
                )
                IconButton(onClick = { onEditList(listWithItems) }, enabled = !isReorderMode) {
                    Icon(Icons.Default.Edit, contentDescription = stringResource(R.string.edit_list))
                }
                IconButton(onClick = { onDeleteList(listWithItems) }, enabled = !isReorderMode) {
                    Icon(Icons.Default.Delete, contentDescription = stringResource(R.string.delete_list))
                }
                if (isReorderMode) {
                    IconButton(modifier = with(scope) { Modifier.draggableHandle() }, onClick = {}) {
                        Icon(Icons.Rounded.DragHandle, contentDescription = stringResource(R.string.reorder))
                    }
                }
            }
        }
    }
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun ShoppingListInputDialog(
    listDetails: ShoppingListDetails,
    onValueChange: (ShoppingListDetails) -> Unit,
    onSave: () -> Unit,
    onDismiss: () -> Unit,
    onImportFromTxt: () -> Unit,
    isNewList: Boolean,
    modifier: Modifier = Modifier
) {
    AlertDialog(
        onDismissRequest = onDismiss,
        title = { Text(text = if (isNewList) stringResource(R.string.add_shopping_list) else stringResource(R.string.edit_shopping_list)) },
        text = {
            Column {
                OutlinedTextField(
                    value = listDetails.name,
                    onValueChange = { onValueChange(listDetails.copy(name = it)) },
                    label = { Text(stringResource(R.string.list_name)) },
                    modifier = Modifier.fillMaxWidth()
                )
                if (isNewList) {
                    Spacer(modifier = Modifier.height(16.dp))
                    Button(onClick = onImportFromTxt, modifier = Modifier.fillMaxWidth()) {
                        Text(stringResource(R.string.import_from_txt))
                    }
                }
            }
        },
        confirmButton = {
            Button(onClick = onSave, enabled = listDetails.isValid()) {
                Text(stringResource(R.string.save))
            }
        },
        dismissButton = {
            TextButton(onClick = onDismiss) {
                Text(stringResource(R.string.cancel))
            }
        },
        modifier = modifier
    )
}