package de.lxtools.noteshop.ui.shopping
import de.lxtools.noteshop.api.WebAppShoppingList
import androidx.compose.material.icons.filled.Refresh
import androidx.compose.material3.LinearProgressIndicator
import androidx.compose.ui.window.DialogProperties
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.material3.TextButton
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.heightIn
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.foundation.lazy.rememberLazyListState
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Lock
import androidx.compose.material.icons.filled.MoreVert
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material.icons.rounded.DragHandle
import androidx.compose.material3.AlertDialog
import androidx.compose.material3.Button
import androidx.compose.material3.Card
import androidx.compose.material3.DropdownMenu
import androidx.compose.material3.DropdownMenuItem
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.input.pointer.pointerInput
import androidx.compose.foundation.gestures.detectTapGestures
import androidx.compose.ui.platform.LocalContext
import androidx.biometric.BiometricManager
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.SpanStyle
import androidx.compose.ui.text.buildAnnotatedString
import androidx.compose.ui.text.style.TextDecoration
import androidx.compose.ui.unit.dp
import de.lxtools.noteshop.R
import de.lxtools.noteshop.data.ShoppingListItem

import kotlinx.coroutines.launch
import sh.calvin.reorderable.ReorderableItem
import sh.calvin.reorderable.rememberReorderableLazyListState
import androidx.compose.runtime.DisposableEffect
import de.lxtools.noteshop.findActivity
import androidx.compose.ui.text.withStyle


@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun ShoppingListDetailScreen(
    listId: Int?,
    shoppingListsViewModel: ShoppingListsViewModel,
    dynamicStrings: de.lxtools.noteshop.DynamicListStrings,
    modifier: Modifier = Modifier,
    isContentDecrypted: Boolean,
    onUnlockClick: (Int) -> Unit,
    onNavigateBack: () -> Unit
) {
    val coroutineScope = rememberCoroutineScope()




    // Collect state from ViewModel
    val newItemName by shoppingListsViewModel.newItemName.collectAsState()
    val showCompletedItems by shoppingListsViewModel.showCompletedItems.collectAsState()
    val shoppingListWithItems by shoppingListsViewModel.getShoppingListWithItemsStream(listId ?: 0)
        .collectAsState(initial = null)
    val selectedItem by shoppingListsViewModel.selectedItem.collectAsState()
    val showRenameDialog by shoppingListsViewModel.showRenameDialog.collectAsState()
    val showQuantityDialog by shoppingListsViewModel.showQuantityDialog.collectAsState()
    val isReorderMode by shoppingListsViewModel.isReorderMode.collectAsState()
    val isDetailSearchActive by shoppingListsViewModel.isDetailSearchActive.collectAsState()

    // Collect new states for WebApp list selection
    val showWebAppListSelectionDialog by shoppingListsViewModel.showWebAppListSelectionDialog.collectAsState()
    val webAppShoppingLists by shoppingListsViewModel.webAppShoppingLists.collectAsState()


    val context = LocalContext.current
    DisposableEffect(listId) {
        onDispose {
            val activity = context.findActivity()
            if (!activity.isChangingConfigurations) {
                onNavigateBack()
            }
        }
    }

    Column(
        modifier = modifier
            .fillMaxSize()
            .padding(horizontal = 16.dp)
    ) {
        if (shoppingListWithItems == null) {
            Text(
                text = stringResource(R.string.shopping_list_not_found),
                style = MaterialTheme.typography.headlineMedium,
                modifier = Modifier.padding(bottom = 16.dp)
            )
            Spacer(modifier = Modifier.height(16.dp))
        } else {
            val listWithItems = shoppingListWithItems!!
    if (listWithItems.shoppingList.protectionHash.isNotEmpty() && !isContentDecrypted) {
        LockedScreenPlaceholder(onUnlockClick = { onUnlockClick(listWithItems.shoppingList.protectionType) })
    } else {
                val itemsForUi = listWithItems.items

                val checkedCount = itemsForUi.count { it.isChecked }
                val totalCount = itemsForUi.size

                Row(
                    modifier = Modifier
                        .fillMaxWidth(),
                    horizontalArrangement = Arrangement.SpaceBetween,
                    verticalAlignment = Alignment.CenterVertically
                ) {
                    Text(
                        text = buildAnnotatedString { 
                            withStyle(style = SpanStyle(color = MaterialTheme.colorScheme.primary)) {
                                append(checkedCount.toString())
                            }
                            append(" / $totalCount ")
                            append(dynamicStrings.itemChecked)
                        },
                    )

                    TextButton(onClick = {
                        coroutineScope.launch {
                            listWithItems.let { list ->
                                shoppingListsViewModel.toggleAllItemsCompletion(list.shoppingList.id, !showCompletedItems)
                            }
                        }
                    }) {
                        Text(if (showCompletedItems) stringResource(R.string.unmark_all_as_completed) else stringResource(R.string.mark_all_as_completed))
                    }
                }

                OutlinedTextField(
                    value = newItemName,
                    onValueChange = { shoppingListsViewModel.onNewItemNameChange(it) },
                    label = { Text(if (isDetailSearchActive) stringResource(R.string.search) else dynamicStrings.addItem) },
                    modifier = Modifier.fillMaxWidth()
                )

                val currentSearchTerm = if (isDetailSearchActive) {
                    newItemName.substringAfterLast(',').trim()
                } else {
                    ""
                }

                if (newItemName.isNotBlank()) {
                    val suggestionSearchTerm = newItemName.substringAfterLast(',').trim()
                    val suggestions = if (suggestionSearchTerm.isNotBlank()) {
                        itemsForUi.filter { 
                            it.name.contains(suggestionSearchTerm, ignoreCase = true)
                        }
                    } else {
                        emptyList()
                    }

                    if (suggestions.isNotEmpty()) {
                        LazyColumn(modifier = Modifier.heightIn(max = 150.dp)) {
                            items(suggestions) { item ->
                                Row(
                                    modifier = Modifier
                                        .fillMaxWidth()
                                        .padding(vertical = 8.dp)
                                        .clickable {
                                            coroutineScope.launch {
                                                shoppingListsViewModel.saveShoppingListItem(item.copy(isChecked = false))
                                                val currentInput = shoppingListsViewModel.newItemName.value
                                                val lastCommaIndex = currentInput.lastIndexOf(',')
                                                val lastSpaceIndex = currentInput.lastIndexOf(' ')

                                                val separatorIndex = maxOf(lastCommaIndex, lastSpaceIndex)

                                                val prefix = if (separatorIndex != -1) {
                                                    currentInput.take(separatorIndex + 1)
                                                } else {
                                                    ""
                                                }
                                                shoppingListsViewModel.onNewItemNameChange(prefix)
                                            }
                                        },
                                    verticalAlignment = Alignment.CenterVertically
                                ) {
                                    val suggestionText = if (item.isChecked) {
                                        stringResource(R.string.completed)
                                    } else {
                                        stringResource(R.string.already_in_list)
                                    }
                                    Text(
                                        text = "${item.name} ($suggestionText)",
                                        style = MaterialTheme.typography.bodyLarge,
                                        modifier = Modifier.weight(1f)
                                    )
                                }
                            }
                        }
                    }
                }

                Spacer(modifier = Modifier.height(16.dp))

                val itemsToDisplay by remember(itemsForUi, showCompletedItems, isDetailSearchActive, currentSearchTerm) {
                    mutableStateOf(
                        itemsForUi.let { allItems ->
                            val filtered = if (showCompletedItems) {
                                allItems.filter { it.isChecked }
                            } else {
                                allItems.filter { !it.isChecked }
                            }
                            if (isDetailSearchActive && currentSearchTerm.isNotBlank()) {
                                filtered.filter { it.name.contains(currentSearchTerm, ignoreCase = true) }
                            } else {
                                filtered
                            }
                        }
                    )
                }

                if (itemsForUi.isEmpty()) {
                    Text(text = stringResource(R.string.no_items_in_list))
                } else {
                    if (itemsToDisplay.isEmpty() && !showCompletedItems) {
                        Text(text = stringResource(R.string.no_uncompleted_items))
                    } else if (itemsToDisplay.isEmpty()) {
                        Text(text = stringResource(R.string.no_items_in_list))
                    } else {
                        val lazyListState = rememberLazyListState()
                        val reorderableLazyListState = rememberReorderableLazyListState(lazyListState) { from, to ->
                            val reorderedItems = itemsToDisplay.toMutableList().apply {
                                add(to.index, removeAt(from.index))
                            }
                            shoppingListsViewModel.moveItem(reorderedItems)
                        }
                        LazyColumn(
                            state = lazyListState,
                        ) {
                            items(itemsToDisplay, { it.id }) { item ->
                                    ReorderableItem(reorderableLazyListState, key = item.id) { _ ->

                                    Card(
                                        modifier = Modifier
                                            .fillMaxWidth()
                                            .padding(vertical = 4.dp)
                                            .pointerInput(isReorderMode) {
                                                if (!isReorderMode) {
                                                    detectTapGestures(
                                                        onLongPress = { shoppingListsViewModel.enableReorderMode() },
                                                        onTap = {
                                                            coroutineScope.launch {
                                                                val updatedItem = item.copy(isChecked = !item.isChecked)
                                                                shoppingListsViewModel.saveShoppingListItem(updatedItem)
                                                                shoppingListsViewModel.markItemInWebApp(item.name)
                                                            }
                                                        }
                                                    )
                                                }
                                            }
                                    ) {
                                        Row(
                                            modifier = Modifier
                                                .fillMaxWidth()
                                                .padding(horizontal = 16.dp, vertical = 12.dp),
                                            verticalAlignment = Alignment.CenterVertically
                                        ) {
                                            if (isReorderMode) {
                                                IconButton(modifier = Modifier.draggableHandle(), onClick = {}) {
                                                    Icon(Icons.Rounded.DragHandle, contentDescription = stringResource(R.string.reorder))
                                                }
                                            }
                                            Text(
                                                text = if (item.quantity != null) "${item.name} (${item.quantity})" else item.name,
                                                style = MaterialTheme.typography.bodyLarge,
                                                textDecoration = if (item.isChecked) TextDecoration.LineThrough else null,
                                                modifier = Modifier.weight(1f)
                                            )
                                            var showMenu by remember { mutableStateOf(false) }
                                            IconButton(onClick = {
                                                shoppingListsViewModel.onSelectItem(item)
                                                showMenu = true
                                            }) {
                                                Icon(Icons.Default.MoreVert, contentDescription = stringResource(R.string.more_options))
                                                DropdownMenu(expanded = showMenu, onDismissRequest = { showMenu = false }) {
                                                    DropdownMenuItem(
                                                        text = { Text(dynamicStrings.editItem) },
                                                        onClick = {
                                                            shoppingListsViewModel.onShowRenameDialog(true)
                                                            showMenu = false
                                                        }
                                                    )
                                                    DropdownMenuItem(
                                                        text = { Text(stringResource(R.string.change_quantity)) },
                                                        onClick = {
                                                            shoppingListsViewModel.onShowQuantityDialog(true)
                                                            showMenu = false
                                                        }
                                                    )
                                                }
                                            }
                                            if (isReorderMode) {
                                                IconButton(modifier = Modifier.draggableHandle(), onClick = {}) {
                                                    Icon(Icons.Rounded.DragHandle, contentDescription = stringResource(R.string.reorder))
                                                }
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    if (showRenameDialog && selectedItem != null) {
        RenameItemDialog(
            item = selectedItem!!,
            onDismiss = { shoppingListsViewModel.onShowRenameDialog(false) },
            onRename = { newName ->
                coroutineScope.launch {
                    shoppingListsViewModel.saveShoppingListItem(selectedItem!!.copy(name = newName))
                }
                shoppingListsViewModel.onShowRenameDialog(false)
                shoppingListsViewModel.onSelectItem(null)
            },
            dynamicStrings = dynamicStrings
        )
    }

    if (showQuantityDialog && selectedItem != null) {
        QuantityDialog(
            item = selectedItem!!,
            onDismiss = { shoppingListsViewModel.onShowQuantityDialog(false) },
            onSetQuantity = { quantity ->
                coroutineScope.launch {
                    shoppingListsViewModel.saveShoppingListItem(selectedItem!!.copy(quantity = quantity))
                }
                shoppingListsViewModel.onShowQuantityDialog(false)
                shoppingListsViewModel.onSelectItem(null)
            }
        )
    }

    // Dialog for selecting WebApp shopping list
    if (showWebAppListSelectionDialog) {
        val deleteCompletedOnImport by shoppingListsViewModel.deleteCompletedOnImport.collectAsState()

        AlertDialog(
            onDismissRequest = { shoppingListsViewModel.dismissWebAppListSelectionDialog() },
            title = { Text(stringResource(R.string.select_web_app_list)) },
            text = {
                if (webAppShoppingLists.isEmpty()) {
                    Column(horizontalAlignment = Alignment.CenterHorizontally, modifier = Modifier.fillMaxWidth()) {
                        LinearProgressIndicator(modifier = Modifier.fillMaxWidth())
                        Spacer(modifier = Modifier.height(8.dp))
                        Text(stringResource(R.string.loading_web_app_lists))
                    }
                } else {
                    Column {
                        Row(
                            verticalAlignment = Alignment.CenterVertically,
                            modifier = Modifier
                                .fillMaxWidth()
                                .clickable { shoppingListsViewModel.onDeleteCompletedOnImportChange(!deleteCompletedOnImport) }
                                .padding(vertical = 8.dp)
                        ) {
                            androidx.compose.material3.Checkbox(
                                checked = deleteCompletedOnImport,
                                onCheckedChange = { shoppingListsViewModel.onDeleteCompletedOnImportChange(it) }
                            )
                            Text(stringResource(R.string.delete_completed_on_server))
                        }
                        LazyColumn {
                            items(webAppShoppingLists) { webAppList ->
                                Button(
                                    onClick = {
                                        listId?.let { localListId ->
                                            shoppingListsViewModel.importItemsFromWebApp(localListId, webAppList.id, deleteCompletedOnImport)
                                        }
                                    },
                                    modifier = Modifier
                                        .fillMaxWidth()
                                        .padding(vertical = 4.dp)
                                ) {
                                    Text(webAppList.name)
                                }
                            }
                        }
                    }
                }
            },
            confirmButton = {
                Button(onClick = { shoppingListsViewModel.dismissWebAppListSelectionDialog() }) {
                    Text(stringResource(R.string.cancel))
                }
            },
            properties = DialogProperties(dismissOnBackPress = true, dismissOnClickOutside = true)
        )
    }
}

@Composable
fun LockedScreenPlaceholder(onUnlockClick: () -> Unit) {
    val context = LocalContext.current
    val biometricManager = BiometricManager.from(context)
    val canUseBiometrics = remember {
        biometricManager.canAuthenticate(BiometricManager.Authenticators.BIOMETRIC_STRONG) == BiometricManager.BIOMETRIC_SUCCESS
    }

    Column(
        modifier = Modifier.fillMaxSize(),
        horizontalAlignment = Alignment.CenterHorizontally,
        verticalArrangement = Arrangement.Center
    ) {
        Icon(Icons.Default.Lock, contentDescription = null, modifier = Modifier.size(48.dp))
        Spacer(modifier = Modifier.height(16.dp))
        if (canUseBiometrics) {
            Button(onClick = onUnlockClick) {
                Text(stringResource(R.string.unlock))
            }
        } else {
            Text(
                text = stringResource(R.string.biometric_required_to_unlock),
                style = MaterialTheme.typography.bodyLarge,
                modifier = Modifier.padding(horizontal = 32.dp)
            )
        }
    }
}



@Composable
fun RenameItemDialog(item: ShoppingListItem, onDismiss: () -> Unit, onRename: (String) -> Unit, dynamicStrings: de.lxtools.noteshop.DynamicListStrings) {
    var text by remember { mutableStateOf(item.name) }
    AlertDialog(
        onDismissRequest = onDismiss,
        title = { Text(dynamicStrings.editItem) },
        text = {
            OutlinedTextField(
                value = text,
                onValueChange = { text = it },
                label = { Text(dynamicStrings.itemName) }
            )
        },
        confirmButton = {
            Button(onClick = { onRename(text) }) {
                Text(stringResource(R.string.save))
            }
        },
        dismissButton = {
            Button(onClick = onDismiss) {
                Text(stringResource(R.string.cancel))
            }
        }
    )
}

@Composable
fun QuantityDialog(item: ShoppingListItem, onDismiss: () -> Unit, onSetQuantity: (String?) -> Unit) {
    var text by remember { mutableStateOf(item.quantity ?: "") }
    AlertDialog(
        onDismissRequest = onDismiss,
        title = { Text(stringResource(R.string.change_quantity)) },
        text = {
            OutlinedTextField(
                value = text,
                onValueChange = { text = it },
                label = { Text(stringResource(R.string.quantity_hint)) }
            )
        },
        confirmButton = {
            Button(onClick = { onSetQuantity(text.ifBlank { null }) }) {
                Text(stringResource(R.string.save))
            }
        },
        dismissButton = {
            Button(onClick = onDismiss) {
                Text(stringResource(R.string.cancel))
            }
        }
    )
}
