package de.lxtools.noteshop.ui.recipes

import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import de.lxtools.noteshop.data.Recipe
import de.lxtools.noteshop.data.NoteshopRepository
import de.lxtools.noteshop.security.hashString
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.flow.firstOrNull
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.launch
import kotlinx.serialization.json.Json
import android.util.Log

class RecipesViewModel(private val noteshopRepository: NoteshopRepository) : ViewModel() {

    private val _searchQuery = MutableStateFlow("")
    val searchQuery: StateFlow<String> = _searchQuery.asStateFlow()

    private val _isReorderMode = MutableStateFlow(false)
    val isReorderMode: StateFlow<Boolean> = _isReorderMode.asStateFlow()

    fun enableReorderMode() {
        _isReorderMode.value = true
    }

    fun disableReorderMode() {
        _isReorderMode.value = false
    }

    fun updateSearchQuery(query: String) {
        _searchQuery.value = query
    }

    val uiState: StateFlow<RecipesUiState> = combine(
        noteshopRepository.getAllRecipesStream(),
        searchQuery
    ) { allRecipes, query ->
        val filteredRecipes = if (query.isBlank()) {
            allRecipes
        } else {
            allRecipes.filter {
                it.title.contains(query, ignoreCase = true) ||
                it.content.contains(query, ignoreCase = true)
            }
        }
        RecipesUiState(filteredRecipes)
    }
        .stateIn(
            scope = viewModelScope,
            started = SharingStarted.WhileSubscribed(TIMEOUT_MILLIS),
            initialValue = RecipesUiState()
        )

    private val _recipeDetails = MutableStateFlow(RecipeDetails())
    val recipeDetails: StateFlow<RecipeDetails> = _recipeDetails.asStateFlow()

    fun updateRecipeDetails(recipe: Recipe) {
        _recipeDetails.value = RecipeDetails(
            id = recipe.id,
            title = recipe.title,
            content = recipe.content,
            displayOrder = recipe.displayOrder,
            protectionHash = recipe.protectionHash,
            protectionType = recipe.protectionType,
            lockMethod = recipe.lockMethod
        )
    }

    fun updateRecipeDetails(recipeDetails: RecipeDetails) {
        Log.d("RecipesViewModel", "updateRecipeDetails called with details: $recipeDetails")
        _recipeDetails.value = recipeDetails
    }

    suspend fun saveRecipe() {
        if (recipeDetails.value.isValid()) {
            val currentRecipe = recipeDetails.value.toRecipe()

            if (currentRecipe.id == 0) {
                val recipesCount = noteshopRepository.getRecipesCount()
                val newRecipe = currentRecipe.copy(displayOrder = recipesCount)
                noteshopRepository.insertRecipe(newRecipe)
            } else {
                noteshopRepository.updateRecipe(currentRecipe)
            }
        }
    }

    suspend fun deleteRecipe(recipe: Recipe) {
        noteshopRepository.deleteRecipe(recipe)
    }

    fun setProtectionPassword(password: String) {
        viewModelScope.launch {
            val currentRecipeDetails = _recipeDetails.value
            if (password.isNotBlank()) {
                val hashedPassword = hashString(password)
                val updatedRecipe = currentRecipeDetails.toRecipe().copy(
                    protectionHash = hashedPassword,
                    protectionType = 1, // 1 for password protection
                    lockMethod = 1
                )
                noteshopRepository.updateRecipe(updatedRecipe)
                updateRecipeDetails(updatedRecipe) // Update the UI state
            } else {
                // Password is blank, so we remove protection
                val updatedRecipe = currentRecipeDetails.toRecipe().copy(
                    protectionHash = "",
                    protectionType = 0,
                    lockMethod = 0
                )
                noteshopRepository.updateRecipe(updatedRecipe)
                updateRecipeDetails(updatedRecipe) // Update the UI state
            }
        }
    }

    fun setProtectionBiometric(recipeId: Int) {
        viewModelScope.launch {
            val recipe = noteshopRepository.getRecipeStream(recipeId).firstOrNull()
            recipe?.let {
                val updatedRecipe = it.copy(
                    protectionHash = "", // Clear password hash
                    protectionType = 2, // 2 for biometric protection
                    lockMethod = 2
                )
                noteshopRepository.updateRecipe(updatedRecipe)
            }
        }
    }

    fun resetRecipeDetails() {
        _recipeDetails.value = RecipeDetails()
    }

    fun moveRecipe(from: Int, to: Int) {
        viewModelScope.launch {
            val list = uiState.value.recipeList.toMutableList()
            list.add(to, list.removeAt(from))
            val updatedRecipes = list.mapIndexed { index, item ->
                item.copy(displayOrder = index)
            }
            updatedRecipes.forEach { recipe ->
                noteshopRepository.updateRecipe(recipe)
            }
        }
    }

    fun formatRecipeForExport(recipe: Recipe): String {
        val builder = StringBuilder()
        builder.appendLine("Title: ${recipe.title}")
        builder.appendLine("Content: ${recipe.content}")
        return builder.toString()
    }

    fun exportRecipesToJson(recipes: List<Recipe>): String {
        return Json.encodeToString(recipes)
    }

    suspend fun importRecipesFromJson(jsonString: String) {
        try {
            val recipes = Json.decodeFromString<List<Recipe>>(jsonString)
            recipes.forEach { recipe ->
                val newRecipe = recipe.copy(id = 0)
                noteshopRepository.upsertRecipe(newRecipe)
            }
        } catch (e: Exception) {
            Log.e("RecipesViewModel", "Error parsing recipes JSON", e)
        }
    }

    companion object {
        private const val TIMEOUT_MILLIS = 5_000L
    }
}

data class RecipesUiState(
    val recipeList: List<Recipe> = listOf()
)

data class RecipeDetails(
    val id: Int = 0,
    val title: String = "",
    val content: String = "",
    val displayOrder: Int = 0,
    val protectionHash: String = "",
    val protectionType: Int = 0,
    val lockMethod: Int = 0
) {
    fun toRecipe(): Recipe = Recipe(
        id = id,
        title = title,
        content = content,
        displayOrder = displayOrder,
        protectionHash = protectionHash,
        protectionType = protectionType,
        lockMethod = lockMethod
    )

    fun isValid(): Boolean {
        return title.isNotBlank()
    }
}