package de.lxtools.noteshop.ui.recipes

import androidx.compose.animation.core.animateFloatAsState
import androidx.compose.foundation.gestures.detectTapGestures
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.foundation.lazy.rememberLazyListState
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Delete
import androidx.compose.material.icons.filled.Edit
import androidx.compose.material.icons.filled.Lock
import androidx.compose.material.icons.rounded.DragHandle
import androidx.compose.material3.Card
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.graphicsLayer
import androidx.compose.ui.input.pointer.pointerInput
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import de.lxtools.noteshop.R
import de.lxtools.noteshop.data.Recipe
import kotlinx.coroutines.launch
import sh.calvin.reorderable.ReorderableCollectionItemScope
import sh.calvin.reorderable.ReorderableItem
import sh.calvin.reorderable.rememberReorderableLazyListState

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun RecipesScreen(
    modifier: Modifier = Modifier,
    recipesViewModel: RecipesViewModel,
    onRecipeClick: (Int, Int) -> Unit,
    onEditRecipe: (Recipe) -> Unit
) {
    val uiState by recipesViewModel.uiState.collectAsState()
    val isReorderMode by recipesViewModel.isReorderMode.collectAsState()
    val coroutineScope = rememberCoroutineScope()

    Column(
        modifier = modifier.fillMaxSize()
    ) {
        if (uiState.recipeList.isEmpty()) {
            Text(text = stringResource(R.string.no_recipes_yet))
        } else {
            val lazyListState = rememberLazyListState()
            val reorderableLazyListState = rememberReorderableLazyListState(lazyListState) { from, to ->
                recipesViewModel.moveRecipe(from.index, to.index)
            }
            LazyColumn(
                state = lazyListState,
            ) {
                items(uiState.recipeList, { it.id }) { recipe ->
                    ReorderableItem(reorderableLazyListState, key = recipe.id) { isDragging ->
                        RecipeCard(
                            recipe = recipe,
                            onEditRecipe = onEditRecipe,
                            onDeleteRecipe = {
                                coroutineScope.launch {
                                    recipesViewModel.deleteRecipe(it)
                                }
                            },
                            modifier = Modifier.pointerInput(isReorderMode) {
                                if (!isReorderMode) {
                                    detectTapGestures(
                                        onLongPress = { recipesViewModel.enableReorderMode() },
                                        onTap = { onRecipeClick(recipe.id, recipe.protectionType) }
                                    )
                                }
                            },
                            scope = this,
                            isDragging = isDragging,
                            isReorderMode = isReorderMode
                        )
                    }
                }
            }
        }
    }
}

@Composable
fun RecipeCard(
    recipe: Recipe,
    onEditRecipe: (Recipe) -> Unit,
    onDeleteRecipe: (Recipe) -> Unit,
    modifier: Modifier = Modifier,
    scope: ReorderableCollectionItemScope,
    isDragging: Boolean,
    isReorderMode: Boolean
) {
    val scale = animateFloatAsState(if (isDragging) 1.05f else 1.0f, label = "scale")

    Card(
        modifier = modifier
            .fillMaxWidth()
            .padding(vertical = 4.dp, horizontal = 16.dp)
            .graphicsLayer {
                scaleX = scale.value
                scaleY = scale.value
            }
    ) {
        Column(modifier = Modifier.padding(16.dp)) {
            Row(
                modifier = Modifier.fillMaxWidth(),
                horizontalArrangement = Arrangement.SpaceBetween,
                verticalAlignment = Alignment.CenterVertically
            ) {
                if (isReorderMode) {
                    IconButton(modifier = with(scope) { Modifier.draggableHandle() }, onClick = {}) {
                        Icon(Icons.Rounded.DragHandle, contentDescription = stringResource(R.string.reorder))
                    }
                }
                if (recipe.lockMethod != 0) {
                    Icon(Icons.Default.Lock, contentDescription = null, modifier = Modifier.padding(end = 8.dp))
                }
                Text(
                    text = recipe.title,
                    style = MaterialTheme.typography.titleMedium,
                    modifier = Modifier.weight(1f)
                )
                IconButton(onClick = { onEditRecipe(recipe) }, enabled = !isReorderMode) {
                    Icon(Icons.Default.Edit, contentDescription = "Edit recipe")
                }
                IconButton(onClick = { onDeleteRecipe(recipe) }, enabled = !isReorderMode) {
                    Icon(Icons.Default.Delete, contentDescription = "Delete recipe")
                }
                if (isReorderMode) {
                    IconButton(modifier = with(scope) { Modifier.draggableHandle() }, onClick = {}) {
                        Icon(Icons.Rounded.DragHandle, contentDescription = stringResource(R.string.reorder))
                    }
                }
            }
        }
    }
}