package de.lxtools.noteshop.ui.recipes

import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Edit
import androidx.compose.material.icons.filled.Lock
import androidx.compose.material.icons.filled.Visibility
import androidx.compose.material3.Button
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import androidx.biometric.BiometricManager
import de.lxtools.noteshop.R
import dev.jeziellago.compose.markdowntext.MarkdownText
import androidx.compose.material3.MaterialTheme

import androidx.compose.runtime.DisposableEffect

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun RecipeDetailScreen(
    recipeId: Int?,
    recipesViewModel: RecipesViewModel,
    modifier: Modifier = Modifier,
    isContentDecrypted: Boolean,
    onUnlockClick: (Int) -> Unit
) {

    val recipeDetails by recipesViewModel.recipeDetails.collectAsState()
    var isEditMode by remember { mutableStateOf(false) }
    DisposableEffect(recipeId) {
        onDispose {
        }
    }

    LaunchedEffect(key1 = recipeId) {
        if (recipeId != null) {
            val recipe = recipesViewModel.uiState.value.recipeList.find { it.id == recipeId }
            if (recipe != null) {
                recipesViewModel.updateRecipeDetails(recipe)
            }
        }
    }

    if (recipeDetails.protectionHash.isNotEmpty() && !isContentDecrypted) {
        LockedScreenPlaceholder(onUnlockClick = { onUnlockClick(recipeDetails.protectionType) })
    } else {
        Column(
            modifier = modifier
                .fillMaxSize()
                .padding(16.dp)
        ) {
            Row(
                modifier = Modifier.fillMaxWidth(),
                horizontalArrangement = Arrangement.End
            ) {
                IconButton(onClick = { isEditMode = !isEditMode }) {
                    Icon(
                        imageVector = if (isEditMode) Icons.Default.Visibility else Icons.Default.Edit,
                        contentDescription = if (isEditMode) "View" else "Edit"
                    )
                }
            }
            if (isEditMode) {
                OutlinedTextField(
                    value = recipeDetails.content,
                    onValueChange = { recipesViewModel.updateRecipeDetails(recipeDetails.copy(content = it)) },
                    modifier = Modifier
                        .fillMaxWidth()
                        .weight(1f)
                )
            } else {
                val scrollState = rememberScrollState()
                MarkdownText(
                    markdown = recipeDetails.content,
                    modifier = Modifier
                        .fillMaxWidth()
                        .weight(1f)
                        .verticalScroll(scrollState)
                )
            }
        }
    }
}

@Composable
fun LockedScreenPlaceholder(onUnlockClick: () -> Unit) {
    val context = LocalContext.current
    val biometricManager = BiometricManager.from(context)
    val canUseBiometrics = remember {
        biometricManager.canAuthenticate(BiometricManager.Authenticators.BIOMETRIC_STRONG) == BiometricManager.BIOMETRIC_SUCCESS
    }

    Column(
        modifier = Modifier.fillMaxSize(),
        horizontalAlignment = Alignment.CenterHorizontally,
        verticalArrangement = Arrangement.Center
    ) {
        Icon(Icons.Default.Lock, contentDescription = null, modifier = Modifier.size(48.dp))
        Spacer(modifier = Modifier.height(16.dp))
        if (canUseBiometrics) {
            Button(onClick = onUnlockClick) {
                Text(stringResource(R.string.unlock))
            }
        } else {
            Text(
                text = stringResource(R.string.biometric_required_to_unlock),
                style = MaterialTheme.typography.bodyLarge,
                modifier = Modifier.padding(horizontal = 32.dp)
            )
        }
    }
}
