package de.lxtools.noteshop.ui.notes

import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import de.lxtools.noteshop.data.Note
import de.lxtools.noteshop.data.NoteshopRepository
import de.lxtools.noteshop.security.hashString
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.flow.firstOrNull
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.launch
import kotlinx.serialization.json.Json
import android.util.Log

class NotesViewModel(private val noteshopRepository: NoteshopRepository) : ViewModel() {

    private val _searchQuery = MutableStateFlow("")
    val searchQuery: StateFlow<String> = _searchQuery.asStateFlow()

    private val _isReorderMode = MutableStateFlow(false)
    val isReorderMode: StateFlow<Boolean> = _isReorderMode.asStateFlow()

    fun enableReorderMode() {
        _isReorderMode.value = true
    }

    fun disableReorderMode() {
        _isReorderMode.value = false
    }

    fun updateSearchQuery(query: String) {
        _searchQuery.value = query
    }

    val uiState: StateFlow<NotesUiState> = combine(
        noteshopRepository.getAllNotesStream(),
        searchQuery
    ) { allNotes, query ->
        val filteredNotes = if (query.isBlank()) {
            allNotes
        } else {
            allNotes.filter {
                it.title.contains(query, ignoreCase = true) ||
                it.content.contains(query, ignoreCase = true)
            }
        }
        NotesUiState(filteredNotes)
    }
        .stateIn(
            scope = viewModelScope,
            started = SharingStarted.WhileSubscribed(TIMEOUT_MILLIS),
            initialValue = NotesUiState()
        )

    private val _noteDetails = MutableStateFlow(NoteDetails())
    val noteDetails: StateFlow<NoteDetails> = _noteDetails.asStateFlow()

    fun updateNoteDetails(note: Note) {
        _noteDetails.value = NoteDetails(
            id = note.id,
            title = note.title,
            content = note.content,
            displayOrder = note.displayOrder,
            protectionHash = note.protectionHash,
            protectionType = note.protectionType,
            lockMethod = note.lockMethod
        )
    }

    fun updateNoteDetails(noteDetails: NoteDetails) {
        _noteDetails.value = noteDetails
    }

    suspend fun saveNote() {
        if (noteDetails.value.isValid()) {
            val currentNote = noteDetails.value.toNote()

            if (currentNote.id == 0) {
                val notesCount = noteshopRepository.getNotesCount()
                val newNote = currentNote.copy(displayOrder = notesCount)
                noteshopRepository.insertNote(newNote)
            } else {
                noteshopRepository.updateNote(currentNote)
            }
        }
    }

    suspend fun deleteNote(note: Note) {
        noteshopRepository.deleteNote(note)
    }

    fun setProtectionPassword(password: String) {
        viewModelScope.launch {
            val currentNoteDetails = _noteDetails.value
            if (password.isNotBlank()) {
                val hashedPassword = hashString(password)
                val updatedNote = currentNoteDetails.toNote().copy(
                    protectionHash = hashedPassword,
                    protectionType = 1, // 1 for password protection
                    lockMethod = 1
                )
                noteshopRepository.updateNote(updatedNote)
                updateNoteDetails(updatedNote) // Update the UI state
            } else {
                // Password is blank, so we remove protection
                val updatedNote = currentNoteDetails.toNote().copy(
                    protectionHash = "",
                    protectionType = 0,
                    lockMethod = 0
                )
                noteshopRepository.updateNote(updatedNote)
                updateNoteDetails(updatedNote) // Update the UI state
            }
        }
    }

    fun setProtectionBiometric(noteId: Int) {
        viewModelScope.launch {
            val note = noteshopRepository.getNoteStream(noteId).firstOrNull()
            note?.let {
                val updatedNote = it.copy(
                    protectionHash = "", // Clear password hash
                    protectionType = 2, // 2 for biometric protection
                    lockMethod = 2
                )
                noteshopRepository.updateNote(updatedNote)
            }
        }
    }

    fun resetNoteDetails() {
        _noteDetails.value = NoteDetails()
    }

    fun moveNote(from: Int, to: Int) {
        viewModelScope.launch {
            val list = uiState.value.noteList.toMutableList()
            list.add(to, list.removeAt(from))
            val updatedNotes = list.mapIndexed { index, item ->
                item.copy(displayOrder = index)
            }
            updatedNotes.forEach { note ->
                noteshopRepository.updateNote(note)
            }
        }
    }

    fun formatNoteForExport(note: Note): String {
        val builder = StringBuilder()
        builder.appendLine("Title: ${note.title}")
        builder.appendLine("Content: ${note.content}")
        return builder.toString()
    }

    fun exportNotesToJson(notes: List<Note>): String {
        return Json.encodeToString(notes)
    }

    suspend fun importNotesFromJson(jsonString: String) {
        try {
            val notes = Json.decodeFromString<List<Note>>(jsonString)
            notes.forEach { note ->
                val newNote = note.copy(id = 0)
                noteshopRepository.upsertNote(newNote)
            }
        } catch (e: Exception) {
            Log.e("NotesViewModel", "Error parsing notes JSON", e)
        }
    }

    companion object {
        private const val TIMEOUT_MILLIS = 5_000L
    }
}

data class NotesUiState(
    val noteList: List<Note> = listOf()
)

data class NoteDetails(
    val id: Int = 0,
    val title: String = "",
    val content: String = "",
    val displayOrder: Int = 0,
    val protectionHash: String = "",
    val protectionType: Int = 0,
    val lockMethod: Int = 0
) {
    fun toNote(): Note = Note(
        id = id,
        title = title,
        content = content,
        displayOrder = displayOrder,
        protectionHash = protectionHash,
        protectionType = protectionType,
        lockMethod = lockMethod
    )

    fun isValid(): Boolean {
        return title.isNotBlank()
    }
}