package de.lxtools.noteshop.ui.notes

import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Lock
import androidx.compose.material3.Button
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import de.lxtools.noteshop.R
import androidx.compose.ui.platform.LocalContext
import androidx.biometric.BiometricManager
import androidx.compose.runtime.remember
import androidx.compose.material3.MaterialTheme
import androidx.compose.runtime.DisposableEffect

import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.ui.platform.LocalLifecycleOwner
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleEventObserver
import kotlinx.coroutines.launch

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun NoteDetailScreen(
    noteId: Int?,
    notesViewModel: NotesViewModel,
    modifier: Modifier = Modifier,
    isContentDecrypted: Boolean,
    onUnlockClick: (Int) -> Unit
) {
    val noteDetails by notesViewModel.noteDetails.collectAsState()
    val scope = rememberCoroutineScope()
    val lifecycleOwner = LocalLifecycleOwner.current

    DisposableEffect(lifecycleOwner) {
        val observer = LifecycleEventObserver { _, event ->
            if (event == Lifecycle.Event.ON_PAUSE) {
                scope.launch {
                    notesViewModel.saveNote()
                }
            }
        }
        lifecycleOwner.lifecycle.addObserver(observer)
        onDispose {
            lifecycleOwner.lifecycle.removeObserver(observer)
        }
    }

    LaunchedEffect(key1 = noteId) {
        if (noteId != null) {
            val note = notesViewModel.uiState.value.noteList.find { it.id == noteId }
            if (note != null) {
                notesViewModel.updateNoteDetails(note)
            }
        } else {
            // If creating a new note, reset details
            notesViewModel.resetNoteDetails()
        }
    }

    if (noteDetails.protectionHash.isNotEmpty() && !isContentDecrypted) {
        LockedScreenPlaceholder(onUnlockClick = { onUnlockClick(noteDetails.protectionType) })
    } else {
        Column(
            modifier = modifier
                .fillMaxSize()
                .padding(16.dp)
        ) {
            OutlinedTextField(
                value = noteDetails.content,
                onValueChange = { notesViewModel.updateNoteDetails(noteDetails.copy(content = it)) },
                modifier = Modifier
                    .fillMaxWidth()
                    .weight(1f)
            )
        }
    }
}

@Composable
fun LockedScreenPlaceholder(onUnlockClick: () -> Unit) {
    val context = LocalContext.current
    val biometricManager = BiometricManager.from(context)
    val canUseBiometrics = remember {
        biometricManager.canAuthenticate(BiometricManager.Authenticators.BIOMETRIC_STRONG) == BiometricManager.BIOMETRIC_SUCCESS
    }

    Column(
        modifier = Modifier.fillMaxSize(),
        horizontalAlignment = Alignment.CenterHorizontally,
        verticalArrangement = Arrangement.Center
    ) {
        Icon(Icons.Default.Lock, contentDescription = null, modifier = Modifier.size(48.dp))
        Spacer(modifier = Modifier.height(16.dp))
        if (canUseBiometrics) {
            Button(onClick = onUnlockClick) {
                Text(stringResource(R.string.unlock))
            }
        } else {
            Text(
                text = stringResource(R.string.biometric_required_to_unlock),
                style = MaterialTheme.typography.bodyLarge,
                modifier = Modifier.padding(horizontal = 32.dp)
            )
        }
    }
}