package de.lxtools.noteshop.ui.appshell

import android.content.Intent
import android.net.Uri
import androidx.activity.result.ActivityResultLauncher
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material.icons.filled.Add
import androidx.compose.material.icons.filled.Cloud
import androidx.compose.material.icons.filled.CloudOff
import androidx.compose.material.icons.filled.Menu
import androidx.compose.material.icons.filled.MoreVert
import androidx.compose.material.icons.filled.Refresh
import androidx.compose.material.icons.filled.Search
import androidx.compose.material3.DropdownMenu
import androidx.compose.material3.DropdownMenuItem
import androidx.compose.material3.DrawerState
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.Text
import androidx.compose.material3.TopAppBar
import androidx.compose.material3.TopAppBarDefaults
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import androidx.core.content.FileProvider
import de.lxtools.noteshop.R
import de.lxtools.noteshop.Screen
import de.lxtools.noteshop.getDynamicRecipeStrings
import de.lxtools.noteshop.ui.LockableItemType
import de.lxtools.noteshop.ui.notes.NotesViewModel
import de.lxtools.noteshop.ui.recipes.RecipesViewModel
import de.lxtools.noteshop.ui.shopping.ShoppingListsViewModel
import kotlinx.coroutines.launch

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun AppTopBar(
    currentScreen: Screen,
    onScreenChange: (Screen) -> Unit,
    drawerState: DrawerState,
    shoppingListsViewModel: ShoppingListsViewModel,
    notesViewModel: NotesViewModel,
    recipesViewModel: RecipesViewModel,
    shoppingListsTitle: String,
    recipesTitle: String,
    onShowListDialog: () -> Unit,
    onShowNoteDialog: () -> Unit,
    onShowRecipeDialog: () -> Unit,
    selectedListId: Int?,
    onSetSelectedListId: (Int?) -> Unit,
    onSetSelectedNoteId: (Int?) -> Unit,
    onSetSelectedRecipeId: (Int?) -> Unit,
    exportLauncher: ActivityResultLauncher<String>,
    noteExportLauncher: ActivityResultLauncher<String>,
    recipeExportLauncher: ActivityResultLauncher<String>,
    onShowChooseLockMethodDialog: (LockableItemType, Int) -> Unit
) {
    val scope = rememberCoroutineScope()
    val context = LocalContext.current

    val isReorderMode by shoppingListsViewModel.isReorderMode.collectAsState()
    val isNotesReorderMode by notesViewModel.isReorderMode.collectAsState()
    val isRecipesReorderMode by recipesViewModel.isReorderMode.collectAsState()
    val isDetailSearchActive by shoppingListsViewModel.isDetailSearchActive.collectAsState()
    val shoppingListWithItems by shoppingListsViewModel.getShoppingListWithItemsStream(
        selectedListId ?: 0
    ).collectAsState(initial = null)
    val noteDetails by notesViewModel.noteDetails.collectAsState()
    val recipeDetails by recipesViewModel.recipeDetails.collectAsState()

    val topBarTitle = when (currentScreen) {
        is Screen.ShoppingListDetail -> shoppingListWithItems?.shoppingList?.name ?: ""
        is Screen.NoteDetail -> notesViewModel.noteDetails.collectAsState().value.title
        is Screen.RecipeDetail -> recipesViewModel.recipeDetails.collectAsState().value.title
        is Screen.ShoppingLists -> shoppingListsTitle
        is Screen.Recipes -> recipesTitle
        else -> stringResource(id = currentScreen.titleRes)
    }

    Column {
        TopAppBar(
            title = {
                Text(
                    if (currentScreen == Screen.ShoppingListDetail && isDetailSearchActive) stringResource(
                        R.string.search
                    ) else topBarTitle
                )
            },
            navigationIcon = {
                if (isReorderMode || isNotesReorderMode || isRecipesReorderMode) {
                    IconButton(onClick = {
                        if (isReorderMode) shoppingListsViewModel.disableReorderMode()
                        if (isNotesReorderMode) notesViewModel.disableReorderMode()
                        if (isRecipesReorderMode) recipesViewModel.disableReorderMode()
                    }) {
                        Icon(
                            Icons.AutoMirrored.Filled.ArrowBack,
                            contentDescription = stringResource(R.string.back)
                        )
                    }
                } else {
                    when (currentScreen) {
                        is Screen.ShoppingListDetail -> {
                            IconButton(onClick = {
                                if (isDetailSearchActive) {
                                    shoppingListsViewModel.toggleDetailSearch()
                                } else {
                                    onScreenChange(Screen.ShoppingLists)
                                    onSetSelectedListId(null)
                                }
                            }) {
                                Icon(
                                    Icons.AutoMirrored.Filled.ArrowBack,
                                    contentDescription = stringResource(R.string.back)
                                )
                            }
                        }

                        is Screen.NoteDetail -> {
                            IconButton(onClick = {
                                scope.launch {
                                    notesViewModel.saveNote()
                                    onScreenChange(Screen.Notes)
                                    onSetSelectedNoteId(null)
                                }
                            }) {
                                Icon(
                                    Icons.AutoMirrored.Filled.ArrowBack,
                                    contentDescription = stringResource(R.string.back)
                                )
                            }
                        }

                        is Screen.RecipeDetail -> {
                            IconButton(onClick = {
                                scope.launch {
                                    recipesViewModel.saveRecipe()
                                    onScreenChange(Screen.Recipes)
                                    onSetSelectedRecipeId(null)
                                }
                            }) {
                                Icon(
                                    Icons.AutoMirrored.Filled.ArrowBack,
                                    contentDescription = stringResource(R.string.back)
                                )
                            }
                        }

                        is Screen.WebAppIntegration -> {
                            IconButton(onClick = { onScreenChange(Screen.Settings) }) {
                                Icon(
                                    Icons.AutoMirrored.Filled.ArrowBack,
                                    contentDescription = stringResource(R.string.back)
                                )
                            }
                        }

                        else -> {
                            IconButton(onClick = { scope.launch { drawerState.apply { if (isClosed) open() else close() } } }) {
                                Icon(
                                    imageVector = Icons.Default.Menu,
                                    contentDescription = stringResource(id = R.string.menu_open)
                                )
                            }
                        }
                    }
                }
            },
            actions = {
                if (currentScreen == Screen.ShoppingLists && !isReorderMode) {
                    IconButton(onClick = {
                        shoppingListsViewModel.resetListDetails()
                        onShowListDialog()
                    }) {
                        Icon(
                            Icons.Default.Add,
                            contentDescription = stringResource(R.string.add_shopping_list)
                        )
                    }
                }
                if (currentScreen == Screen.Notes && !isNotesReorderMode) {
                    IconButton(onClick = {
                        notesViewModel.resetNoteDetails()
                        onShowNoteDialog()
                    }) {
                        Icon(
                            Icons.Default.Add,
                            contentDescription = stringResource(R.string.add_note)
                        )
                    }
                }
                if (currentScreen == Screen.Recipes && !isRecipesReorderMode) {
                    IconButton(onClick = {
                        recipesViewModel.resetRecipeDetails()
                        onShowRecipeDialog()
                    }) {
                        Icon(
                            Icons.Default.Add,
                            contentDescription = getDynamicRecipeStrings(recipesTitle).addItem
                        )
                    }
                }
                when (currentScreen) {
                    is Screen.ShoppingListDetail -> {
                        val isWebAppEnabled by shoppingListsViewModel.isWebAppEnabled.collectAsState()
                        IconButton(
                            onClick = {
                                selectedListId?.let {
                                    shoppingListsViewModel.refreshWebAppShoppingLists(it)
                                }
                            },
                            enabled = isWebAppEnabled
                        ) {
                            Icon(
                                Icons.Default.Refresh,
                                contentDescription = stringResource(R.string.import_from_web_app),
                                tint = if (isWebAppEnabled) MaterialTheme.colorScheme.primary else Color.Gray
                            )
                        }
                        IconButton(onClick = { shoppingListsViewModel.toggleWebAppEnabled() }) {
                            Icon(
                                imageVector = if (isWebAppEnabled) Icons.Default.Cloud else Icons.Default.CloudOff,
                                contentDescription = stringResource(R.string.toggle_web_app_integration),
                                tint = if (isWebAppEnabled) MaterialTheme.colorScheme.primary else Color.Gray
                            )
                        }
                        IconButton(onClick = { shoppingListsViewModel.toggleDetailSearch() }) {
                            Icon(Icons.Default.Search, contentDescription = "Search")
                        }
                        var showMenu by remember { mutableStateOf(false) }
                        IconButton(onClick = { showMenu = !showMenu }) {
                            Icon(Icons.Default.MoreVert, contentDescription = "More")
                        }
                        DropdownMenu(
                            expanded = showMenu,
                            onDismissRequest = { showMenu = false }
                        ) {
                            DropdownMenuItem(
                                text = { Text(stringResource(R.string.export)) },
                                onClick = {
                                    showMenu = false
                                    shoppingListWithItems?.let {
                                        val fileName = "${it.shoppingList.name}.txt"
                                        exportLauncher.launch(fileName)
                                    }
                                }
                            )
                            DropdownMenuItem(
                                text = { Text(stringResource(R.string.share)) },
                                onClick = {
                                    showMenu = false
                                    shoppingListWithItems?.let { list ->
                                        val content =
                                            shoppingListsViewModel.formatShoppingListForExport(
                                                list
                                            )
                                        val fileName = "${list.shoppingList.name}.txt"
                                        val cachePath = java.io.File(
                                            context.cacheDir,
                                            "shared_files"
                                        )
                                        cachePath.mkdirs()
                                        val newFile = java.io.File(cachePath, fileName)
                                        newFile.writeText(content)

                                        val contentUri: Uri =
                                            FileProvider.getUriForFile(
                                                context,
                                                "${context.packageName}.provider",
                                                newFile
                                            )

                                        val shareIntent = Intent(Intent.ACTION_SEND).apply {
                                            type = "text/plain"
                                            putExtra(Intent.EXTRA_STREAM, contentUri)
                                            addFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION)
                                        }
                                        context.startActivity(
                                            Intent.createChooser(
                                                shareIntent,
                                                context.getString(R.string.share)
                                            )
                                        )
                                    }
                                }
                            )
                            shoppingListWithItems?.let { listWithItems ->
                                DropdownMenuItem(
                                    text = { Text(stringResource(R.string.lock)) },
                                    onClick = {
                                        showMenu = false
                                        onShowChooseLockMethodDialog(LockableItemType.SHOPPING_LIST, listWithItems.shoppingList.id)
                                    }
                                )
                                if (listWithItems.shoppingList.protectionType != 0) {
                                    DropdownMenuItem(
                                        text = { Text(stringResource(R.string.remove_lock)) },
                                        onClick = {
                                            showMenu = false
                                            shoppingListsViewModel.setProtection(listWithItems.shoppingList.id, "")
                                        }
                                    )
                                }
                            }
                        }
                    }

                    is Screen.NoteDetail -> {
                        var showMenu by remember { mutableStateOf(false) }
                        IconButton(onClick = { showMenu = !showMenu }) {
                            Icon(Icons.Default.MoreVert, contentDescription = "More")
                        }
                        DropdownMenu(
                            expanded = showMenu,
                            onDismissRequest = { showMenu = false }
                        ) {
                            DropdownMenuItem(
                                text = { Text(stringResource(R.string.export)) },
                                onClick = {
                                    showMenu = false
                                    notesViewModel.noteDetails.value.toNote()
                                        .let { note ->
                                            val fileName = "${note.title}.txt"
                                            noteExportLauncher.launch(fileName)
                                        }
                                }
                            )
                            DropdownMenuItem(
                                text = { Text(stringResource(R.string.share)) },
                                onClick = {
                                    showMenu = false
                                    notesViewModel.noteDetails.value.toNote()
                                        .let { note ->
                                            val content =
                                                notesViewModel.formatNoteForExport(note)
                                            val fileName = "${note.title}.txt"
                                            val cachePath = java.io.File(
                                                context.cacheDir,
                                                "shared_files"
                                            )
                                            cachePath.mkdirs()
                                            val newFile =
                                                java.io.File(cachePath, fileName)
                                            newFile.writeText(content)

                                            val contentUri: Uri =
                                                FileProvider.getUriForFile(
                                                    context,
                                                    "${context.packageName}.provider",
                                                    newFile
                                                )

                                            val shareIntent =
                                                Intent(Intent.ACTION_SEND).apply {
                                                    type = "text/plain"
                                                    putExtra(
                                                        Intent.EXTRA_STREAM,
                                                        contentUri
                                                    )
                                                    addFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION)
                                                }
                                            context.startActivity(
                                                Intent.createChooser(
                                                    shareIntent,
                                                    context.getString(R.string.share)
                                                )
                                            )
                                        }
                                }
                            )
                            noteDetails.let {
                                DropdownMenuItem(
                                    text = { Text(stringResource(R.string.lock)) },
                                    onClick = {
                                        showMenu = false
                                        onShowChooseLockMethodDialog(LockableItemType.NOTE, it.id)
                                    }
                                )
                                if (it.protectionType != 0) {
                                    DropdownMenuItem(
                                        text = { Text(stringResource(R.string.remove_lock)) },
                                        onClick = {
                                            showMenu = false
                                            notesViewModel.setProtectionPassword("")
                                        }
                                    )
                                }
                            }
                        }
                    }

                    is Screen.RecipeDetail -> {
                        var showMenu by remember { mutableStateOf(false) }
                        IconButton(onClick = { showMenu = !showMenu }) {
                            Icon(Icons.Default.MoreVert, contentDescription = "More")
                        }
                        DropdownMenu(
                            expanded = showMenu,
                            onDismissRequest = { showMenu = false }
                        ) {
                            DropdownMenuItem(
                                text = { Text(stringResource(R.string.export)) },
                                onClick = {
                                    showMenu = false
                                    recipesViewModel.recipeDetails.value.toRecipe()
                                        .let { recipe ->
                                            val fileName = "${recipe.title}.txt"
                                            recipeExportLauncher.launch(fileName)
                                        }
                                }
                            )
                            DropdownMenuItem(
                                text = { Text(stringResource(R.string.share)) },
                                onClick = {
                                    showMenu = false
                                    recipesViewModel.recipeDetails.value.toRecipe()
                                        .let { recipe ->
                                            val content =
                                                recipesViewModel.formatRecipeForExport(
                                                    recipe
                                                )
                                            val fileName = "${recipe.title}.txt"
                                            val cachePath = java.io.File(
                                                context.cacheDir,
                                                "shared_files"
                                            )
                                            cachePath.mkdirs()
                                            val newFile =
                                                java.io.File(cachePath, fileName)
                                            newFile.writeText(content)

                                            val contentUri: Uri =
                                                FileProvider.getUriForFile(
                                                    context,
                                                    "${context.packageName}.provider",
                                                    newFile
                                                )

                                            val shareIntent =
                                                Intent(Intent.ACTION_SEND).apply {
                                                    type = "text/plain"
                                                    putExtra(
                                                        Intent.EXTRA_STREAM,
                                                        contentUri
                                                    )
                                                    addFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION)
                                                }
                                            context.startActivity(
                                                Intent.createChooser(
                                                    shareIntent,
                                                    context.getString(R.string.share)
                                                )
                                            )
                                        }
                                }
                            )
                            recipeDetails.let {
                                DropdownMenuItem(
                                    text = { Text(stringResource(R.string.lock)) },
                                    onClick = {
                                        showMenu = false
                                        onShowChooseLockMethodDialog(LockableItemType.RECIPE, it.id)
                                    }
                                )
                                if (it.protectionType != 0) {
                                    DropdownMenuItem(
                                        text = { Text(stringResource(R.string.remove_lock)) },
                                        onClick = {
                                            showMenu = false
                                            recipesViewModel.setProtectionPassword("")
                                        }
                                    )
                                }
                            }
                        }
                    }

                    else -> {}
                }
            },
            colors = TopAppBarDefaults.topAppBarColors(
                containerColor = Color.Transparent
            )
        )

        when (currentScreen) {
            Screen.ShoppingLists -> {
                OutlinedTextField(
                    value = shoppingListsViewModel.searchQuery.collectAsState().value,
                    onValueChange = shoppingListsViewModel::updateSearchQuery,
                    label = { Text(stringResource(R.string.search_list_hint)) },
                    leadingIcon = {
                        Icon(
                            Icons.Default.Search,
                            contentDescription = null
                        )
                    },
                    modifier = Modifier
                        .fillMaxWidth()
                        .padding(horizontal = 16.dp)
                )
                Spacer(modifier = Modifier.height(16.dp))
            }

            Screen.Notes -> {
                OutlinedTextField(
                    value = notesViewModel.searchQuery.collectAsState().value,
                    onValueChange = notesViewModel::updateSearchQuery,
                    label = { Text(stringResource(R.string.search_notes_hint)) },
                    leadingIcon = {
                        Icon(
                            Icons.Default.Search,
                            contentDescription = null
                        )
                    },
                    modifier = Modifier
                        .fillMaxWidth()
                        .padding(horizontal = 16.dp)
                )
                Spacer(modifier = Modifier.height(16.dp))
            }

            Screen.Recipes -> {
                OutlinedTextField(
                    value = recipesViewModel.searchQuery.collectAsState().value,
                    onValueChange = recipesViewModel::updateSearchQuery,
                    label = { Text(getDynamicRecipeStrings(recipesTitle).searchHint) },
                    leadingIcon = {
                        Icon(
                            Icons.Default.Search,
                            contentDescription = null
                        )
                    },
                    modifier = Modifier
                        .fillMaxWidth()
                        .padding(horizontal = 16.dp)
                )
                Spacer(modifier = Modifier.height(16.dp))
            }

            else -> {}
        }
    }
}