package de.lxtools.noteshop.ui.appshell

import androidx.compose.foundation.Image
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Edit
import androidx.compose.material.icons.filled.Info
import androidx.compose.material.icons.filled.Settings
import androidx.compose.material3.*
import androidx.compose.runtime.Composable
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import de.lxtools.noteshop.R
import de.lxtools.noteshop.Screen
import de.lxtools.noteshop.ui.notes.NotesViewModel
import de.lxtools.noteshop.ui.shopping.ShoppingListsViewModel
import kotlinx.coroutines.launch

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun AppDrawer(
    drawerState: DrawerState,
    currentScreen: Screen,
    onScreenChange: (Screen) -> Unit,
    shoppingListsViewModel: ShoppingListsViewModel,
    notesViewModel: NotesViewModel,
    shoppingListsTitle: String,
    recipesTitle: String,
    onShowRenameDialog: () -> Unit,
    onShowRecipeRenameDialog: () -> Unit,
    onShowJsonDialog: () -> Unit,
    databasePath: String?,
    onChooseDatabaseLocation: () -> Unit,
    onSetSelectedListId: (Int?) -> Unit,
    onSetSelectedNoteId: (Int?) -> Unit,
) {
    val scope = rememberCoroutineScope()
    val context = LocalContext.current
    val navigationItems = listOf(
        Screen.ShoppingLists,
        Screen.Recipes,
        Screen.Notes
    )

    ModalDrawerSheet(modifier = Modifier.windowInsetsPadding(WindowInsets.systemBars)) {
        LazyColumn {
            item {
                Row(
                    modifier = Modifier
                        .fillMaxWidth()
                        .background(Color(0xFF6975BC))
                        .padding(16.dp),
                    verticalAlignment = Alignment.CenterVertically
                ) {
                    Image(
                        painter = painterResource(id = R.drawable.noteshop_fg_logo6),
                        contentDescription = null,
                        modifier = Modifier.size(48.dp)
                    )
                    Spacer(modifier = Modifier.width(12.dp))
                    Text(
                        text = stringResource(id = R.string.app_name),
                        style = MaterialTheme.typography.titleLarge,
                        color = Color.White
                    )
                }
            }

            items(navigationItems, key = { it.route }) { screen ->
                when (screen) {
                    is Screen.ShoppingLists -> {
                        NavigationDrawerItem(
                            label = {
                                Row(verticalAlignment = Alignment.CenterVertically) {
                                    Text(shoppingListsTitle, modifier = Modifier.weight(1f))
                                    IconButton(onClick = { onShowRenameDialog() }) {
                                        Icon(
                                            Icons.Default.Edit,
                                            contentDescription = stringResource(R.string.rename_title)
                                        )
                                    }
                                }
                            },
                            selected = screen == currentScreen,
                            onClick = {
                                shoppingListsViewModel.disableReorderMode()
                                notesViewModel.disableReorderMode()
                                onScreenChange(screen)
                                onSetSelectedListId(null)
                                onSetSelectedNoteId(null)
                                scope.launch { drawerState.close() }
                            }
                        )
                    }

                    is Screen.Recipes -> {
                        NavigationDrawerItem(
                            label = {
                                Row(verticalAlignment = Alignment.CenterVertically) {
                                    Text(recipesTitle, modifier = Modifier.weight(1f))
                                    IconButton(onClick = {
                                        onShowRecipeRenameDialog()
                                    }) {
                                        Icon(
                                            Icons.Default.Edit,
                                            contentDescription = stringResource(R.string.rename_title)
                                        )
                                    }
                                }
                            },
                            selected = screen == currentScreen,
                            onClick = {
                                shoppingListsViewModel.disableReorderMode()
                                notesViewModel.disableReorderMode()
                                onScreenChange(screen)
                                onSetSelectedListId(null)
                                onSetSelectedNoteId(null)
                                scope.launch { drawerState.close() }
                            }
                        )
                    }

                    else -> {
                        NavigationDrawerItem(
                            label = { Text(stringResource(id = screen.titleRes)) },
                            selected = screen == currentScreen,
                            onClick = {
                                shoppingListsViewModel.disableReorderMode()
                                notesViewModel.disableReorderMode()
                                onScreenChange(screen)
                                onSetSelectedListId(null)
                                onSetSelectedNoteId(null)
                                scope.launch { drawerState.close() }
                            }
                        )
                    }
                }
            }

            item {
                val standardListName = stringResource(R.string.standard_list_name)
                NavigationDrawerItem(
                    label = { Text(stringResource(id = R.string.load_standard_list)) },
                    selected = false,
                    onClick = {
                        shoppingListsViewModel.createStandardList(standardListName)
                        scope.launch { drawerState.close() }
                    }
                )
            }

            item {
                HorizontalDivider()
            }

            item {
                NavigationDrawerItem(
                    label = { Text(stringResource(R.string.json_import_export_title)) },
                    selected = false,
                    onClick = {
                        onShowJsonDialog()
                        scope.launch { drawerState.close() }
                    }
                )
            }

            item {
                NavigationDrawerItem(
                    label = {
                        Column {
                            Text(stringResource(id = R.string.database_location))
                            Text(
                                text = databasePath ?: stringResource(id = R.string.no_folder_selected),
                                style = MaterialTheme.typography.bodySmall,
                                color = MaterialTheme.colorScheme.onSurfaceVariant,
                                maxLines = 1,
                                overflow = androidx.compose.ui.text.style.TextOverflow.Ellipsis
                            )
                        }
                    },
                    selected = false,
                    onClick = {
                        onChooseDatabaseLocation()
                        scope.launch { drawerState.close() }
                    }
                )
            }

            item {
                HorizontalDivider()
            }

            item {
                NavigationDrawerItem(
                    label = { Text(stringResource(R.string.menu_settings)) },
                    selected = currentScreen == Screen.Settings,
                    onClick = {
                        onScreenChange(Screen.Settings)
                        scope.launch { drawerState.close() }
                    },
                    icon = {
                        Icon(
                            Icons.Default.Settings,
                            contentDescription = stringResource(R.string.menu_settings)
                        )
                    }
                )
            }

            item {
                NavigationDrawerItem(
                    label = { Text(stringResource(R.string.menu_about)) },
                    selected = currentScreen == Screen.About,
                    onClick = {
                        onScreenChange(Screen.About)
                        scope.launch { drawerState.close() }
                    },
                    icon = {
                        Icon(
                            Icons.Default.Info,
                            contentDescription = stringResource(R.string.menu_about)
                        )
                    }
                )
            }
        }
    }
}
