package de.lxtools.noteshop.ui.appshell


import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.padding
import androidx.compose.material3.MaterialTheme
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import de.lxtools.noteshop.BiometricAuthenticator
import de.lxtools.noteshop.Screen
import de.lxtools.noteshop.getDynamicStrings
import de.lxtools.noteshop.ui.AboutScreen
import de.lxtools.noteshop.ui.GuidedTourScreen
import de.lxtools.noteshop.ui.SettingsScreen
import de.lxtools.noteshop.ui.notes.NoteDetailScreen
import de.lxtools.noteshop.ui.notes.NotesScreen
import de.lxtools.noteshop.ui.notes.NotesViewModel
import de.lxtools.noteshop.ui.recipes.RecipeDetailScreen
import de.lxtools.noteshop.ui.recipes.RecipesScreen
import de.lxtools.noteshop.ui.recipes.RecipesViewModel
import de.lxtools.noteshop.ui.shopping.ShoppingListDetailScreen
import de.lxtools.noteshop.ui.shopping.ShoppingListsScreen
import de.lxtools.noteshop.ui.shopping.ShoppingListsViewModel
import de.lxtools.noteshop.ui.theme.ColorTheme
import de.lxtools.noteshop.ui.webapp.WebAppIntegrationScreen
import de.lxtools.noteshop.ui.webapp.WebAppIntegrationViewModel

@Composable
fun AppContent(
    innerPadding: PaddingValues,
    currentScreen: Screen,
    onScreenChange: (Screen) -> Unit,
    notesViewModel: NotesViewModel,
    shoppingListsViewModel: ShoppingListsViewModel,
    recipesViewModel: RecipesViewModel,
    selectedListId: Int?,
    onSetSelectedListId: (Int?) -> Unit,
    onShowListDialog: () -> Unit,
    selectedNoteId: Int?,
    onSetSelectedNoteId: (Int?) -> Unit,
    onShowNoteDialog: () -> Unit,
    selectedRecipeId: Int?,
    onSetSelectedRecipeId: (Int?) -> Unit,
    onShowRecipeDialog: () -> Unit,
    unlockedItemIds: Set<Int>,
    onUnlockItem: (Int, Screen, Int) -> Unit,
    shoppingListsTitle: String,
    onThemeChange: (ColorTheme) -> Unit,
    colorTheme: ColorTheme,
    isAppLockEnabled: Boolean,
    onAppLockChange: (Boolean) -> Unit,
    biometricAuthenticator: BiometricAuthenticator,
    sharedPrefs: android.content.SharedPreferences,
    onResetShoppingListsTitle: () -> Unit,
    onResetRecipesTitle: () -> Unit,
    startScreen: String,
    onStartScreenChange: (String) -> Unit,
    canUseBiometrics: Boolean,
    webAppIntegrationViewModel: WebAppIntegrationViewModel,
    onAuthenticateAndTogglePasswordVisibility: (Boolean) -> Unit
) {
    val dynamicStrings = getDynamicStrings(shoppingListsTitle)

    Column(
        modifier = Modifier
            .fillMaxSize()
            .padding(innerPadding)
            .background(MaterialTheme.colorScheme.background),
        horizontalAlignment = Alignment.CenterHorizontally
    ) {
        when (currentScreen) {
            is Screen.ShoppingLists -> {
                ShoppingListsScreen(
                    shoppingListsViewModel = shoppingListsViewModel,
                    onListClick = { listId, protectionType ->
                        if (protectionType != 0) {
                            onUnlockItem(listId, Screen.ShoppingListDetail, protectionType)
                        } else {
                            onSetSelectedListId(listId)
                            onScreenChange(Screen.ShoppingListDetail)
                        }
                    },
                    onEditList = { listWithItems ->
                        shoppingListsViewModel.updateListDetails(listWithItems.shoppingList)
                        onShowListDialog()
                    }
                )
            }

            is Screen.Notes -> {
                NotesScreen(
                    notesViewModel = notesViewModel,
                    onNoteClick = { noteId, protectionType ->
                        if (protectionType != 0) {
                            onUnlockItem(noteId, Screen.NoteDetail, protectionType)
                        } else {
                            onSetSelectedNoteId(noteId)
                            onScreenChange(Screen.NoteDetail)
                        }
                    },
                    onEditNote = { note ->
                        notesViewModel.updateNoteDetails(note)
                        onShowNoteDialog()
                    }
                )
            }

            is Screen.ShoppingListDetail -> {
                ShoppingListDetailScreen(
                    listId = selectedListId,
                    shoppingListsViewModel = shoppingListsViewModel,
                    dynamicStrings = dynamicStrings,
                    isContentDecrypted = unlockedItemIds.contains(selectedListId),
                    onUnlockClick = {
                        val list = shoppingListsViewModel.uiState.value.shoppingLists.find { it.shoppingList.id == selectedListId }?.shoppingList
                        onUnlockItem(selectedListId!!, Screen.ShoppingListDetail, list?.protectionType ?: 0)
                    },
                    onNavigateBack = {
                        onScreenChange(Screen.ShoppingLists)
                        onSetSelectedListId(null)
                    }
                )
            }
            is Screen.NoteDetail -> {
                NoteDetailScreen(
                    noteId = selectedNoteId,
                    notesViewModel = notesViewModel,
                    isContentDecrypted = unlockedItemIds.contains(selectedNoteId),
                    onUnlockClick = {
                        val note = notesViewModel.uiState.value.noteList.find { it.id == selectedNoteId }
                        onUnlockItem(selectedNoteId!!, Screen.NoteDetail, note?.protectionType ?: 0)
                    }
                )
            }

            is Screen.Recipes -> {
                RecipesScreen(
                    recipesViewModel = recipesViewModel,
                    onRecipeClick = { recipeId, protectionType ->
                        if (protectionType != 0) {
                            onUnlockItem(recipeId, Screen.RecipeDetail, protectionType)
                        } else {
                            onSetSelectedRecipeId(recipeId)
                            onScreenChange(Screen.RecipeDetail)
                        }
                    },
                    onEditRecipe = { recipe ->
                        recipesViewModel.updateRecipeDetails(recipe)
                        onShowRecipeDialog()
                    }
                )
            }

            is Screen.RecipeDetail -> {
                RecipeDetailScreen(
                    recipeId = selectedRecipeId,
                    recipesViewModel = recipesViewModel,
                    isContentDecrypted = unlockedItemIds.contains(selectedRecipeId),
                    onUnlockClick = {
                        val recipe = recipesViewModel.uiState.value.recipeList.find { it.id == selectedRecipeId }
                        onUnlockItem(selectedRecipeId!!, Screen.RecipeDetail, recipe?.protectionType ?: 0)
                    }
                )
            }

            is Screen.About -> {
                AboutScreen()
            }

            is Screen.Settings -> {
                SettingsScreen(
                    onThemeChange = onThemeChange,
                    currentTheme = colorTheme,
                    isAppLockEnabled = isAppLockEnabled,
                    onAppLockChange = onAppLockChange,
                    biometricAuthenticator = biometricAuthenticator,
                    sharedPrefs = sharedPrefs,
                    onResetShoppingListsTitle = onResetShoppingListsTitle,
                    onResetRecipesTitle = onResetRecipesTitle,
                    currentStartScreen = startScreen,
                    onStartScreenChange = onStartScreenChange,
                    canUseBiometrics = canUseBiometrics,
                    onNavigate = { screen -> onScreenChange(screen) }
                )
            }
            is Screen.WebAppIntegration -> {
                WebAppIntegrationScreen(
                    webAppIntegrationViewModel = webAppIntegrationViewModel,
                    padding = innerPadding,
                    onAuthenticateAndTogglePasswordVisibility = onAuthenticateAndTogglePasswordVisibility
                )
            }
            is Screen.GuidedTour -> {
                GuidedTourScreen(
                    onTourFinished = { onScreenChange(Screen.Settings) }
                )
            }
        }
    }
}