package de.lxtools.noteshop.ui

import android.content.SharedPreferences
import androidx.compose.foundation.background
import androidx.compose.foundation.border
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.ExperimentalLayoutApi
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.shape.CircleShape
import androidx.compose.animation.AnimatedVisibility
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Check
import androidx.compose.material.icons.filled.KeyboardArrowDown
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.Icon
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Switch
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import de.lxtools.noteshop.R
import androidx.compose.ui.platform.LocalContext
import de.lxtools.noteshop.findActivity
import de.lxtools.noteshop.Screen
import androidx.compose.ui.draw.alpha
import de.lxtools.noteshop.ui.theme.*
import androidx.compose.foundation.lazy.LazyRow
import androidx.compose.foundation.lazy.items
import androidx.compose.material3.Button
import androidx.compose.material3.DropdownMenu
import androidx.compose.material3.DropdownMenuItem
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.core.content.edit
import de.lxtools.noteshop.BiometricAuthenticator
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.rememberLazyListState
import androidx.fragment.app.FragmentActivity

@OptIn(ExperimentalLayoutApi::class, ExperimentalMaterial3Api::class)
@Composable
fun SettingsScreen(
    currentTheme: ColorTheme,
    onThemeChange: (ColorTheme) -> Unit,
    isAppLockEnabled: Boolean,
    onAppLockChange: (Boolean) -> Unit,
    biometricAuthenticator: BiometricAuthenticator,
    sharedPrefs: SharedPreferences,
    onResetShoppingListsTitle: () -> Unit,
    onResetRecipesTitle: () -> Unit,
    currentStartScreen: String,
    onStartScreenChange: (String) -> Unit,
    canUseBiometrics: Boolean, // New parameter
    onNavigate: (Screen) -> Unit
) {
    var showStartScreenMenu by remember { mutableStateOf(false) }
    val startScreenOptions = mapOf(
        "last_screen" to stringResource(R.string.start_screen_last),
        Screen.ShoppingLists.route to stringResource(R.string.start_screen_shopping_lists),
        Screen.Notes.route to stringResource(R.string.start_screen_notes),
        Screen.Recipes.route to stringResource(R.string.start_screen_recipes)
    )
    val currentStartScreenName = startScreenOptions[currentStartScreen] ?: stringResource(R.string.start_screen_last)
    val context = LocalContext.current

    val listState = rememberLazyListState()

    Box(modifier = Modifier.fillMaxSize()) {
        LazyColumn(
            state = listState,
            modifier = Modifier
                .fillMaxSize()
                .padding(16.dp)
        ) {
            item {
                Text(
                    text = stringResource(R.string.color_scheme),
                    style = MaterialTheme.typography.titleLarge
                )
            }
            item { Spacer(modifier = Modifier.height(16.dp)) }

            item {
                LazyRow(
                    modifier = Modifier.fillMaxWidth(),
                    horizontalArrangement = Arrangement.spacedBy(8.dp) // Reduced spacing
                ) {
                    items(ColorTheme.entries.toTypedArray()) { theme ->
                        ThemeChooserItem(
                            theme = theme,
                            isSelected = theme == currentTheme,
                            onClick = { onThemeChange(theme) }
                        )
                    }
                }
            }

            item { Spacer(modifier = Modifier.height(24.dp)) }
            item { HorizontalDivider() }
            item { Spacer(modifier = Modifier.height(24.dp)) }

            item {
                Text(
                    text = stringResource(R.string.settings_general),
                    style = MaterialTheme.typography.titleLarge
                )
            }
            item { Spacer(modifier = Modifier.height(16.dp)) }

            item {
                Box(modifier = Modifier.fillMaxWidth()) {
                    Button(
                        onClick = { showStartScreenMenu = true },
                        modifier = Modifier.fillMaxWidth()
                    ) {
                        Text(text = stringResource(R.string.settings_start_screen) + ": " + currentStartScreenName)
                    }
                    DropdownMenu(
                        expanded = showStartScreenMenu,
                        onDismissRequest = { showStartScreenMenu = false }
                    ) {
                        startScreenOptions.forEach { (key, name) ->
                            DropdownMenuItem(
                                text = { Text(name) },
                                onClick = {
                                    onStartScreenChange(key)
                                    showStartScreenMenu = false
                                }
                            )
                        }
                    }
                }
            }

            item { Spacer(modifier = Modifier.height(8.dp)) }

            item {
                Button(onClick = onResetShoppingListsTitle, modifier = Modifier.fillMaxWidth()) {
                    Text(text = stringResource(R.string.reset_shopping_lists_name))
                }
            }
            item { Spacer(modifier = Modifier.height(8.dp)) }
            item {
                Button(onClick = onResetRecipesTitle, modifier = Modifier.fillMaxWidth()) {
                    Text(text = stringResource(R.string.reset_recipes_name))
                }
            }

            item { Spacer(modifier = Modifier.height(8.dp)) }

            item {
                Button(onClick = { onNavigate(Screen.WebAppIntegration) }, modifier = Modifier.fillMaxWidth()) {
                    Text(text = stringResource(R.string.webapp_integration_title))
                }
            }

            item { Spacer(modifier = Modifier.height(8.dp)) }

            item {
                Button(onClick = { onNavigate(Screen.GuidedTour) }, modifier = Modifier.fillMaxWidth()) {
                    Text(text = stringResource(R.string.show_guided_tour))
                }
            }

            item { Spacer(modifier = Modifier.height(24.dp)) }
            item { HorizontalDivider() }
            item { Spacer(modifier = Modifier.height(24.dp)) }

            item {
                Text(
                    text = stringResource(R.string.security),
                    style = MaterialTheme.typography.titleLarge
                )
            }
            item { Spacer(modifier = Modifier.height(16.dp)) }

            item {
                Row(
                    modifier = Modifier
                        .fillMaxWidth()
                        .clickable(enabled = canUseBiometrics) { if (canUseBiometrics) onAppLockChange(!isAppLockEnabled) }
                        .padding(vertical = 8.dp),
                    verticalAlignment = Alignment.CenterVertically,
                    horizontalArrangement = Arrangement.SpaceBetween
                ) {
                    Text(
                        text = stringResource(R.string.app_lock),
                        style = MaterialTheme.typography.bodyLarge,
                        modifier = Modifier
                            .weight(1f)
                            .alpha(if (canUseBiometrics) 1f else 0.5f)
                    )
                    Switch(
                        checked = isAppLockEnabled,
                        onCheckedChange = onAppLockChange,
                        enabled = canUseBiometrics
                    )
                }
            }
        }

        AnimatedVisibility(
            visible = listState.canScrollForward,
            modifier = Modifier
                .align(Alignment.BottomCenter)
                .padding(bottom = 8.dp)
        ) {
            Icon(
                imageVector = Icons.Default.KeyboardArrowDown,
                contentDescription = null, // Decorative
                tint = MaterialTheme.colorScheme.onSurface,
                modifier = Modifier.size(24.dp)
            )
        }
    }
}


@Composable
private fun ThemeChooserItem(
    theme: ColorTheme,
    isSelected: Boolean,
    onClick: () -> Unit
) {
    val themeColor = when (theme) {
        ColorTheme.STANDARD -> BluePrimary
        ColorTheme.GREEN -> GreenPrimary
        ColorTheme.VIOLET -> VioletPrimary
        ColorTheme.YELLOW -> YellowPrimary
        ColorTheme.ORANGE -> OrangePrimary
        ColorTheme.RED -> RedPrimary
    }

    Column(
        horizontalAlignment = Alignment.CenterHorizontally,
        modifier = Modifier
            .clickable(onClick = onClick)
            .padding(4.dp)
    ) {
        Box(
            modifier = Modifier
                .size(48.dp) // Reduced size
                .clip(CircleShape)
                .background(themeColor)
                .then(
                    if (isSelected) {
                        Modifier.border(2.dp, MaterialTheme.colorScheme.onBackground, CircleShape)
                    } else {
                        Modifier
                    }
                ),
            contentAlignment = Alignment.Center
        ) {
            if (isSelected) {
                val checkmarkColor = if (theme == ColorTheme.YELLOW) Color.Black else Color.White
                Icon(
                    imageVector = Icons.Default.Check,
                    contentDescription = stringResource(R.string.selected),
                    tint = checkmarkColor,
                    modifier = Modifier.size(24.dp) // Reduced size
                )
            }
        }
        Spacer(modifier = Modifier.height(8.dp))
        Text(text = stringResource(id = theme.titleRes), style = MaterialTheme.typography.bodyMedium)
    }
}