package de.lxtools.noteshop.ui

import android.util.Log
import androidx.activity.compose.rememberLauncherForActivityResult
import androidx.activity.result.contract.ActivityResultContracts
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.height
import androidx.compose.material3.AlertDialog
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.runtime.Composable
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import de.lxtools.noteshop.R
import android.widget.Toast
import kotlinx.serialization.SerializationException
import de.lxtools.noteshop.ui.notes.NotesViewModel
import de.lxtools.noteshop.ui.shopping.ShoppingListsViewModel
import de.lxtools.noteshop.ui.recipes.RecipesViewModel
import kotlinx.coroutines.launch

@Composable
fun JsonImportExportDialog(
    onDismissRequest: () -> Unit,
    notesViewModel: NotesViewModel,
    shoppingListsViewModel: ShoppingListsViewModel,
    recipesViewModel: RecipesViewModel,
    shoppingListsTitle: String,
    recipesTitle: String,
) {
    val scope = rememberCoroutineScope()
    val context = LocalContext.current

    val notesImportJsonLauncher = rememberLauncherForActivityResult(
        contract = ActivityResultContracts.OpenDocument(),
        onResult = { uri ->
            uri?.let {
                scope.launch {
                    context.contentResolver.openInputStream(it)?.use { inputStream ->
                        val jsonString = inputStream.readBytes().toString(Charsets.UTF_8)
                        try {
                            notesViewModel.importNotesFromJson(jsonString)
                            Toast.makeText(context, R.string.json_import_successful, Toast.LENGTH_SHORT).show()
                        } catch (e: SerializationException) {
                            Log.e("JsonImportExportDialog", "Error parsing notes JSON: ${e.message}")
                            Toast.makeText(context, context.getString(R.string.json_import_failed_invalid_format, e.message), Toast.LENGTH_LONG).show()
                        } catch (e: Exception) {
                            Log.e("JsonImportExportDialog", "Error importing notes: ${e.message}")
                            Toast.makeText(context, context.getString(R.string.json_import_failed_generic, e.message), Toast.LENGTH_LONG).show()
                        }
                    }
                    onDismissRequest()
                }
            }
        }
    )

    val notesExportJsonLauncher = rememberLauncherForActivityResult(
        contract = ActivityResultContracts.CreateDocument("application/json"),
        onResult = { uri ->
            uri?.let {
                scope.launch {
                    val allNotes = notesViewModel.uiState.value.noteList
                    val content = notesViewModel.exportNotesToJson(allNotes)
                    context.contentResolver.openOutputStream(it, "wt")?.use { outputStream ->
                        outputStream.write(content.toByteArray(Charsets.UTF_8))
                        Toast.makeText(context, R.string.json_export_successful, Toast.LENGTH_SHORT).show()
                    }
                }
            }
            onDismissRequest()
        }
    )

    val shoppingListsExportJsonLauncher = rememberLauncherForActivityResult(
        contract = ActivityResultContracts.CreateDocument("application/json"),
        onResult = { uri ->
            uri?.let {
                scope.launch {
                    val allShoppingLists = shoppingListsViewModel.uiState.value.shoppingLists
                    val content = shoppingListsViewModel.exportShoppingListsToJson(allShoppingLists)
                    context.contentResolver.openOutputStream(it, "wt")?.use { outputStream ->
                        outputStream.write(content.toByteArray(Charsets.UTF_8))
                        Toast.makeText(context, R.string.json_export_successful, Toast.LENGTH_SHORT).show()
                    }
                }
            }
            onDismissRequest()
        }
    )

    val shoppingListsImportJsonLauncher = rememberLauncherForActivityResult(
        contract = ActivityResultContracts.OpenDocument(),
        onResult = { uri ->
            uri?.let {
                scope.launch {
                    context.contentResolver.openInputStream(it)?.use { inputStream ->
                        val jsonString = inputStream.readBytes().toString(Charsets.UTF_8)
                        try {
                            shoppingListsViewModel.importShoppingListsFromJson(jsonString)
                            Toast.makeText(context, R.string.json_import_successful, Toast.LENGTH_SHORT).show()
                        } catch (e: SerializationException) {
                            Log.e("JsonImportExportDialog", "Error parsing shopping lists JSON: ${e.message}")
                            Toast.makeText(context, context.getString(R.string.json_import_failed_invalid_format, e.message), Toast.LENGTH_LONG).show()
                        } catch (e: Exception) {
                            Log.e("JsonImportExportDialog", "Error importing shopping lists: ${e.message}")
                            Toast.makeText(context, context.getString(R.string.json_import_failed_generic, e.message), Toast.LENGTH_LONG).show()
                        }
                    }
                    onDismissRequest()
                }
            }
        }
    )

    val recipesExportJsonLauncher = rememberLauncherForActivityResult(
        contract = ActivityResultContracts.CreateDocument("application/json"),
        onResult = { uri ->
            uri?.let {
                scope.launch {
                    val allRecipes = recipesViewModel.uiState.value.recipeList
                    val content = recipesViewModel.exportRecipesToJson(allRecipes)
                    context.contentResolver.openOutputStream(it, "wt")?.use { outputStream ->
                        outputStream.write(content.toByteArray(Charsets.UTF_8))
                        Toast.makeText(context, R.string.json_export_successful, Toast.LENGTH_SHORT).show()
                    }
                }
            }
            onDismissRequest()
        }
    )

    val recipesImportJsonLauncher = rememberLauncherForActivityResult(
        contract = ActivityResultContracts.OpenDocument(),
        onResult = { uri ->
            uri?.let {
                scope.launch {
                    context.contentResolver.openInputStream(it)?.use { inputStream ->
                        val jsonString = inputStream.readBytes().toString(Charsets.UTF_8)
                        try {
                            recipesViewModel.importRecipesFromJson(jsonString)
                            Toast.makeText(context, R.string.json_import_successful, Toast.LENGTH_SHORT).show()
                        } catch (e: SerializationException) {
                            Log.e("JsonImportExportDialog", "Error parsing recipes JSON: ${e.message}")
                            Toast.makeText(context, context.getString(R.string.json_import_failed_invalid_format, e.message), Toast.LENGTH_LONG).show()
                        } catch (e: Exception) {
                            Log.e("JsonImportExportDialog", "Error importing recipes: ${e.message}")
                            Toast.makeText(context, context.getString(R.string.json_import_failed_generic, e.message), Toast.LENGTH_LONG).show()
                        }
                    }
                    onDismissRequest()
                }
            }
        }
    )

    AlertDialog(
        onDismissRequest = onDismissRequest,
        title = { Text(text = stringResource(R.string.json_import_export_title)) },
        text = {
            Column {
                Text(text = stringResource(R.string.menu_notes))
                TextButton(
                    onClick = {
                        val fileName = "notes.json"
                        notesExportJsonLauncher.launch(fileName)
                    }
                ) {
                    Text(stringResource(R.string.export_notes))
                }
                TextButton(
                    onClick = {
                        notesImportJsonLauncher.launch(arrayOf("application/json"))
                    }
                ) {
                    Text(stringResource(R.string.import_notes))
                }

                Spacer(modifier = Modifier.height(16.dp))
                HorizontalDivider()
                Spacer(modifier = Modifier.height(16.dp))

                Text(text = shoppingListsTitle)
                TextButton(
                    onClick = {
                        val fileName = "shopping_lists.json"
                        shoppingListsExportJsonLauncher.launch(fileName)
                    }
                ) {
                    Text(stringResource(R.string.export_shopping_lists))
                }
                TextButton(
                    onClick = {
                        shoppingListsImportJsonLauncher.launch(arrayOf("application/json"))
                    }
                ) {
                    Text(stringResource(R.string.import_shopping_lists))
                }

                Spacer(modifier = Modifier.height(16.dp))
                HorizontalDivider()
                Spacer(modifier = Modifier.height(16.dp))

                Text(text = recipesTitle)
                TextButton(
                    onClick = {
                        val fileName = "recipes.json"
                        recipesExportJsonLauncher.launch(fileName)
                    }
                ) {
                    Text(stringResource(R.string.export_recipes))
                }
                TextButton(
                    onClick = {
                        recipesImportJsonLauncher.launch(arrayOf("application/json"))
                    }
                ) {
                    Text(stringResource(R.string.import_recipes))
                }
            }
        },
        confirmButton = {
            TextButton(onClick = onDismissRequest) {
                Text(stringResource(R.string.cancel))
            }
        }
    )
}
